/****************************************************************************
*
*  dcomp.cmd
*
*  Compare directories and synchronize content
*
*  Release: */opt.version = "0.3"/*
*  Create:  01.09.1999
*  Update:  31.07.2003
*  (C)      Marcel Mller 1999-2003
*
*  Known problems:
*  - /Q does not work with Classic REXX. Use -Q or "/Q" instead.
*    It ist a problem with the Classic REXX implementation.
*
****************************************************************************/

/* load REXXUTIL funcs */
CALL RxFuncAdd "SysLoadFuncs", "RexxUtil", "SysLoadFuncs"
CALL SysLoadFuncs
NUMERIC DIGITS 12
CALL CRC32_Init

/* init vars */
opt.sdir = ''
opt.debug = 0
opt.mout = 1
opt.fcomp = 0
opt.sync = 0
opt.syncsimilar = 0
opt.nodel = 1
opt.move = 0
opt.longname = 0
opt.folder = 0
opt.thres = 0
opt.tz = 0
opt.updfile = ''
opt.volfile = ''
opt.numvol = -1 /* invalid */
opt.volno = -1 /* invalid */
opt.sayeq = 0
opt.ignoredowngrade = 0
opt.help = 0
/* autoconfig */
IF FindFilePath('XCOPY.EXE') = '' THEN
   opt.copycmd = "@COPY"
 ELSE
   opt.copycmd = "@XCOPY /H /T /R /O"
IF FindFilePath('XCOMP.EXE') = '' THEN
   opt.compcmd = "@ECHO n|COMP"
 ELSE
   opt.compcmd = "@XCOMP"
opt.delcmd = "@DEL"
opt.movecmd = "@MOVE"
opt.saveea = "@EAUTIL /S /R /P"
opt.loadea = "@EAUTIL /O /J"

dlerr = 0
source.2 = ''
opt.xcl = 0

/****************************************************************************
   parse command line
****************************************************************************/
kstat = 0
params = ARG(1)
DO WHILE params \= ''
   IF LEFT(params,1) = '"' THEN
      PARSE VAR params '"'param'" 'params
    ELSE
      PARSE VAR params param params
   CALL ParseArg(param)
   END

IF kstat < 1 | opt.help THEN DO
   SAY "Compare and/or synchronize files and directories version "opt.version
   SAY "(C) Marcel Mller 1999"
   SAY
   SAY "Usage: DSYNC filespec1 filespec2 [options]"
   SAY
   SAY "compare results:"
   SAY " *O  file in tree 1 newer"
   SAY " O*  file in tree 2 newer"
   SAY " *-  file in tree 2 not found"
   SAY " -*  file in tree 1 not found"
   SAY " >>  file in tree 1 is longer with same time stamp"
   SAY " <<  file in tree 2 is longer with same time stamp"
   SAY " <>  files have different content with same time stamp"
   SAY " EE  error during compare"
   SAY
   SAY "options:"
   SAY " /S      include subdirectories"
   SAY " /F      compare file content"
   SAY " /E      synchronize folders too"
   SAY " /Xpat   exclude files matching wildcard pattern"
   SAY " /V      compare result in text format instead of the default as noted above"
   SAY " /Q      quiet, i.e. no output to stdout"
   SAY " /Y      synchronize files"
   SAY " /Y1     update only 1st tree"
   SAY " /Y2     update only 2nd tree"
   SAY " /FY     force syncronisation of objects that only differ in size"
   SAY " /Ufile  create update file/directory in filespec1 using file as index"
   SAY " /Pnum:no switch filespec1 in multiple volume mode"
   SAY "         The number of volumes (num) and the volume number (no) are optional."
   SAY " /ZI     shortcut for /S /Q /U, create initial update file"
   SAY " /ZP     shortcut for /S /Q /U /Y1, pack update packet"
   SAY " /ZU     shortcut for /S /Q /U /Y2, unpack update packet"
   SAY " /ZA     shortcut for /S /Q /U /Y /M, fully automatic resync of update packet"
   SAY " /D      delete files when removed in the other tree"
   SAY " /M      move files from update directory instead of copying"
   SAY " /Tthres threshold in seconds for time compares"
   SAY " /T      use timezone corretion"
   SAY " /I      ignore downgrades (see reference)"
   SAY " /L      use PMSHELL copy routine and .LONGNAME EA"
   EXIT 48
   END

IF kstat > 2 THEN
   CALL Error 49, "Syntax error: more the 2 source paths."

/* get root path & filespec */
DO i = 1 TO 2
   /* translate '/' */
   source.i = TRANSLATE(source.i, '\', '/')
   /* get pattern */
   name.i = FILESPEC("N", source.i)
   /* check if directory */
   IF (name.i \= '') & (VERIFY(name.i, "*?", 'M') = 0) THEN DO
      CALL SysFileTree source.i, tree.i, "DO"
      IF tree.i \= 0 THEN DO
         IF opt.debug THEN
            SAY source.i" seems to be a directory, '\' added."
         source.i = source.i'\'
         name.i = ''
         END
      END
   /* get root directory */
   root.i = DirSpec(source.i)
   CALL SETLOCAL
   IF root.i = '' THEN
      root.i = DIRECTORY()
    ELSE IF LEFT(root.i, 2) \= '\\' THEN /* not for UNC path */
      root.i = DIRECTORY(root.i)
   IF root.i = '' THEN
      CALL Error 8, "Source tree "i" ("""DirSpec(source.i)""") does not exist."
   IF RIGHT(root.i, 1) \= '\' THEN
      root.i = root.i'\'
   CALL ENDLOCAL
   offset.i = LENGTH(root.i) +23
   END

/* some defaults */
IF name.1 = '' THEN
   name.1 = '*'
IF name.2 = '' THEN
   name.2 = name.1
IF LENGTH(opt.updfile) = 1 THEN
   opt.updfile = root.1||"UPDATE.DIR"
 ELSE
   opt.updfile = SUBSTR(opt.updfile, 2)
IF opt.volfile \= '' THEN
   opt.volfile = root.1||opt.volfile

IF opt.debug THEN DO
   DO i = 1 TO 2
      SAY "Filespec "i": root = """root.i""", pattern = """name.i""""
      END
   SAY "Timezone: "opt.tz
   END

/* some consistency checks */
IF opt.updfile \= '' & opt.fcomp THEN
   CALL Error 34, "/U and /F are mutual exclusive."
IF opt.updfile \= '' & opt.volfile \= '' THEN
   CALL Error 34, "/U and /P are mutual exclusive."
IF opt.longname & opt.fcomp THEN
   CALL Error 34, "/L and /F are currently not supported together."
IF opt.sync = 3 & \opt.nodel THEN
   CALL Error 34, "/Y and /D are mutual exclusive."
IF opt.syncsimilar & opt.sync = 3 THEN
   CALL Error 34, "/FY and /Y are mutual exclusive."
IF root.1 = root.2 THEN
   CALL Error 32, "Cannot compare files to themselves."
IF name.1 \= name.2 THEN
   CALL Error 32, "Cannot compare directories with different file patterns."
tmp = MIN(LENGTH(root.1), LENGTH(root.2))
IF LEFT(root.1, tmp) = LEFT(root.2, tmp) THEN
   CALL Error 32, "Source path "root.1" is not independent of source path "root.2"."

/****************************************************************************
   read volume info
****************************************************************************/
IF opt.volfile \= '' THEN DO
   IF opt.volno = -1 | opt.numvol = -1 THEN DO
      IF STREAM(opt.volfile, 'C', 'OPEN READ') = 'READY:' THEN DO
         PARSE VALUE LINEIN(opt.volfile) WITH v'/'n
         IF \DATATYPE(v, 'N') | \DATATYPE(n, 'N') THEN
            CALL Error 24, "The volume info "opt.volfile" is invalid. Use -pnum:no to override."
         CALL STREAM opt.volfile, 'C', 'CLOSE'
         IF opt.volno = -1 THEN
            opt.volno = v
         IF opt.numvol = -1 THEN
            opt.numvol = n
         END
       ELSE
         CALL Error 22, "Cannot deduce the volume information from "opt.volfile". Use -pnum:no."
      END
   opt.volno = opt.volno -1 /* convert to zero based index */
   /* calculate distribution function */
   CALL mmapinit
   DO i = 2 TO opt.numvol
      CALL mmincrease
      /*CALL mmanalysis*/
      END
   END

/****************************************************************************
   get file lists
****************************************************************************/
DO i = 1 TO 2
   IF i = 1 & opt.updfile \= '' THEN DO
      tree.1 = 0
      IF opt.sync = 0 THEN ITERATE
      /* fetch data from update file */
      IF STREAM(opt.updfile, 'C', 'open read') \= 'READY:' THEN DO
         IF opt.sync = 3 THEN DO
            CALL LINEOUT STDERR, "Could not find index file """opt.updfile"""."
            CALL LINEOUT STDERR, "In case of the first call with /ZA you may want to create an initial one."
            CALL LINEOUT STDERR, "Create initial index file [y|n] ?"
            tmp = TRANSLATE(LINEIN())
            IF LEFT(tmp,1) = 'Y' THEN DO
               opt.sync = 0
               ITERATE
               END
            END
         CALL Error 20, "Failed to read update index file "opt.updfile"."
         END
      voldesc.1 = ''
      DO j = 1
         DO FOREVER
            tmp = TRANSLATE(LINEIN(opt.updfile), '\', '/')
            IF LEFT(tmp, 1) \= ':' THEN LEAVE
            /* volume descriptor */
            voldesc.1 = SUBSTR(tmp, 2)
            END
         PARSE VAR tmp time size name
         IF name = '' THEN
            LEAVE
         tree.1.j = RIGHT(time+0, 10, 0)" "RIGHT(size+0, 10, 0)" "STRIP(name)
         END
      tree.1 = j -1
      CALL STREAM opt.updfile, 'C', 'close'
      offset.1 = 23
      END
    ELSE DO
      CALL ScanTree i, root.i||name.i
      /* discard files on other volumes */
      IF opt.volfile \= '' & i = 2 THEN
         DO j = tree.i.0 TO 1 BY -1
            part = mmlook(CRC32(TRANSLATE(TRANSLATE(SUBSTR(tree.i.j, offset.i)),'/','\')))
            IF part \= opt.volno THEN DO
               IF opt.debug THEN
                  SAY "V: "part SUBSTR(tree.i.j, offset.i)
               tree.i.0 = tree.i.0 -1 /* delete */
               DO k = j TO tree.i.0
                  tree.i.j = VALUE('tree.'i'.'j+1)
                  END
               END
            END
      END
   /* sort file list */
   CALL fqsort i, 1, tree.i
   /*DO j = 1 TO tree.i
      SAY tree.i.j
      END*/
   END
/* volume descriptors */
IF opt.updfile \= '' THEN DO
   tmp = FILESPEC('D', root.2)
   IF tmp \= '' THEN
      voldesc.2 = '<'WORD(SysDriveInfo(tmp), 4)'> 'root.2
    ELSE
      voldesc.2 = root.2
   IF voldesc.1 \= '' & opt.sync // 3 \= 0 & COMPARE(TRANSLATE(voldesc.1), TRANSLATE(voldesc.2), '00'x) = 0 THEN DO
      CALL LINEOUT STDERR, "Update tree is possibly the same then at the last call."
      CALL LINEOUT STDERR, "You propably want to use /za or /y."
      CALL LINEOUT STDERR, "Continue anyway [y|n] ?"
      IF LEFT(TRANSLATE(LINEIN()), 1) \= 'Y' THEN
         EXIT 18
      END
   END

/****************************************************************************
   compare ...
****************************************************************************/
root.3.0 = 0 /* delete list */
i1 = 0
i2 = 0
name1 = NextName(1)
name2 = NextName(2)
DO WHILE name1 \= '' | name2 \= ''
   /* filenames equal ? */
   name = TRANSLATE(STRIP(name1,'T','\'))
   IF '_'name = '_'TRANSLATE(STRIP(name2,'T','\')) THEN DO
      /* file vs directory */
      IF (RIGHT(name1, 1) = '\') + (RIGHT(name2, 1) = '\') = 1 THEN
         CALL DoItem 'FF', name, 'file/folder conflict'
       ELSE DO
         /* compare time stamp */
         diff = ABS(LEFT(tree.1.i1, 10) - LEFT(tree.2.i2, 10))
         IF diff > opt.thres THEN DO
            IF diff < 86400 & diff // 3600 = 0 THEN
               CALL DaylightErr
            IF opt.debug THEN
               SAY "Time diff: "name1" "||(LEFT(tree.1.i1, 10) - LEFT(tree.2.i2, 10))
            IF LEFT(tree.1.i1, 10) < LEFT(tree.2.i2, 10) THEN
               CALL DoItem 'O*', name2
             ELSE
               CALL DoItem '*O', name1
            END
         /* directory => always equal */
          ELSE IF RIGHT(name1, 1) = '\' THEN
            CALL DoItem '<>', name2 /* Well, hot hack to force EA synchronisation */
         /* compare size */
          ELSE IF SUBSTR(tree.1.i1, 12, 10) \= SUBSTR(tree.2.i2, 12, 10) THEN DO
            IF SUBSTR(tree.1.i1, 12, 10) > SUBSTR(tree.2.i2, 12, 10) THEN
               CALL DoItem '>>', name1
             ELSE
               CALL DoItem '<<', name2
            END
         /* compare content */
          ELSE IF opt.fcomp THEN DO
            opt.compcmd' "'PCTesc(SUBSTR(tree.1.i1, 23)'" "'SUBSTR(tree.2.i2, 23))'" >NUL'
            IF rc = 1 | rc = 5 THEN
               CALL DoItem '<>', name2
             ELSE IF rc > 1 THEN
               CALL DoItem 'EE', name2, rc
            END
          ELSE
            CALL DoItem '==', name2
         END
    next:
      name1 = NextName(1)
      name2 = NextName(2)
      END
   /* filenames not equal */
    ELSE IF "_"name < "_"TRANSLATE(name2) THEN DO
      CALL DoItem '*-', name1
      name1 = NextName(1)
      END
    ELSE DO
      CALL DoItem '-*', name2
      name2 = NextName(2)
      END
   END

/****************************************************************************
   remove directories scheduled for deletion
****************************************************************************/
DO i = root.3.0 TO 1 BY -1
   CALL Warning DeleteDir(root.3.i)
   END

/****************************************************************************
   create update file
****************************************************************************/
IF opt.updfile \= '' & opt.sync \= 2 & \opt.debug THEN DO
   IF opt.sync = 3 THEN
      /* rescan tree to include recent changes */
      CALL ScanTree 2, root.2||name.2
   IF opt.sync \= 0 THEN
      /* remove old file if any */
      opt.delcmd' "'PCTesc(opt.updfile)'" 2>NUL'
    ELSE IF opt.sync = 0 & STREAM(opt.updfile, 'c', 'query exists') \= '' THEN
      CALL Error 14, "Update file "opt.updfile" alredy exists. Maybe you forgot /Y."
   IF STREAM(opt.updfile, 'C', 'open write') \= 'READY:' THEN
      CALL Error 27, "Failed to create update index file "opt.updfile"."
   CALL LINEOUT opt.updfile, ':'voldesc.2
   DO i = 1 TO tree.2
      CALL LINEOUT opt.updfile, LEFT(tree.2.i, 10)+0" "SUBSTR(tree.2.i, 12, 10)+0" "SUBSTR(tree.2.i, offset.2)
      END
   CALL STREAM opt.updfile, 'C', 'close'
   END

IF opt.volfile \= '' & opt.sync//2 = 1 & \opt.debug THEN DO
   opt.delcmd' "'PCTesc(opt.volfile)'" 2>NUL'
   IF STREAM(opt.volfile, 'C', 'OPEN WRITE') \= 'READY:' THEN
      CALL Error 21, "Cannot create volume info "opt.volfile"."
   CALL LINEOUT opt.volfile, opt.volno+1'/'opt.numvol
   CALL STREAM opt.volfile, 'C', 'close'
   END

EXIT 0


/****************************************************************************
   main functions
****************************************************************************/

/* parse command line argument
   ParseArg(arg)
*/
ParseArg: PROCEDURE EXPOSE opt. source. kstat
   IF ARG(1) = '' THEN
      RETURN
   IF LEFT(ARG(1), 1) = '@' THEN DO
      /* option file */
      param = SUBSTR(ARG(1), 2)
      IF STREAM(param, 'c', 'open read') \= 'READY:' THEN
         CALL Error 30, "Failed to open indirect file "param"."
      DO WHILE STREAM(param) = 'READY'
         CALL ParseArg LINEIN(param)
         END
      CALL STREAM param, 'c', 'close'
      RETURN
      END
   IF LEFT(ARG(1), 1) \= '/' & LEFT(ARG(1), 1) \= '-' THEN DO
      /* filename */
      kstat = kstat +1
      source.kstat = ARG(1)
      RETURN
      END
   /* option */
   param = TRANSLATE(SUBSTR(ARG(1), 2))
   SELECT
    WHEN param = '_' THEN
      opt.debug = 1 /* debug mode */
    WHEN param = 'S' THEN
      opt.sdir = 'S' /* subdirectories */
    WHEN LEFT(param, 1) = 'X' THEN DO
      opt.xcl = opt.xcl + 1  /* file eopt.xcludes */
      i = opt.xcl
      opt.xcl.i = TRANSLATE(SUBSTR(param, 2))
      END
    WHEN param = 'V0' | param = 'Q' THEN
      opt.mout = 0  /* quiet */
    WHEN param = 'V1' THEN
      opt.mout = 1  /* normal output */
    WHEN param = 'V2' | param = 'V' THEN
      opt.mout = 2  /* verbose output */
    WHEN param = 'F' THEN
      opt.fcomp = 1 /* compare file content */
    WHEN param = 'FY' THEN
      opt.syncsimilar = 1 /* compare file content */
    WHEN param = 'E' THEN
      opt.folder = 1 /* synchronize folders */
    WHEN param = 'D' THEN
      opt.nodel = 0 /* delete */
    WHEN param = 'M' THEN
      opt.move = 1  /* move from 2nd */
    WHEN LEFT(param, 1) = 'T' THEN
      IF LENGTH(param) = 1 THEN
         opt.tz = InitTimezone() /* timezone corrections */
       ELSE IF param = 'TD' THEN DO
         opt.tz = InitTimezone() /* timezone corrections */
         opt.tz = "0 "SUBSTR(opt.tz, WORDINDEX(opt.tz, 2))
         END
       ELSE IF param = 'TZ' THEN
         opt.tz = WORD(InitTimezone(), 1) /* timezone corrections */
       ELSE
         opt.thres = SUBSTR(param, 2) /* threshold */
    WHEN param = 'L' THEN
      opt.longname = 1 /* longname EA */
    WHEN param = 'Y1' THEN
      opt.sync = 1
    WHEN param = 'Y2' THEN
      opt.sync = 2
    WHEN param = 'Y' | param = 'Y3' THEN
      opt.sync = 3
    WHEN param = 'I' THEN
      opt.ignoredowngrade = 1
    WHEN LEFT(param, 1) = 'U' THEN
      opt.updfile = param
    WHEN LEFT(param, 1) = 'P' THEN DO
      opt.volfile = 'VOLINFO.DIR'
      PARSE VALUE SUBSTR(param, 2) WITH n':'v
      IF n \= '' THEN DO
         IF \DATATYPE(n, 'N') THEN
            CALL Error 42, 'The number of volumes at -'param' is not numeric'
         IF n < 1 THEN
            CALL Error 42, 'The number of volumes at -'param' is not valid'
         opt.numvol = n
         END
      IF v \= '' THEN DO
         IF \DATATYPE(v, 'N') THEN
            CALL Error 42, 'The volume number at -'param' is not numeric'
         IF v < 1 | v > opt.numvol THEN
            CALL Error 42, 'The volume number at -'param' is not valid'
         opt.volno = v
         END
      END
    WHEN LEFT(param, 2) = 'ZI' | LEFT(param, 2) = 'Z0' THEN DO
      opt.sdir = 'S' /* subdirectories */
      opt.mout = 0  /* quiet */
      opt.updfile = SUBSTR(param, 2)
      END
    WHEN LEFT(param, 2) = 'ZP' | LEFT(param, 2) = 'Z1' THEN DO
      opt.sdir = 'S' /* subdirectories */
      opt.mout = 0  /* quiet */
      opt.sync = 1
      opt.updfile = SUBSTR(param, 2)
      END
    WHEN LEFT(param, 2) = 'ZU' | LEFT(param, 2) = 'Z2' THEN DO
      opt.sdir = 'S' /* subdirectories */
      opt.mout = 0  /* quiet */
      opt.sync = 2
      opt.updfile = SUBSTR(param, 2)
      END
    WHEN LEFT(param, 2) = 'ZA' | LEFT(param, 2) = 'Z3' THEN DO
      opt.sdir = 'S' /* subdirectories */
      opt.mout = 0  /* quiet */
      opt.move = 1  /* move files from tree2 */
      opt.sync = 3
      opt.updfile = SUBSTR(param, 2)
      END
    WHEN param = '?' THEN
      opt.help = 1
    OTHERWISE
      CALL Error 44, "Invalid option "ARG(1)"."
      END
   RETURN

/* scan source tree
   ScanTree(tree#, filespec)
*/
ScanTree: PROCEDURE EXPOSE tree. opt. monsum
   i = ARG(1)
   /* get file tree */
   DROP tree.i
   IF opt.folder THEN
      sel = 'B'
    ELSE
      sel = 'F'
   CALL SysFileTree ARG(2), tree.i, sel"T"opt.sdir
   tree.i = tree.i.0
   /* convert internal format */
   DO j = 1 TO tree.i
      PARSE VAR tree.i.j time size attrib file
      file = STRIP(file)
      PARSE VALUE STREAM(file, 'c', 'query datetime') WITH month'-'day'-'year' 'hour':'min':'sec
      IF opt.longname THEN
         IF SysGetEA(file, '.LONGNAME', 'tmp') = 0 THEN
            IF tmp \= '' THEN
               file = DirSpec(file)'\'SUBSTR(tmp, 5)
      IF SUBSTR(attrib, 2, 1) = 'D' THEN /* directory */
         file = file'\'
      /* Y2k fix */
      tree.i.j = RIGHT(ToUnixTime(RIGHT(year+30,2)+1970, month, day, STRIP(hour), min, sec), 10, 0)" "RIGHT(size, 10, 0)" "file
      IF opt.debug THEN
         SAY "F: "tree.i.j
      END
   RETURN

/* handle difference
   DoItem(operation, filename [, additional info])
   operation:
     '*O' file in tree 1 newer
     'O*' file in tree 2 newer
     '*-' file in tree 2 not found
     '-*' file in tree 1 not found
     '>>' file in tree 1 is longer with same time stamp
     '<<' file in tree 2 is longer with same time stamp
     '<>' files have different content with same time stamp
     'EE' error during compare
     '==' files are identical (as far as checked)
   filename:
     filename including relative path to root
   additional info:
     in case of an error (operation = 'EE') this is written to the screen
*/
DoItem: PROCEDURE EXPOSE opt. root.
   IF opt.mout = 1 THEN DO
      /* machine readable output */
      IF (opt.sayeq | ARG(1) \= '==') & \(opt.ignoredowngrade & ((opt.sync = 1 & ARG(1) = '*O') | (opt.sync = 2 & ARG(1) = 'O*'))) THEN
         SAY ARG(1)" "ARG(2)
      END
    ELSE IF opt.mout = 2 THEN
      /* normal output */
      SELECT
       WHEN ARG(1) = 'O*' THEN
         SAY "File "ARG(2)" in "root.1" is older than in "root.2"."
       WHEN ARG(1) = '*O' THEN
         SAY "File "ARG(2)" in "root.1" is newer than in "root.2"."
       WHEN ARG(1) = '-*' THEN
         SAY "File "ARG(2)" is not found in "root.1"."
       WHEN ARG(1) = '*-' THEN
         SAY "File "ARG(2)" is not found in "root.2"."
       WHEN ARG(1) = '>>' THEN
         SAY "File "ARG(2)" in "root.1" is bigger."
       WHEN ARG(1) = '<<' THEN
         SAY "File "ARG(2)" in "root.1" is smaller."
       WHEN ARG(1) = '<>' THEN
         SAY "File "ARG(2)" has different content."
       WHEN ARG(1) = 'EE' THEN
         SAY "Error during compare of "ARG(2)": "ARG(3)"."
       WHEN ARG(1) = 'FF' THEN
         SAY "Error during synchronisation of "ARG(2)": "ARG(3)"."
       WHEN ARG(1) = '==' THEN
         IF opt.sayeq THEN
            SAY "File "ARG(2)" is identical in both trees."
       OTHERWISE
         SAY ARG(1)" "ARG(2)
         END

   IF opt.sync = 0 THEN
      RETURN
   /* synchronize */
   SELECT
    WHEN RIGHT(ARG(1), 1) = '*' THEN
      /* file in tree 1 is older or does not exist */
      IF opt.sync \= 2 THEN
         CALL Warning CopyFile(ARG(2), root.2, root.1)
       ELSE IF ARG(1) = 'O*' THEN DO
         IF \opt.ignoredowngrade & RIGHT(ARG(2), 1) \= '\'  THEN
            CALL Warning "Cannot handle downgrade of file "ARG(2)" automatically."
         END
       ELSE IF ARG(1) = '-*' & \opt.nodel THEN
         CALL Warning DeleteFile(ARG(2), root.2)
    WHEN LEFT(ARG(1), 1) = '*' THEN
      /* file in tree 2 is older or does not exist */
      IF opt.sync \= 1 THEN DO
         IF \opt.move | opt.updfile = '' THEN
            CALL Warning CopyFile(ARG(2), root.1, root.2)
          ELSE
            CALL Warning MoveFile(ARG(2), root.1, root.2)
         END
       ELSE IF ARG(1) = '*O' THEN DO
         IF \opt.ignoredowngrade & RIGHT(ARG(2), 1) \= '\' THEN
            CALL Warning "Cannot handle downgrade of file "ARG(2)" automatically."
         END
       ELSE IF ARG(1) = '*-' & \opt.nodel & opt.updfile = '' THEN
         CALL Warning DeleteFile(ARG(2), root.1)
    WHEN ARG(1) = '==' THEN
      IF opt.sync \= 1 & \opt.nodel & opt.updfile \= '' & opt.move THEN
         CALL DeleteFile ARG(2), root.1
    OTHERWISE
      IF opt.syncsimilar THEN DO
         IF opt.sync = 1 THEN
            CALL Warning CopyFile(ARG(2), root.2, root.1)
          ELSE IF opt.sync = 2 THEN
            CALL Warning CopyFile(ARG(2), root.1, root.2)
         END
       ELSE
         CALL Warning "Synchronize failed ("ARG(1)"): "ARG(2)
      END
   RETURN

/* fetch next name from list index
   NextName(index)
*/
NextName: PROCEDURE EXPOSE i1 i2 tree. opt. offset.
   i = ARG(1)
   DO p = VALUE('i'i) +1 TO tree.i
      CALL VALUE 'i'i, p
      name = SUBSTR(tree.i.p, offset.i)
      /* check exclude list */
      IF \CheckXcl(TRANSLATE(name)) THEN
         RETURN name
      END
   RETURN ''

/* check if name is in exclude list
   CheckXcl(name)
*/
CheckXcl: PROCEDURE EXPOSE opt.
   DO i = 1 TO opt.xcl
      IF StringMatchQ(ARG(1), opt.xcl.i) THEN
         RETURN 1
      END
   RETURN 0


/****************************************************************************
   file functions
****************************************************************************/

/* remove file
   DeleteFile(file, source)
   removes file source||file and remove the directory if empty
*/
DeleteFile: PROCEDURE EXPOSE opt. root.
   file = ARG(2)ARG(1)
   IF opt.debug THEN DO
      SAY "DEL: "file
      IF RIGHT(ARG(1), 1) = '\' THEN DO /* directory */
         /* schedule for deletion, since we can't delete with content */
         i = root.3.0 +1
         root.3.i = STRIP(file, 'T', '\')
         root.3.0 = i
         END
      RETURN ''
      END
   IF RIGHT(ARG(1), 1) = '\' THEN DO /* directory */
      /* schedule for deletion, since we can't delete with content */
      i = root.3.0 +1
      root.3.i = STRIP(file, 'T', '\')
      root.3.0 = i
/*      IF opt.longname THEN DO
         IF SysDestroyObject(file) = 0 THEN
            RETURN "Removal of "file" failed."
         END
       ELSE DO
         RC = SysRmDir(file)
         IF RC \= 0 THEN
            RETURN "Removal of "file" failed (rc = "RC")."
         END*/
      END
    ELSE DO
      IF opt.longname THEN DO
         IF SysDestroyObject(file) = 0 THEN
            RETURN "Removal of "file" failed."
         END
       ELSE DO
         opt.delcmd' "'PCTesc(file)'" 2>NUL'
         IF RC \= 0 THEN
            RETURN "Removal of "file" failed (rc = "RC")."
         END
      /* remove directory if empty */
      IF \opt.folder THEN
         CALL CheckDirRM file
      END
   RETURN ''

/* copy file
   CopyFile(file, source, destination)
   copies source||file to destination||file
   If the destination directory does not exist it is created.
*/
CopyFile: PROCEDURE EXPOSE opt.
   dst = DirSpec(ARG(3)ARG(1))
   CALL Warning CheckDir(dst)
   IF opt.debug THEN DO
      SAY "COPY: "ARG(2)ARG(1)" "dst
      RETURN ''
      END
   IF RIGHT(ARG(1), 1) = '\' THEN /* directory */
      /* We can't copy directories, but we can copy their attributes */
      RETURN CopyDirAttr(DirSpec(ARG(2)ARG(1)), dst)
    ELSE DO
      IF opt.longname THEN DO
         IF SysCopyObject(ARG(2)ARG(1), dst) = 0 THEN
            RETURN "Copy of "ARG(2)ARG(1)" to "dst" failed."
         END
       ELSE DO
         opt.copycmd' "'PCTesc(ARG(2)ARG(1)'" "'dst)'" >NUL'
         IF RC \= 0 THEN
            RETURN "Copy of "ARG(2)ARG(1)" to "dst" failed (rc = "RC")."
         END
      END
   RETURN ''

/* move file
   MoveFile(file, source, destination)
   moves source||file to destination||file
   If the destination directory does not exist it is created.
*/
MoveFile: PROCEDURE EXPOSE opt. root.
   dst = DirSpec(ARG(3)ARG(1))
   CALL Warning CheckDir(dst)
   IF opt.debug THEN DO
      SAY "MOVE: "ARG(2)ARG(1)" "dst
      IF RIGHT(ARG(1), 1) = '\' THEN DO /* directory */
         src = STRIP(ARG(2)ARG(1), 'T', '\')
         /* schedule source directory for deletion */
         i = root.3.0 +1
         root.3.i = src
         root.3.0 = i
         END
      RETURN ''
      END
   IF RIGHT(ARG(1), 1) = '\' THEN DO /* directory */
      src = STRIP(ARG(2)ARG(1), 'T', '\')
      /* schedule source directory for deletion */
      i = root.3.0 +1
      root.3.i = src
      root.3.0 = i
      RETURN CopyDirAttr(src, dst)
      END
    ELSE DO
      IF opt.longname THEN DO
         IF SysMoveObject(ARG(2)ARG(1), dst) = 0 THEN
            RETURN "Move of "ARG(2)ARG(1)" to "dst" failed."
         END
   /*  I had no luck with move
       ELSE IF LEFT(ARG(2), POS(':',ARG(2)':')) = LEFT(ARG(3), POS(':',ARG(3))) THEN DO
         opt.movecmd' "'PCTesc(ARG(2)ARG(1)'" "'dst)'" >NUL'
         IF RC \= 0 THEN
            RETURN "Move of "ARG(2)ARG(1)" to "dst" failed (rc = "RC")."
         END */
       ELSE DO
         /* well, somtimes move may not work ... */
         opt.copycmd' "'PCTesc(ARG(2)ARG(1)'" "'dst)'" >NUL'
         IF RC \= 0 THEN
            RETURN "Copy (move) of "ARG(2)ARG(1)" to "dst" failed (rc = "RC")."
         opt.delcmd' "'PCTesc(ARG(2)ARG(1))'" >NUL'
         IF RC \= 0 THEN
            CALL Warning "Removal of "ARG(2)ARG(1)" during move to "dst" failed (rc = "RC")."
         END
      END
   /* remove directory if empty */
   IF \opt.folder THEN
      CALL CheckDirRM ARG(2)ARG(1)
   RETURN ''

/* remove diretory
   DeleteDir(dir)
*/
DeleteDir: PROCEDURE EXPOSE opt.
   IF opt.debug THEN DO
      SAY "RMDIR: "ARG(1)
      RETURN ''
      END
   /* IF opt.longname THEN DO   well, we should implement another way... */
   RC = SysRmDir(ARG(1))
   IF RC \= 0 THEN
      RETURN "Failed to remove directory "ARG(1)" (rc = "RC")."
   RETURN ''

/* Copy directory attributes
   CopyDirAttr(sourcedir, destdir)
*/
CopyDirAttr: PROCEDURE EXPOSE opt.
   opt.saveea' "'PCTesc(ARG(1))'" %TEMP%\dsyncea.tmp'
   IF RC \= 0 THEN
      RETURN "Failed to save extenden attributes of "ARG(1)" (rc = "RC")."
   opt.loadea' "'PCTesc(ARG(2))'" %TEMP%\dsyncea.tmp'
   IF RC \= 0 THEN
      RETURN "Failed to assign extended attributes to "ARG(1)" (rc = "RC")."
   RETURN ''

/* create directory if neccessary
   CheckDir(path)
   This function calls itself recursively.
*/
CheckDir: PROCEDURE EXPOSE opt.
   IF ARG(1) = '' THEN RETURN ''
   IF VALUE('opt.dir.'TRANSLATE(ARG(1),,'   !"#$%&'||"'()+,-./:;=@[\]^`{|}~",'_'), '') = '' THEN
      RETURN ''
   CALL SysFileTree ARG(1), res, 'DO'
   IF res.0 \= 0 THEN
      RETURN ''
   /* directory does not exist */
   CALL CheckDir DirSpec(ARG(1))
   IF RESULT \= '' THEN
      RETURN RESULT
   IF opt.debug THEN
      SAY "MKDIR: "ARG(1)
    ELSE DO
      CALL SysMkDir ARG(1)
      IF RESULT \= 0 THEN
         RETURN "Create directory "ARG(1)" failed ("RESULT")"
      END
   RETURN ''

/* remove parent directory if empty
   CheckDirRM(path)
*/
CheckDirRM: PROCEDURE EXPOSE root.
   dir = DirSpec(ARG(1))
   IF dir = '' | VERIFY(RIGHT(dir, 1), ': ') = 0 | dir'\' = root.1 | dir'\' = root.2 THEN
      RETURN
   DO WHILE SysRmDir(dir) = 0
      dir = DirSpec(dir)
      END
   RETURN


/****************************************************************************
   hash distribution map functions
****************************************************************************/

mmapinit: PROCEDURE EXPOSE mmap.
   mmap.0 = 1
   mmap.1 = '0 4294967296 0'
   mmap.parts = 1
   mmap.scale = 4294967296
   RETURN

/* view statistics
*/
mmanalysis: PROCEDURE EXPOSE mmap.
   DO i = 0 TO mmap.parts-1
      size.i = 0
      frags.i = 0
      END
   nsize = 0
   DO i = 1 TO mmap.0
      nsize = nsize +1
      tmp = WORD(mmap.i, 3)
      size.tmp = size.tmp + word(mmap.i, 2)
      frags.tmp = frags.tmp +1
      END
   SAY "Part.	Size	Fragments"
   frags = 0
   DO i = 0 TO mmap.parts-1
      SAY i"	"size.i"	"frags.i
      frags = frags + frags.i
      END
   SAY "========================"
   SAY mmap.parts"	"mmap.scale"	"frags
   SAY
   RETURN

/* dump map table
*/
mmdump: PROCEDURE EXPOSE mmap.
   SAY "Parts:	"mmap.parts
   SAY "Divisor:	"mmap.scale
   SAY "Offset	Length	Part"
   DO i = 1 TO mmap.0
      SAY TRANSLATE(mmap.i, '	', ' ')
      END
   RETURN

/* increment number of parts
*/
mmincrease: PROCEDURE EXPOSE mmap.
   mmap.parts = mmap.parts + 1
   /* calculate goal and padding */
   pad = mmap.scale // mmap.parts /* all part no >= $mmparts - $pad */
   goal = (mmap.scale - pad) / mmap.parts
   /* sort fragments by part ... */
   DO i = 1 TO mmap.0
      ndx.i = i
      END
   CALL mmqsort 1, mmap.0
   /* generate newpart out of the others */
   cur = 1
   DO part = 0 TO mmap.parts-2
      /* calculate current size of part */
      sum = 0
      DO i = cur TO mmap.0 WHILE WORD(VALUE('mmap.'ndx.i), 3) = part
         sum = sum + WORD(VALUE('mmap.'ndx.i), 2)
         END
      diff = sum - goal - (part >= mmap.parts - pad)
      next = i
      /* SAY "D: "part sum diff next */
      DO i = cur WHILE WORD(VALUE('mmap.'ndx.i), 2) <= diff
         diff = diff - WORD(VALUE('mmap.'ndx.i), 2)
         CALL VALUE 'mmap.'ndx.i, SUBWORD(VALUE('mmap.'ndx.i), 1, 2)' 'mmap.parts-1
         END
      IF diff \= 0 THEN /* split next fragment */
         CALL VALUE 'mmap.'ndx.i, VALUE('mmap.'ndx.i)' 'diff /* tag only */
      cur = next
      END
   DROP ndx.
   /* now split tagged fragments */
   DO i = 1 WHILE i <= mmap.0
      cur = mmap.i
      IF WORDS(cur) > 3 THEN DO
         IF i > 1 THEN DO
            last = VALUE('mmap.'i-1)
            IF WORD(last, 3) = mmap.parts-1 THEN DO
               /* cut and merge at front */
               CALL VALUE 'mmap.'i-1, WORD(last, 1)' 'WORD(last, 2)+WORD(cur, 4)' 'mmap.parts-1
               mmap.i = WORD(cur, 1)+WORD(cur, 4)' 'WORD(cur, 2)-WORD(cur, 4)' 'WORD(cur, 3)
               ITERATE
               END
            END
         /* cut at back */
         mmap.i = WORD(cur, 1)' 'WORD(cur, 2)-WORD(cur, 4)' 'WORD(cur, 3)
         i = i +1
         DO j = mmap.0 TO i BY -1
            CALL VALUE 'mmap.'j+1, mmap.j
            END
         mmap.0 = mmap.0 +1
         mmap.i = WORD(cur, 1)+WORD(cur, 2)-WORD(cur, 4)' 'WORD(cur, 4)' 'mmap.parts-1
         ITERATE
         END
      IF i > 1 THEN DO
         last = VALUE('mmap.'i-1)
         IF WORD(last, 3) = WORD(cur, 3) THEN DO
            /* merge */
            CALL VALUE 'mmap.'i-1, WORD(last, 1)' 'WORD(last, 2)+WORD(cur,2)' 'WORD(cur, 3)
            mmap.0 = mmap.0 -1
            DO j = i TO mmap.0
               mmap.j = VALUE('mmap.'j+1)
               END
            i = i -1
            END
         END
      END
   RETURN

/* sort fragment index by size and location
   arg(1)   start index
   arg(2)   stop index (inclusive)
*/
mmqsort: PROCEDURE EXPOSE mmap. ndx.
   lo = ARG(1)
   hi = ARG(2)
   mi = (hi + lo) %2
   m = VALUE('mmap.'ndx.mi)
   /*SAY lo hi mi m*/
   DO WHILE hi >= lo
      DO WHILE mmecmp(VALUE('mmap.'ndx.lo), m) < 0
         lo = lo +1
         END
      DO WHILE mmecmp(VALUE('mmap.'ndx.hi), m) > 0
         hi = hi -1
         END
      IF hi >= lo THEN DO
         tmp = ndx.lo
         ndx.lo = ndx.hi
         ndx.hi = tmp
         lo = lo +1
         hi = hi -1
         END
      END
   IF hi > ARG(1) THEN
      CALL mmqsort ARG(1), hi
   IF ARG(2) > lo THEN
      CALL mmqsort lo, ARG(2)
   RETURN

/* internal compare function */
mmecmp: PROCEDURE
   IF WORD(ARG(1), 3) < WORD(ARG(2), 3) THEN
      RETURN -1
   IF WORD(ARG(1), 3) > WORD(ARG(2), 3) THEN
      RETURN 1
   IF WORD(ARG(1), 2) < WORD(ARG(2), 2) THEN
      RETURN -1
   IF WORD(ARG(1), 2) > WORD(ARG(2), 2) THEN
      RETURN 1
   IF WORD(ARG(1), 1) < WORD(ARG(2), 1) THEN
      RETURN -1
   IF WORD(ARG(1), 1) > WORD(ARG(2), 1) THEN
      RETURN 1
   RETURN 0


/* lookup part number by hash
   ARG(1)   hash
   RETURN   part
*/
mmlook: PROCEDURE EXPOSE mmap.
   l = 1
   r = mmap.0
   DO WHILE l < r
      m = (l+r+1) %2
      /* SAY "S: "l r m WORD(mmap.m, 1)"	"ARG(1) */
      IF ARG(1) < WORD(mmap.m, 1) THEN
         r = m-1
       ELSE
         l = m
      END
   /* SAY "F: "D2X(WORD(mmap.l, 1))' 'D2X(ARG(1))' 'D2X(WORD(mmap.l, 1) + WORD(mmap.l, 2)) */
   RETURN WORD(mmap.l, 3)


/****************************************************************************
   helper functions
****************************************************************************/

SHR: PROCEDURE
   RETURN D2C(C2D(ARG(1)) % 2, LENGTH(ARG(1)))

CRC32_Init: PROCEDURE EXPOSE CRC32_Table
   CRC32_Table = ''
   CRC32_Polynomial = 'EDB88320'x /* REXX -> big endian! */
   DO i = 0 TO 255
   	crc = D2C(i,4);
	   DO j = 0 TO 7
	      IF BITAND(crc, '00000001'x) \= '00000000'x THEN
		   	crc = BITXOR(SHR(crc), CRC32_Polynomial)
		    ELSE
            crc = SHR(crc)
         END
   	CRC32_Table = CRC32_Table||crc;
      END		
   RETURN

/* Calculate CRC32 checksum
   ARG(1)   string
   RETURN   checksum as integer
*/
CRC32: PROCEDURE EXPOSE CRC32_Table
   crc = 'FFFFFFFF'x
   DO i = 1 TO LENGTH(ARG(1))
      tabptr = C2D(BITXOR(RIGHT(crc, 1), SUBSTR(ARG(1), i,1)))
      crc = '00'x||LEFT(crc, 3) /* >>= 8 */
      crc = BITXOR(crc, SUBSTR(CRC32_Table, 1+4*tabptr, 4))
      END
   RETURN C2D(BITXOR(crc, 'FFFFFFFF'x))


/* sort directory tree
   fqsort(index, first, last)
   index tree table index
     1 = tree.1..., 2 = tree.2...
   first, last
     sort range tree.index.first to tree.index.last
   This function calls itself recursively.
*/
fqsort: PROCEDURE EXPOSE tree. offset.
   list = ARG(1)
   lo = ARG(2)
   hi = ARG(3)
   mi = (hi + lo) %2
   m = TRANSLATE(SUBSTR(tree.list.mi, offset.list))
   /*SAY lo hi mi m*/
   DO WHILE hi >= lo
      DO WHILE TRANSLATE(SUBSTR(tree.list.lo, offset.list)) < m
         lo = lo +1
         END
      DO WHILE TRANSLATE(SUBSTR(tree.list.hi, offset.list)) > m
         hi = hi -1
         END
      IF hi >= lo THEN DO
         tmp = tree.list.lo
         tree.list.lo = tree.list.hi
         tree.list.hi = tmp
         lo = lo +1
         hi = hi -1
         END
      END
   IF hi > ARG(2) THEN
      CALL fqsort list, ARG(2), hi
   IF ARG(3) > lo THEN
      CALL fqsort list, lo, ARG(3)
   RETURN

/* wildcard match
   StringMatchQ(string, template)
*/
StringMatchQ: PROCEDURE
   str = ARG(1)
   tmp = ARG(2)
 redo:
   IF LENGTH(str) = 0 THEN
      RETURN VERIFY(tmp, '*') = 0 /* '' matches only if template consists only of '*' */
   p = VERIFY(tmp, '?*', 'M')
   IF p = 0 THEN
      RETURN COMPARE(str, tmp, '*') = 0 /* no wildcards => compare strings */
   IF COMPARE(LEFT(str, p-1), LEFT(tmp, p-1, '*')) \= 0 THEN
      RETURN 0 /* compare of non-wildcard section failed */
   j = p
   DO i = p TO VERIFY(tmp' ', '*?',, p +1) -1 /* count # of ? */
      IF SUBSTR(tmp, i, 1) = '?' THEN
         j = j +1
      END
   IF j > LENGTH(str)+1 THEN
      RETURN 0 /* more ? than length of str */
   tmp = SUBSTR(tmp, i)
   str = SUBSTR(str, j)
   IF i = j THEN
      SIGNAL redo
   /* '*' */
   IF LENGTH(tmp) = 0 THEN
      RETURN 1 /* nothing after '*' */
   DO p = 1
      p = POS(LEFT(tmp, 1), str, p)
      IF p = 0 THEN
         RETURN 0 /* character after '*' not found */
      IF StringMatchQ(SUBSTR(str, p+1), SUBSTR(tmp, 2)) THEN
         RETURN 1 /* got it! */
      END

InitTimezone: PROCEDURE
   monsum = "0 31 59 90 120 151 181 212 243 273 304 334 365"
   PARSE VALUE VALUE('TZ',,'OS2ENVIRONMENT') WITH tz','sm','sw','sd','st','em','ew','ed','et','offset
   tz = SUBSTR(tz, 4)
   p = VERIFY(tz, '-0123456789:')
   IF p = 0 THEN
      offset = 0
    ELSE
      tz = LEFT(tz, p -1)
   PARSE VAR tz hour':'min':'sec
   IF min = '' THEN min = 0
   IF sec = '' THEN sec = 0
   IF hour = '' THEN hour = 0
   IF hour < 0 THEN DO
      min = -min
      sec = -sec
      END
   tz = 60*(60*hour + min) + sec
   IF offset = 0 THEN
      RETURN tz
   IF sm = '' THEN sm = 4
   IF sw = '' THEN sw = 1
   IF sd = '' THEN sd = 0
   IF st = '' THEN st = 3600
   IF em = '' THEN em = 10
   IF ew = '' THEN ew = -1
   IF ed = '' THEN ed = 0
   IF et = '' THEN et = 7200
   IF offset = '' THEN offset = 3600
   IF sw = 0 THEN
      st = st + 86400 * (WORD(monsum, sm) + sd-1) + offset
    ELSE DO
      IF sw < 0 THEN
         sm = sm +1
       ELSE
         sw = sw -1
      st = st + 86400 * (WORD(monsum, sm) + 7*sw) + offset
      sd = 11 - sd
      END
   IF ew = 0 THEN
      et = et + 86400 * (WORD(monsum, em) + ed-1) + offset
    ELSE DO
      IF ew < 0 THEN
         em = em +1
       ELSE
         ew = ew -1
      et = et + 86400 * (WORD(monsum, em) + 7*ew) + offset
      ed = 11 - ed
      END
   RETURN tz offset st st + 86400 * (sm >= 3) sd et et + 86400 * (em >= 3) ed

/* convert to unix time
   ToUnixTime(year, month, day, hour, minute, second)
*/
ToUnixtime: PROCEDURE EXPOSE opt.
   monsum = "0 31 59 90 120 151 181 212 243 273 304 334 365"
   year70 = ARG(1) - 1970;
   /* check
   if ( tm->Sec > 59U || tm->Min > 59U || tm->Hour > 23U
       || (unsigned)year70 >= 136U || (tm->Month-1U) > 11U
       || (tm->Day-1) >= monlen[tm->Month-1]
       || (tm->Day == 29 && tm->Month == 2 && (year70&3)) )
      return false;*/
   ysec = ARG(6) + 60*ARG(5) + 3600*ARG(4) + 86400*(ARG(3)-1 + WORD(monsum, ARG(2)) + (ARG(2) >= 3 & year70//4 == 2))
   ydays = 365*year70 + (year70+1)%4
   PARSE VAR opt.tz delta offset sn ss sd en es ed
   tmp = ysec + 86400*ydays + delta
   IF offset = '' THEN RETURN tmp
   /* daylight saving correction */
   IF year70//4 == 2 THEN DO
      sn = ss
      en = es
      END
   IF sd \= 0 THEN
      sn = sn + 86400*(6 - (ydays + sn%86400 + sd) // 7)
   IF ed \= 0 THEN
      en = en + 86400*(6 - (ydays + en%86400 + ed) // 7)
   IF BITXOR(ysec < sn, ysec >= en) \= 1 THEN /* XOR -> even works on the southern hemisphere */
      tmp = tmp - offset
/*   SAY uc-tmp uc tmp ARG(4)':'ARG(5)':'ARG(6) ARG(3)'.'ARG(2)'.'ARG(1)*/
   RETURN tmp

DaylightErr: PROCEDURE EXPOSE dlerr
   dlerr = dlerr + 1
   IF dlerr \= 3 THEN RETURN
   CALL LINEOUT STDERR, "Warning: several time stamps differ exactly by a few hole hours."
   CALL LINEOUT STDERR, " Maybe there is a problem with the time zone or with daylight saving."
   RETURN

/* quote % characters */
PCTesc: PROCEDURE
   tmp = ARG(1)
   p = POS('%', tmp)
   DO WHILE p \= 0
      tmp = INSERT('%', tmp, p)
      p = POS('%', tmp, p+2)
      END
   RETURN tmp

/* locate file (in PATH) */
FindFilePath: PROCEDURE
   /* first check .\ */
   tmp = STREAM(ARG(1), 'c', 'query exists')
   IF tmp = '' THEN
      tmp = SysSearchPath('PATH', ARG(1))
   RETURN tmp

DirSpec: PROCEDURE
   RETURN STRIP(FILESPEC('D', ARG(1))FILESPEC('P', ARG(1)), 'T', '\')

Error: PROCEDURE
   CALL LINEOUT STDERR, ARG(2)
   EXIT ARG(1)

Warning: PROCEDURE
   IF ARG(1) \= '' THEN
      CALL LINEOUT STDERR, ARG(1)
   RETURN ARG(1)

