/****************************************************************************
*
*  MMPMtool.cpp
*
*  Utility to read and modify MMPMMMIO.INI
*
*  Create:  02.03.2002
*  Update:  02.03.2002
*  (C)      Marcel Mller 2002
*
****************************************************************************/


#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#define INCL_BASE
#define INCL_OS2MM
#define INCL_GPIBITMAPS
#define INCL_MMIO_CODEC
#include <os2.h>
#include <os2me.h>


// Error handling
struct Error
{  int err;
   char buf[1024];
   Error(int err, const char* fmt, ...);
};

Error::Error(int err, const char* fmt, ...)
:  err(err)
{  va_list va;
   va_start((void*&)va, fmt);
   vsprintf(buf, fmt, va);
   va_end(va);
}

static FILE* f;
static MMINIFILEINFO ii;
static CODECINIFILEINFO ci;
static int mode;
static char buf[512];

struct item
{  char  name[20];
   char  omask[14];
   char  imask[10];
   void* ip[2];
   int   repeat;
   int   inc;
} const itemlist[] =
{  { "FourCC:",          "%1.1s",        "%c%n",    { &ii.fccIOProc,    &ci.fcc              }, 4,1}
 , { "DLLName:",         "%s",           "%255s%n", { ii.szDLLName,     ci.szDLLName         }, 1,1}
 , { "ProcName:",        "%s",           "%31s%n",  { ii.szProcName,    ci.szProcName        }, 1,1}
 , { "CapsFlags:",       "0x%lX",        "%li%n",   { NULL,             &ci.ulCapsFlags      }, 1,4}
 , { "Flags:",           "0x%lX",        "%li%n",   { &ii.ulFlags,      &ci.ulFlags          }, 1,4}
 , { "ExtendLen:",       "%lu",          "%li%n",   { &ii.ulExtendLen,  NULL                 }, 1,4}
 , { "CompressType:",    "0x%lX (%.4s)", "%li%n",   { NULL,             &ci.ulCompressType   }, 1,4}
 , { "CompressSubType:", "%lu",          "%li%n",   { NULL,             &ci.ulCompressSubType}, 1,4}
 , { "MediaType:",       "%lu",          "%li%n",   { &ii.ulMediaType,  &ci.ulMediaType      }, 1,4}
 , { "IOProcType:",      "%lu",          "%li%n",   { &ii.ulIOProcType, NULL                 }, 1,4}
 , { "DefExt:",          "%s",           "%3s%n",   { ii.szDefExt,      NULL                 }, 1,1}
 , { "HWID:",            "%s",           "%31s%n",  { NULL,             &ci.szHWID           }, 1,1}
 , { "MaxSrcBufLen:",    "%lu",          "%li%n",   { NULL,             &ci.ulMaxSrcBufLen   }, 1,4}
 , { "SyncMethod:",      "%lu",          "%li%n",   { NULL,             &ci.ulSyncMethod     }, 1,4}
 , { "Alignment:",       "%lu ",         "%li%n",   { NULL,             &ci.ulXalignment     }, 2,4}
 , { "SpecInfo:",        "0x%08lX ",     "%li%n",   { NULL,             &ci.ulSpecInfo       }, 8,4}
};
static const char modname[][8] = {"$IOproc", "$Codec"};


void* memxchr(const void* buf, int c, size_t sz)
{  while (sz--)
   {  if (*(char*)buf != c)
         return (void*)buf;
      ++(char*&)buf;
   }
   return NULL;
}

void initiici()
{  if (mode)
   {  memset(&ci, '\0', sizeof ci);
      ci.ulStructLen = sizeof ci;
   } else
      memset(&ii, '\0', sizeof ii);
}

static void dump()
{  puts(modname[mode]);
   for (const item* p = itemlist; p < itemlist + sizeof(itemlist)/sizeof(*itemlist); ++p)
   {  const char* vp = (const char*) p->ip[mode];
      if (vp == NULL || memxchr(vp, 0, p->repeat * p->inc) == NULL)
         continue;
      printf("%-20s ", p->name);
      int n = p->repeat;
      while (n--)
      {  printf(p->omask, p->inc == 4 ? *(ULONG*)vp : (ULONG)vp, vp);
         vp += p->inc;
      }
      putchar('\n');
   }
   putchar('\n');
   fflush(stdout);
}

static void dumpall()
{  ULONG rc;
   ULONG flg = MMIO_FINDPROC|MMIO_MATCHFIRST;
   initiici();
   for(;;)
   {  rc = mode
       ? mmioIniFileCODEC(&ci, flg)
       : mmioIniFileHandler(&ii, flg);
      if (rc == 0xFFFFFFFF)
         break;
      if (rc)
         throw Error(1, "mmio error: %li", rc);
      dump();
      flg = MMIO_FINDPROC|MMIO_MATCHNEXT;
}  }

static inline void dumpall(int md)
{  mode = md;
   dumpall();
}

static int readline()
{  *buf = 0;
   int r = fscanf(f, "%511[^\n]", buf);
   fgetc(f);
   //fprintf(stderr, "r %i %i %s\n", r, feof(f), buf);
   return r || !feof(f);
}

static void scanitem()
{  for (const item* p = itemlist; p < itemlist + sizeof(itemlist)/sizeof(*itemlist); ++p)
   {  const char* vp = (const char*) p->ip[mode];
      if (vp == NULL || memicmp(buf, p->name, strlen(p->name)) != 0)
         continue;
      char* cp = buf + strlen(p->name);
      //printf("ID: %s_%s\n", p->name, cp);
      int nc;
      if (sscanf(cp, "%*1s%n", &nc) != 0)
         return;  // default
      cp += nc-1;
      int r = 0;
      int n = p->repeat;
      while (n--)
      {  r += sscanf(cp, p->imask, vp, &nc);
         cp += nc;
         vp += p->inc;
      }
      if (r == 0)
         throw Error(2, "Parameter invalid: %s", buf);
      return;
   }
   throw Error(2, "Unknown token %s", buf);
}

enum
{  rr_none,
   rr_NOP,
   rr_install,
   rr_remove
};

static int readrecord()
{  if (readline() == 0)
      return rr_none;

   char * cp = buf;
   int r = rr_NOP;
   if (memicmp(buf, "install", 7) == 0)
   {  r = rr_install;
      cp = buf +7;
   } else if (memicmp(buf, "remove", 6) == 0)
   {  r = rr_remove;
      cp = buf +6;
   }
   int nc = 0;
   sscanf(cp, "%*[ \t]%n", &nc);
   cp += nc;

   if (memicmp(cp, modname[0], 8) == 0)
      mode = 0;
    else if (memicmp(cp, modname[1], 7) == 0)
      mode = 1;
    else
      throw Error(1, "Unknown record type %s", buf);

   initiici();

   readline();
   do
   {  scanitem();
      readline();
   } while (*buf);

   if (ii.ulExtendLen)
      throw Error(3, "Extend record not supported");

   return r;
}

static void installentry()
{  fputs("install ", stdout);
   dump();
   ULONG rc = mode
    ? mmioIniFileCODEC(&ci, MMIO_INSTALLPROC)
    : mmioIniFileHandler(&ii, MMIO_INSTALLPROC);
   if (rc)
      throw Error(1, "mmio error: %li", rc);
}

static void removeentry()
{  fputs("remove ", stdout);
   dump();
   ULONG rc = mode
    ? mmioIniFileCODEC(&ci, MMIO_REMOVEPROC|MMIO_MATCHCOMPRESSTYPE|MMIO_MATCHCOMPRESSSUBTYPE
       |MMIO_MATCHHWID|MMIO_MATCHCAPSFLAGS|MMIO_MATCHFOURCC|MMIO_MATCHDLL|MMIO_MATCHPROCEDURENAME)
    : mmioIniFileHandler(&ii, MMIO_REMOVEPROC|MMIO_MATCHFOURCC|MMIO_MATCHDLL|MMIO_MATCHPROCEDURENAME);
   if (rc == 0xFFFFFFFF)
      fputs("Entry not found\n", stderr);
    else if (rc)
      throw Error(1, "mmio error: %li", rc);
}

// main

int main(int argc, char** argv)
{  try
   {  if (argc <= 1)
      {  dumpall(0);
         dumpall(1);
      } else
      {  // command file
         f = fopen(argv[1], "rt");
         if (f == NULL)
            throw Error(1, "Failed to open %s", argv[1]);
         for (;;)
         {  switch (readrecord())
            {case rr_install:
               installentry();
               continue;
             case rr_remove:
               removeentry();
               //continue;
             case rr_NOP:
               //dump();
               continue;
            }
            break;
         }
      }

   } catch (const Error& err)
   {  fprintf(stderr, "%s\n", err.buf);
      return err.err;
   }

   return 0;
}
