/**
***  This is a part of the DCFVBA project.   This set of functions handle
***  The PCL datastreams beginning with:  <Esc>&
**/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "dcfvba.h"

/* -------------------------------------------------------------------------- */

void pcl_ampersand(void)
   {
   char pcl_command;

   double number;  /* holds PCL argument */
   next_char();
   pcl_command = (char)toupper(*current);
   switch(pcl_command)
      {
      case 'A':
         {
         do
            {
            next_char();
            pcl_ampersand_A();
            }
         while(strchr(PCL_END_SET,*current) == NULL);
         break;
         } /* case 'A' */
      case 'D':
         {
         do
            {
            next_char();
            number = get_pcl_arg();
            pcl_ampersand_D(number);
            }
         while(strchr(PCL_END_SET,*current) == NULL);
         break;
         } /* case 'D' */
      case 'K': /* Font pitch */
         {
         do
            {
            next_char();
            number = get_pcl_arg();
            pcl_ampersand_K(number);
            }
         while(strchr(PCL_END_SET,*current) == NULL);
         break;
         } /* case 'K' */
      case 'L':
         {
         do
            {
            next_char();
            number = get_pcl_arg();
            pcl_ampersand_L(number);
            }
         while(strchr(PCL_END_SET,*current) == NULL);
         break;
         } /* case 'L' */
      case 'S': /* End of line wrap */
         {
         error("End-of-line wrap not supported.",0);
         break;
         }
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
         {
         number = get_pcl_arg();
         pcl_ampersand_number(number);
         break;
         }
      default:
         {
         char error_message[255];

         sprintf(error_message,"Unsupported PCL command: <Esc>&%c",*current);
         error(error_message,8);
         } /* default */
      } /* switch */
   } /* pcl_ampersand */

/* -------------------------------------------------------------------------- */

void pcl_ampersand_A(void)
   {
   double number;
   int  signed_arg = NOSIGN; /* Flag indicating whether the argument was signed */
   int  target_row;      /* Target output row on the page. */
   char pcl_char;

   switch(*current)
      {
      case '+':
         {
         signed_arg = POSITIVE;
         next_char();
         break;
         }
      case '-':
         {
         signed_arg = NEGATIVE;
         next_char();
         break;
         }
      } /* switch */

   number = get_pcl_arg();
   pcl_char = (char)toupper(*current);
   switch(pcl_char)
      {
      case 'C': /* Absolute column positioning */
         {
         switch(signed_arg)
            {
            case POSITIVE:
               {
               strpad((int)number+strlen(line));
               break;
               }
            case NOSIGN:
               {
               strpad((int)number);
               break;
               }
            case NEGATIVE:
               {
               strpad((int)number-strlen(line));
               break;
               }
            } /* switch(signed_arg) */
         break;
         } /* case 'C' */
      case 'H': /* Absolute horizontal positioning by decipoint */
         {
         switch(signed_arg)
            {
            case POSITIVE:
               {
               strpad(column_by_decipoint(number)+strlen(line));
               break;
               }
            case NOSIGN:
               {
               strpad(column_by_decipoint(number));
               break;
               }
            case NEGATIVE:
               {
               strpad(column_by_decipoint(number)-strlen(line));
               break;
               }
            } /* switch(signed_arg) */
         break;
         }
      case 'L': /* Left margin (column number) */
         {
         switch(signed_arg)
            {
            case POSITIVE:
               {
               lmargin += (int)number;
               break;
               }
            case NOSIGN:
               {
               lmargin = (int)number;
               break;
               }
            case NEGATIVE:
               {
               lmargin -= (int)number;
               break;
               }
            } /* switch(signed_arg) */
         lmargin = (int)number;
         break;
         }
      case 'M': /* Right margin (column number) */
         {
         switch(signed_arg)
            {
            case POSITIVE:
               {
               rmargin += (int)number;
               break;
               }
            case NOSIGN:
               {
               rmargin = (int)number;
               break;
               }
            case NEGATIVE:
               {
               rmargin -= (int)number;
               break;
               }
            } /* switch(signed_arg) */
         break;
         }
      case 'R': /* Absolute row number of page */
         {
         target_row = (int)number;
         switch(signed_arg)
            {
            case POSITIVE:
               {
               position_vertical(logical_vpos+target_row);
               break;
               }
            case NOSIGN:
               {
               if (target_row < logical_vpos)
                  {
                  error("Negative vertical positioning may not work as expected.",0);
                  logical_vpos = target_row;
                  }
               else
                  position_vertical(target_row);
               break;
               }
            case NEGATIVE:
               {
               error("Negative vertical positioning may not work as expected.",0);
               logical_vpos -= target_row;
               break;
               }
            } /* switch */
         break;
         } /* case 'V' */
      case 'V': /* Absolute vertical decipoint of page */
         {
         target_row = row_by_decipoint(number);
         switch(signed_arg)
            {
            case POSITIVE:
               {
               position_vertical(logical_vpos+target_row);
               break;
               }
            case NOSIGN:
               {
               if (target_row < logical_vpos)
                  {
                  error("Negative vertical positioning may not work as expected.",0);
                  logical_vpos = target_row;
                  }
               else
                  position_vertical(target_row);
               break;
               }
            case NEGATIVE:
               {
               error("Negative vertical positioning may not work as expected.",0);
               logical_vpos -= target_row;
               break;
               }
            } /* switch */
         break;
         } /* case 'V' */
      default:
         error("Invalid PCL data stream",0);
      } /* switch */
   } /* pcl_ampersand_A */

/* -------------------------------------------------------------------------- */

void pcl_ampersand_D(double number)
   {
   char pcl_char;

   pcl_char = (char)toupper(*current);
   switch(pcl_char)
      {
      case 'D':
         {
         if ((number == 1) ||
             (number == 0))
            {
            new_line();
            fputs(".us on\n",vba_file);
            underscore = 1;
            }
         else
            error("Invalid PCL datastream.",0);
         break;
         }
      case '@':
         {
         if (underscore)
            {
            new_line();
            fputs(".us off\n",vba_file);
            strcpy(line,".ct ");
            underscore = 0;
            }
         break;
         }
      default:
         error("Invalid PCL data stream",0);
      } /* switch */
   }

/* -------------------------------------------------------------------------- */

void pcl_ampersand_K(double number)
   {
   char pcl_char;

   pcl_char = (char)toupper(*current);
   switch(pcl_char)
      {
      case 'G': /* End of line termination */
         break;
      case 'S': /* Font pitch (alternate method) */
         {
         if (number != 0)  /* 10 CPI */
            error("Unsupported pitch",0);
         break;
         }
      default:
         error("Invalid PCL datastream",12);
      } /* switch */
   }

/* -------------------------------------------------------------------------- */

void pcl_ampersand_L(double number)
   {
   char pcl_char;

   pcl_char = (char)toupper(*current);
   switch(pcl_char)
      {
      case 'A': /* Page size */
         {
         if (number != 2)  /* Letter size paper */
            error("Unsupported paper size",0);
         break;
         }
      case 'C': /* Vertical motion index */
         {
         error("VMI PCL command not supported",0);
         break;
         }
      case 'D': /* Lines per inch */
         {
         if (number != 6)
            error("Only 6 LPI supported",0);
         break;
         }
      case 'E': /* Top margin */
         {
         error("Top margin specification not supported",0);
         break;
         }
      case 'F': /* Text length */
         {
         error("Text length PCL command not supported",0);
         break;
         }
      case 'H': /* Paper input control */
         {
         if (number == 0)  /* Eject page */
            new_page();
         break;
         }
      case 'L': /* Perf skip mode */
         {
         error("Perforation skip mode not supported",0);
         break;
         }
      case 'O': /* Orientation */
         {
         if (number == 1)
            error("Landscape orientation not supported",0);
         break;
         }
      case 'P': /* Page length (in number of lines) */
         {
         error("Page length PCL command unsupported",0);
         break;
         }
      case 'S': /* Duplex */
         {
         error("Duplexing not supported",0);
         break;
         }
      case 'X': /* Number of copies */
         {
         if (number != 1)
             error("'Copies' PCL command not supported",0);
         break;
         }
      default:
         error("Invalid PCL datastream",12);
      } /* switch */
   }

/* -------------------------------------------------------------------------- */

void pcl_ampersand_number(double number)
   {
   if (toupper(*current) == 'D')
      {
      if (number == 3)
         fputs(".us on\n",vba_file);
      else
         error("Invalid PCL data stream.",12);
      }
   else
      {
      char error_message[255];

      sprintf(error_message,"Unsupported PCL command: <Esc>&%f.0%c",number,*current);
      error(error_message,8);
      } /* default */
   }
