/*
 * xbin -- unpack BinHex format file into suitable
 * format for downloading with macput
 * Dave Johnson, Brown University Computer Science
 *
 * (c) 1984 Brown University
 * may be used but not sold without permission
 *
 * created ddj 12/16/84
 * revised ddj 03/10/85 -- version 4.0 compatibility, other minor mods
 * revised ddj 03/11/85 -- strip LOCKED bit from m_flags
 * revised ahm 03/12/85 -- System V compatibility
 * revised dba 03/16/85 -- (Darin Adler, TMQ Software)  4.0 EOF fixed,
 *			   4.0 checksum added
 * revised ddj 03/17/85 -- extend new features to older formats: -l, stdin
 * revised ddj 03/24/85 -- check for filename truncation, allow multiple files
 * revised ddj 03/26/85 -- fixed USG botches, many problems w/multiple files
 * revised jcb 03/30/85 -- (Jim Budler, amdcad!jimb), revised for compatibility
 *			   with 16-bit int machines
 * revised dl  06/16/85 -- (Dan LaLiberte, liberte@uiucdcs) character
 *			   translation speedup
 * revised ddj 09/30/85 -- fixed problem with run of RUNCHAR
 * revised mot 08/30/94 -- modified to strict ANSI
 */

static char version[] = "ANSI xbin.c Version 2.3a 08/30/94";

#define _POSIX_SOURCE

/* The only non-portable feature used here is timezone - deal with! */
#ifndef __IBMC__						/* For UNIX systems - non POSIX code */
	#define	_XOPEN_SOURCE
	#define	_timezone	timezone
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>

#ifndef TRUE
	#define	TRUE	1
#endif
#ifndef FALSE
	#define	FALSE	0
#endif

/* Mac time of 00:00:00 GMT, Jan 1, 1970 */
#define TIMEDIFF 0x7c25b080

#define DATABYTES 128

#define BYTEMASK	0xff
#define BYTEBIT	0x100
#define WORDMASK	0xffff
#define WORDBIT	0x10000

#define NAMEBYTES 63
#define H_NLENOFF 1
#define H_NAMEOFF 2

/* 65 <-> 80 is the FInfo structure */
#define H_TYPEOFF 65
#define H_AUTHOFF 69
#define H_FLAGOFF 73

#define H_LOCKOFF 81
#define H_DLENOFF 83
#define H_RLENOFF 87
#define H_CTIMOFF 91
#define H_MTIMOFF 95

#define H_OLD_DLENOFF 81
#define H_OLD_RLENOFF 85

#define F_BUNDLE 0x2000
#define F_LOCKED 0x8000

struct macheader {
	char m_name[NAMEBYTES+1];
	char m_type[4];
	char m_author[4];
	short m_flags;
	long m_datalen;
	long m_rsrclen;
	long m_createtime;
	long m_modifytime;
} mh;

struct filenames {
	char f_info[256];
	char f_data[256];
	char f_rsrc[256];
} files;

int pre_beta = FALSE;						/* options */
int listmode = FALSE;
int verbose  = FALSE;

int compressed;	/* state variables */
int qformat;
FILE *ifp;

/* ANSI Prototypes */
void setup_files(char *filename, char *macname);
void print_header(void);
void process_forks(void);
void forge_info(void);
int find_header(void);
void do_q_header(char *macname);
void do_q_fork(char *fname, long len);
void verify_crc(unsigned int calc_crc, unsigned int file_crc);
void q_init(void);
short get2q(void);
long get4q(void);
int getqbuf(char *buf, int n);
int getq(void);
int getq_nocrc(void);
char getq_raw(void);
int get6bits(void);
void comp_q_crc(unsigned int c);
void do_o_header(char *macname, char *filename);
void do_o_forks(void);
long make_file(char *fname, int compressed);
void comp_c_crc(unsigned char c);
void comp_e_crc(unsigned char c);
int comp_to_bin(char ibuf[], FILE *outf);
int hex_to_bin(char ibuf[], FILE *outf);
char hexit(int c);
void put2(char *bp, short value);
void put4(char *bp, long value);
void PrintUsage(FILE *funit);

char usage[] = "usage: \"xbin [-?] [-v] [-l] [-o] [-b] [-n name] [-] filename\"\n";

main(int argc, char *argv[]) {

	char *filename, *macname;

	filename = ""; macname = "";
	argc--; argv++;
	while (argc) {
		if (*argv[0] == '-') {
			switch (argv[0][1]) {
				case '\0':
					filename = "-";			break;
				case 'v':
					verbose = TRUE;			break;
				case 'l':
					listmode = TRUE;			break;
				case 'o':
					pre_beta = TRUE;			break;
				case 'n':
					if (argc <= 1) goto bad_usage;
					argc--; argv++;
					macname = argv[0];
					filename = "";
					break;
				case '?':
				case 'h':
					PrintUsage(stdout);
					return(EXIT_SUCCESS);
				default:
					fprintf(stderr, "Option %s unrecognized\n", argv[0]);
					goto bad_usage;
			}
		} else {
			filename = argv[0];
		}

		if (*filename != '\0') {
			setup_files(filename, macname);
			if (listmode) {
				print_header();
			} else {
				process_forks();		/* now that we know the size of the forks */
				forge_info();
			}
			if (ifp != stdin) fclose(ifp);
			macname = "";
			ifp = NULL;					/* reset state */
			qformat = 0;
			compressed = 0;
		}
		argc--; argv++;
	}
	if (*filename != '\0') return(EXIT_SUCCESS);

bad_usage:
	fprintf(stderr, usage);
	return(EXIT_FAILURE);
}


/* ===========================================================================
   char *filename;	input file name -- extension optional
   char *macname;		name to use on the mac side of things
=========================================================================== */
void setup_files(char *filename, char *macname) {

	static char *extensions[] = {".hqx", ".hcx",	".hex", "",	NULL};
	char namebuf[256], *np;
	char **ep;
	int n;
	struct stat stbuf;
	long curtime;

#ifndef NAME_MAX
	int NAME_MAX=0;
#endif

	if (filename[0] == '-') {
		ifp = stdin;
		filename = "stdin";
	} else {
		/* find input file and open it */
		for (ep=extensions; *ep!=NULL; ep++) {
			sprintf(namebuf, "%s%s", filename, *ep);
			if (stat(namebuf, &stbuf) == 0) break;
		}
		if (*ep == NULL) {
			perror(namebuf);
			exit(-1);
		}
		ifp = fopen(namebuf, "r");
		if (ifp == NULL) {
			perror(namebuf);
			exit(-1);
		}
	}
	if (ifp == stdin) {
		curtime = time(0);
		mh.m_createtime = curtime;
		mh.m_modifytime = curtime;
	} else {
		mh.m_createtime = stbuf.st_mtime;
		mh.m_modifytime = stbuf.st_mtime;
	}
	if (listmode || verbose) {
		fprintf(stderr, "%s %s%s",
			listmode ? "\nListing" : "Converting",
			namebuf, listmode ? ":\n" : " ");
	}

	qformat = find_header(); /* eat mailer header &cetera, intuit format */

	if (qformat) {
		do_q_header(macname);
	} else {
		do_o_header(macname, filename);
	}

#ifndef NAME_MAX										/* Use value as if written here */
	NAME_MAX = pathconf(".", _PC_NAME_MAX);
#endif

	/* make sure host file name doesn't get truncated beyond recognition */
	n = strlen(mh.m_name);
	if (n > NAME_MAX - 2) n = NAME_MAX - 2;
	strncpy(namebuf, mh.m_name, n);
	namebuf[n] = '\0';

	/* get rid of troublesome characters */
	for (np = namebuf; *np; np++)
		if (*np == ' ' || *np == '/')	*np = '_';

	sprintf(files.f_data, "%s.data", namebuf);
	sprintf(files.f_rsrc, "%s.rsrc", namebuf);
	sprintf(files.f_info, "%s.info", namebuf);
	if (verbose)
		fprintf(stderr, "==> %s.{info,data,rsrc}\n", namebuf);
	return;
}

/* print out header information in human-readable format */
void print_header(void) {

	printf("macname: %s\n", mh.m_name);
	printf("filetype: %.4s, ", mh.m_type);
	printf("author: %.4s, ", mh.m_author);
	printf("flags: 0x%x\n", mh.m_flags);
	if (qformat) {
		printf("data length: %ld, ", mh.m_datalen);
		printf("rsrc length: %ld\n", mh.m_rsrclen);
	}
	if (!pre_beta) {
		printf("create time: %s", ctime(&mh.m_createtime));
	}
	return;
}

void process_forks(void) {

	if (qformat) {
		/* read data and resource forks of .hqx file */
		do_q_fork(files.f_data, mh.m_datalen);
		do_q_fork(files.f_rsrc, mh.m_rsrclen);
	} else {
		do_o_forks();
	}
	return;
}

/* write out .info file from information in the mh structure */
void forge_info(void) {

	static char buf[DATABYTES];
	char *np;
	FILE *fp;
	int n;
	long tdiff;
	struct tm *tp;
	long bs;

	for (np = mh.m_name; *np; np++)
		if (*np == '_') *np = ' ';

	buf[H_NLENOFF] = n = np - mh.m_name;
	strncpy(buf + H_NAMEOFF, mh.m_name, n);
	strncpy(buf + H_TYPEOFF, mh.m_type, 4);
	strncpy(buf + H_AUTHOFF, mh.m_author, 4);
	put2(buf + H_FLAGOFF, mh.m_flags & ~F_LOCKED);
	if (pre_beta) {
		put4(buf + H_OLD_DLENOFF, mh.m_datalen);
		put4(buf + H_OLD_RLENOFF, mh.m_rsrclen);
	} else {
		put4(buf + H_DLENOFF, mh.m_datalen);
		put4(buf + H_RLENOFF, mh.m_rsrclen);

		/* convert unix file time to mac time format */
		/* I hope this is right! -andy */
		time(&bs);
		tp = localtime(&bs);
		tdiff = TIMEDIFF - _timezone;
		if (tp->tm_isdst)
			tdiff += 60 * 60;
		put4(buf + H_CTIMOFF, mh.m_createtime + tdiff);
		put4(buf + H_MTIMOFF, mh.m_modifytime + tdiff);
	}
	fp = fopen(files.f_info, "wb");
	if (fp == NULL) {
		perror("info file");
		exit(-1);
	}
	fwrite(buf, 1, DATABYTES, fp);
	fclose(fp);
	return;
}

/* eat characters until header detected, return which format */
int find_header(void) {

	int c, at_bol;
	char ibuf[BUFSIZ];

	/* look for "(This file ...)" line */
	while (fgets(ibuf, BUFSIZ, ifp) != NULL) {
		if (strncmp(ibuf, "(This file", 10) == 0)
			break;
	}
	at_bol = 1;
	while ((c = getc(ifp)) != EOF) {
		switch (c) {
		case '\n':
		case '\r':
			at_bol = 1;
			break;
		case ':':
			if (at_bol)	/* q format */
				return 1;
			break;
		case '#':
			if (at_bol) {	/* old format */
				ungetc(c, ifp);
				return 0;
			}
			break;
		default:
			at_bol = 0;
			break;
		}
	}

	fprintf(stderr, "unexpected EOF\n");
	exit(2);
	/* NOTREACHED */
	return(-1);
}

static unsigned int crc;

short get2q();
long get4q();

/* read header of .hqx file */
void do_q_header(char *macname) {

	char namebuf[256];		/* big enough for both att & bsd */
	int n;
	unsigned int calc_crc, file_crc;

	crc = 0;			/* compute a crc for the header */
	q_init();			/* reset static variables */

	n = getq();			/* namelength */
	n++;				/* must read trailing null also */
	getqbuf(namebuf, n);		/* read name */
	if (macname[0] == '\0')
		macname = namebuf;

	n = strlen(macname);
	if (n > NAMEBYTES)
		n = NAMEBYTES;
	strncpy(mh.m_name, macname, n);
	mh.m_name[n] = '\0';

	getqbuf(mh.m_type, 4);
	getqbuf(mh.m_author, 4);
	mh.m_flags = get2q();
	mh.m_datalen = get4q();
	mh.m_rsrclen = get4q();

	comp_q_crc(0);
	comp_q_crc(0);
	calc_crc = crc;
	file_crc = get2q();
	verify_crc(calc_crc, file_crc);
	return;
}

void do_q_fork(char *fname, long len) {

	FILE *outf;
	register int c, i;
	unsigned int calc_crc, file_crc;

	outf = fopen(fname, "wb");
	if (outf == NULL) {
		perror(fname);
		exit(-1);
	}

	crc = 0;	/* compute a crc for a fork */

	if (len)
		for (i = 0; i < len; i++) {
			if ((c = getq()) == EOF) {
				fprintf(stderr, "unexpected EOF\n");
				exit(2);
			}
			putc(c, outf);
		}

	comp_q_crc(0);
	comp_q_crc(0);
	calc_crc = crc;
	file_crc = get2q();
	verify_crc(calc_crc, file_crc);
	fclose(outf);
	return;
}

/* verify_crc(); -- check if crc's check out */
void verify_crc(unsigned int calc_crc, unsigned int file_crc) {

	calc_crc &= WORDMASK;
	file_crc &= WORDMASK;

	if (calc_crc != file_crc) {
		fprintf(stderr, "CRC error\n---------\n");
		fprintf(stderr, "CRC in file:\t0x%x\n", file_crc);
		fprintf(stderr, "calculated CRC:\t0x%x\n", calc_crc);
		exit(3);
	}
	return;
}

static int eof;
static char obuf[3];
static char *op, *oend;

/* initialize static variables for q format input */
void q_init(void) {

	eof = 0;
	op = obuf;
	oend = obuf + sizeof obuf;
	return;
}

/* get2q(); q format -- read 2 bytes from input, return short */
short get2q(void) {

	register int c;
	short value = 0;

	c = getq();
	value = (c & BYTEMASK) << 8;
	c = getq();
	value |= (c & BYTEMASK);

	return(value);
}

/* get4q(); q format -- read 4 bytes from input, return long */
long get4q(void) {

	register int c, i;
	long value = 0L;

	for (i = 0; i < 4; i++) {
		c = getq();
		value <<= 8;
		value |= (c & BYTEMASK);
	}
	return(value);
}

/* getqbuf(); q format -- read n characters from input into buf */
/*		All or nothing -- no partial buffer allowed */
int getqbuf(char *buf, int n) {

	register int c, i;

	for (i = 0; i < n; i++) {
		if ((c = getq()) == EOF)
			return EOF;
		*buf++ = c;
	}
	return 0;
}

#define RUNCHAR 0x90

/* q format -- return one byte per call, keeping track of run codes */
int getq(void) {

	register int c;

	if ((c = getq_nocrc()) == EOF)
		return EOF;
	comp_q_crc((unsigned)c);
	return(c);
}

int getq_nocrc(void) {

	static int rep, lastc;
	int c;

	if (rep) {
		rep--;
		return(lastc);
	}
	if ((c = getq_raw()) == EOF) {
		return(EOF);
	}
	if (c == RUNCHAR) {
		if ((rep = getq_raw()) == EOF)
			return(EOF);
		if (rep != 0) {
			/* already returned one, about to return another */
			rep -= 2;
			return(lastc);
		} else {
			lastc = RUNCHAR;
			return RUNCHAR;
		}
	} else {
		lastc = c;
		return(c);
	}
}

/* q format -- return next 8 bits from file without interpreting run codes */
char getq_raw(void) {

	char ibuf[4];
	register char *ip = ibuf, *iend = ibuf + sizeof ibuf;
	int c;

	if (op == obuf) {
		for (ip = ibuf; ip < iend; ip++) {
			if ((c = get6bits()) == EOF)
				if (ip <= &ibuf[1])
					return EOF;
				else if (ip == &ibuf[2])
					eof = 1;
				else
					eof = 2;
			*ip = c;
		}
		obuf[0] = (ibuf[0] << 2 | ibuf[1] >> 4);
		obuf[1] = (ibuf[1] << 4 | ibuf[2] >> 2);
		obuf[2] = (ibuf[2] << 6 | ibuf[3]);
	}
	if ((eof) & (op >= &obuf[eof]))
		return EOF;
	c = *op++;
	if (op >= oend)
		op = obuf;
	return (c & BYTEMASK);
}

/*
char tr[] = "!\"#$%&'()*+,-012345689@ABCDEFGHIJKLMNPQRSTUVXYZ[`abcdefhijklmpqr";
	     0 123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef
	     0                1               2               3 
trlookup is used to translate by direct lookup.  The input character
is an index into trlookup.  If the result is 0xFF, a bad char has been read.
Added by:  Dan LaLiberte, liberte@uiucdcs.Uiuc.ARPA, ihnp4!uiucdcs!liberte
*/
char trlookup[83] = { 	0xFF, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
			0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0xFF, 0xFF,
			0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0xFF,
			0x14, 0x15, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
			0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D,
			0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0xFF,
			0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0xFF,
			0x2C, 0x2D, 0x2E, 0x2F, 0xFF, 0xFF, 0xFF, 0xFF,
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0xFF,
			0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0xFF, 0xFF,
			0x3D, 0x3E, 0x3F };

/* q format -- decode one byte into 6 bit binary */
int get6bits(void) {

	register int c;
	register int tc;

	while (1) {
		c = getc(ifp);
		switch (c) {
		case '\n':
		case '\r':
		case '\t':
		case ' ':
			continue;
		case ':':
		case EOF:
			return EOF;
		default:
		 	tc = ((c-' ') < 83) ? trlookup[c-' '] : 0xff;
/*			fprintf(stderr, "c = '%c'  tc = %4x\n", c, tc); */
			if (tc != 0xff)
				return (tc);
			fprintf(stderr, "bad char: '%c'\n", c);
			return EOF;
		}
	}
	return(0);
}


#define CRCCONSTANT 0x1021

void comp_q_crc(unsigned int c) {

	register int i;
	register unsigned long temp = crc;

	for (i=0; i<8; i++) {
		c <<= 1;
		if ((temp <<= 1) & WORDBIT)
			temp = (temp & WORDMASK) ^ CRCCONSTANT;
		temp ^= (c >> 8);
		c &= BYTEMASK;
	}
	crc = temp;
	return;
}

/* old format -- process .hex and .hcx files */
void do_o_header(char *macname, char *filename) {

	char namebuf[256];		/* big enough for both att & bsd */
	char ibuf[BUFSIZ];
	int n;

	/* set up name for output files */
	if (macname[0] == '\0') {
		strcpy(namebuf, filename);

		/* strip directories */
		macname = strrchr(namebuf, '/');
		if (macname == NULL)
			macname = namebuf;
		else
			macname++;

		/* strip extension */
		n = strlen(macname);
		if (n > 4) {
		    n -= 4;
		    if (macname[n] == '.' && macname[n+1] == 'h'
					    && macname[n+3] == 'x')
			    macname[n] = '\0';
		}
	}
	n = strlen(macname);
	if (n > NAMEBYTES)
		n = NAMEBYTES;
	strncpy(mh.m_name, macname, n);
	mh.m_name[n] = '\0';

	/* read "#TYPEAUTH$flag"  line */
	if (fgets(ibuf, BUFSIZ, ifp) == NULL) {
		fprintf(stderr, "unexpected EOF\n");
		exit(2);
	}
	n = strlen(ibuf);
	if (n >= 7 && ibuf[0] == '#' && ibuf[n-6] == '$') {
		if (n >= 11)
			strncpy(mh.m_type, &ibuf[1], 4);
		if (n >= 15)
			strncpy(mh.m_author, &ibuf[5], 4);
		sscanf(&ibuf[n-5], "%4hx", &mh.m_flags);
	}
	return;
}

void do_o_forks(void) {

	char ibuf[BUFSIZ];
	int forks = 0, found_crc = 0;
	unsigned int calc_crc, file_crc;

	crc = 0;	/* calculate a crc for both forks */

	/* create empty files ahead of time */
	close(creat(files.f_data, 0666));
	close(creat(files.f_rsrc, 0666));

	while (!found_crc && fgets(ibuf, BUFSIZ, ifp) != NULL) {
		if (forks == 0 && strncmp(ibuf, "***COMPRESSED", 13) == 0) {
			compressed++;
			continue;
		}
		if (strncmp(ibuf, "***DATA", 7) == 0) {
			mh.m_datalen = make_file(files.f_data, compressed);
			forks++;
			continue;
		}
		if (strncmp(ibuf, "***RESOURCE", 11) == 0) {
			mh.m_rsrclen = make_file(files.f_rsrc, compressed);
			forks++;
			continue;
		}
		if (compressed && strncmp(ibuf, "***CRC:", 7) == 0) {
			found_crc++;
			calc_crc = crc;
			sscanf(&ibuf[7], "%x", &file_crc);
			break;
		}
		if (!compressed && strncmp(ibuf, "***CHECKSUM:", 12) == 0) {
			found_crc++;
			calc_crc = crc & BYTEMASK;
			sscanf(&ibuf[12], "%x", &file_crc);
			file_crc &= BYTEMASK;
			break;
		}
	}

	if (found_crc)
		verify_crc(calc_crc, file_crc);
	else {
		fprintf(stderr, "missing CRC\n");
		exit(3);
	}
	return;
}

long make_file(char *fname, int compressed) {

	char ibuf[BUFSIZ];
	FILE *outf;
	register long nbytes = 0L;

	outf = fopen(fname, "wb");
	if (outf == NULL) {
		perror(fname);
		exit(-1);
	}

	while (fgets(ibuf, BUFSIZ, ifp) != NULL) {
		if (strncmp(ibuf, "***END", 6) == 0)
			break;
		if (compressed)
			nbytes += comp_to_bin(ibuf, outf);
		else
			nbytes += hex_to_bin(ibuf, outf);
	}

	fclose(outf);
	return nbytes;
}

void comp_c_crc(unsigned char c) {
	crc = (crc + c) & WORDMASK;
	crc = ((crc << 3) & WORDMASK) | (crc >> 13);
	return;
}

void comp_e_crc(unsigned char c) {
	crc += c;
	return;
}

#define SIXB(c) (((c)-0x20) & 0x3f)

int comp_to_bin(char ibuf[], FILE *outf) {

	char oobuf[BUFSIZ];
	register char *ip = ibuf;
	register char *oop = oobuf;
	register int n, outcount;
	int numread, incount;

	numread = strlen(ibuf);
	ip[numread-1] = ' ';		/* zap out the newline */
	outcount = (SIXB(ip[0]) << 2) | (SIXB(ip[1]) >> 4);
	incount = ((outcount / 3) + 1) * 4;
	for (n = numread; n < incount; n++)	/* restore lost spaces */
		ibuf[n] = ' ';

	n = 0;
	while (n <= outcount) {
		*oop++ = SIXB(ip[0]) << 2 | SIXB(ip[1]) >> 4;
		*oop++ = SIXB(ip[1]) << 4 | SIXB(ip[2]) >> 2;
		*oop++ = SIXB(ip[2]) << 6 | SIXB(ip[3]);
		ip += 4;
		n += 3;
	}

	for (n=1; n <= outcount; n++)
		comp_c_crc((unsigned)oobuf[n]);

	fwrite(oobuf+1, 1, outcount, outf);
	return outcount;
}

int hex_to_bin(char ibuf[], FILE *outf) {
	register char *ip = ibuf;
	register int n, outcount;
	int c;

	n = strlen(ibuf) - 1;
	outcount = n / 2;
	for (n = 0; n < outcount; n++) {
		c = hexit(*ip++);
		comp_e_crc((unsigned)(c = (c << 4) | hexit(*ip++)));
		fputc(c, outf);
	}
	return outcount;
}

char hexit(int c) {

	if ('0' <= c && c <= '9')
		return c - '0';
	if ('A' <= c && c <= 'F')
		return c - 'A' + 10;

	fprintf(stderr, "illegal hex digit: %c", c);
	exit(4);
	return(-1);				/* NOTREACHED */
}

void put2(char *bp, short value) {
	*bp++ = (value >> 8) & BYTEMASK;
	*bp++ = value & BYTEMASK;
	return;
}

void put4(char *bp, long value) {

	register int i, c;

	for (i = 0; i < 4; i++) {
		c = (value >> 24) & BYTEMASK;
		value <<= 8;
		*bp++ = c;
	}
	return;
}

void PrintUsage(FILE *funit) {

	fprintf(funit,
"     xbin  -  convert mailable format BinHex file into binary\n"
"\n"
"     Xbin converts a file created by BinHex (usually named with one of the\n"
"     extensions .hex, .hcx, or .hqx) into three host-system files (.info,\n"
"     .data, .rsrc) suitable for use on a *real* computer (or send to a Mac).\n"
"\n"
"     Usage: xbin [ -o ] [ -v ] [ -l ] [[ -n name ] file] ...  \n"
"\n"
"         -l (list)    reads & prints header info without creating files\n"
"         -v (verbose) prints line for each file converting indicating names\n"
"         -o (old)     specifies 'old' (version -0.15X) MacTerminal mode.\n"
"         -n (name)    specifies host names (must precede input filename).\n"
"                      Without option, name is derived from input filename\n"
"                      or name in header.  HPFS capabilities are assumed\n"
"\n"
"        o  The file '-' causes the input file to be taken from stdin\n"
"        o  Headers and signatures will be ignored when converting\n"
"        o  Input files must contain a line beginning '(This file' to mark\n"
"           the beginning of the BinHex information.\n"
"\n"
"     Author: Dave Johnson, Brown University 12/16/84\n"
"             CRC handling code by Darin Adler, TMQ Software 3/16/85\n"
"             Current port Michael Thompson (mot1@cornell.edu) 8/30/94\n"
	);
	return;
}
