/* REXX
 *
 * del_ea.cmd deletes extended attributes of all files and directories
 * of a specified path.
 *
 * Error codes: 0 - success
 *              1 - usage
 *              2 - calculation of temporary file for eas failed
 *              3 - eautil execution failed
 *
 * Author: Heiko Nitzsche
 * Date  : 20-August-2004
 * Since : 22-August-2004:
 *           -> add options /f, /r and /d
 *           -> fix some error handling issues
 *         26-August-2004:
 *           -> add /R option for EAUTIL
 *           -> add support for working path files/directories with white spaces ("path\mask")
 *           -> add support for directories with white spaces in the name for EAUTIL
 *         14-August-2008:
 *           -> move EAs to be deleted to NUL instead to temp file
 */

call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
call SysLoadFuncs

/******************* Parse and validate parameters. ******************/

parse arg fullOptions

sourcePath  = ''
fullOptions = strip(fullOptions)
options     = fullOptions

/* check that at least 1 argument exists */
numOptions = words(options)
if (numOptions < 1) then
do
   call usage('Error: Working directory not specified.')
   exit 1
end

/* Look for pathname with white spaces encapsulated by "path". */
beginPathPos = 0
endPathPos   = lastpos('"', options)

/* if trailing " found ... */
if (endPathPos > 0) then
do
   /* ... look for leading " */
   beginPathPos = lastpos('"', options, endPathPos - 1)

   /* and if found ... */
   if (beginPathPos > 0) then
   do
      /* extract source path and strip options */
      lengthPath = endPathPos - beginPathPos - 1
      if (lengthPath > 0) then
      do
         sourcePath = substr(options, beginPathPos + 1, lengthPath)
         options    = left(options, beginPathPos - 1)

         /* update numOptions but add 1 because the analyzer loop below expects a filename */
         numOptions = words(options) + 1
      end
      else
      do
         call usage('Error: Incomplete name of working directory.')
         exit 1
      end
   end
   else
   do
      call usage('Error: Incomplete name of working directory.')
      exit 1
   end
end

/* check for additional options */
fileOnlySearchMask = 'FO'
dirOnlySearchMask  = 'DO'
fileDirSearchMask  = 'BO'
recurseOption      = 'S'
searchMask         = fileDirSearchMask

/* check if additional options were specified */
if (numOptions > 1) then
do
   /* convert to capital letters */
   capOptions = translate(options)

   hasFoption = 0 /* files only       */
   hasDoption = 0 /* directories only */
   hasRoption = 0 /* recursive        */

   /* Loop over all option words minus the one for the directory at the end */
   do i = 1 to (numOptions - 1)

      option = word(capOptions, i)

      /* Check for /F or -F options (files only) */
      isFoption = ((option = '/F') | (option = '-F'))

      /* Check for /D or -D options (directories only) */
      isDoption = ((option = '/D') | (option = '-D'))

      /* Check for /R or -R options (recursive) */
      isRoption = ((option = '/R') | (option = '-R'))

      if ((isFoption = 0) & (isDoption = 0) & (isRoption = 0)) then
      do
         call usage('Error: Unknown option "'word(options, i)'" specified.')
         exit 1
      end

      /* Check for /F or -F options (files only) */
      hasFoption = hasFoption | isFoption

      /* Check for /D or -D options (directories only) */
      hasDoption = hasDoption | isDoption

      /* Check for /R or -R options (recursive) */
      hasRoption = hasRoption | isRoption

   end /* do */

   /* Check for exclusive options */
   if (hasFoption & hasDoption) then
   do
       call usage('Error: The option /f and /d are mutual exclusive.')
       exit 1
   end

   /* set search mask for SysFileTree (files only) */
   if (hasFoption) then
   do
      searchMask = fileOnlySearchMask
   end

   /* set search mask for SysFileTree (directories only) */
   if (hasDoption) then
   do
      searchMask = dirOnlySearchMask
   end

   /* add search mask recursive option for SysFileTree if requested */
   if (hasRoption) then
   do
      searchMask = searchMask''recurseOption
   end

   /* get sourcepath directory (the rest) */
   beginPathPos = wordindex(fullOptions, numOptions-1) + wordlength(fullOptions, numOptions-1)
   sourcePath   = right(fullOptions, length(fullOptions) - beginPathPos)
end
else
do
   /* get sourcepath directory */
   sourcePath = fullOptions
end

/* strip white spaces enclosing the path if specified */
sourcePath = strip(sourcePath)

/* strip "s enclosing the path if specified */
sourcePath = strip(sourcePath,,'"')

/****************************** Execute ****************************/

/* Find all files by pattern. */
retVal = SysFileTree(sourcePath, 'filenames', searchMask)
if (\ (retVal = 0)) then
do
   call usage('Error: Illegal directory specified.')
   exit 1
end

/* Check if any files were found. */
if (filenames.0 > 0) then
do
   say

   /* Remove EAs */
   do i = 1 to filenames.0
      say '"'filenames.i'"'
      address CMD '@eautil "' || filenames.i || '" NUL /S /R'
      if (\ (rc = 0)) then
      do
         say
         say 'Error during removing EAs.'
         exit 3
      end
   end
end
else
do
   say
   say 'No matching files found.'
end

say
say 'Operation complete.'
exit 0

/********************************************************/

usage:procedure

   parse arg errorMessage

   say
   say errorMessage
   say
   say 'This program removes all EAs from all files/directories of a specified'
   say 'directory. A filemask like for the dir command can be optionally added.'
   say
   say 'Usage: del_ea.cmd [/f] [/d] [/r] <working directory with optional filemask>'
   say
   say '  /f - Remove EAs only from files'
   say '  /d - Remove EAs only from directories'
   say '  /r - Include subdirectories (recursive)'
   say
   say '  If /f and /d are not specified, EAs will be removed from directories'
   say '  and files. The options /f and /d are mutual exclusive.'
   say
   say '  Examples:'
   say '    del_ea /f /r x:\images\*.jpg (EAs of all .jpg files recursive)'
   say '    del_ea /d x:\images          (EAs of directory x:\images)'
   say '    del_ea /f x:\images\         (EAs of all files in x:\images)'
   say '    del_ea /d x:\images\         (EAs of all directories in x:\images)'
   say
   say '(Copyright: Heiko Nitzsche, Version: 1.2, 14-August-2008)'

   return

