/*
    Animated Mouse Pointer
    Copyright (C) 1997 Christian Langanke

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

*/
// C Runtime
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <stdio.h>
#include <ctype.h>

// OS/2 Toolkit
#define  INCL_ERRORS
#define  INCL_PM
#define  INCL_WIN
#define  INCL_DOS
#define  INCL_DOSDEVIOCTL
#define  INCL_DOSMISC
#include <os2.h>

#include "mptrppl.h"
#include "mptrset.h"
#include "mptrptr.h"
#include "mptrcnr.h"
#include "mptrutil.h"
#include "mptranim.h"
#include "wmuser.h"
#include "wpamptr.rch"
#include "pointer.h"
#include "cursor.h"
#include "dll.h"
#include "script.h"
#include "macros.h"
#include "debug.h"

// reload unterbinden
// #define SURPRESS_RELOAD

// global data
// Array fr Pointerlisten
static POINTERLIST appl[ NUM_OF_SYSCURSORS];
static BOOL fPointerlistInitialized = FALSE;

// Daten fr Single Animationthread
static HMTX hmtxDataAccess     = NULLHANDLE;
static HEV  hevStartup         = NULLHANDLE;
static TID  tidAnimation       = 0;

// global vars
static BOOL fDemoActive        = FALSE;

/*Ŀ
 * access mutex sem handle - accessed by REQUEST_DATA_ACCESS* macros      
 *
 */

HMTX QueryDataAccessSem() {
  return hmtxDataAccess;
}

BOOL
SetDataAccessSem( HMTX hmtx )
{
  hmtxDataAccess = hmtx;
  return TRUE;
}

/*Ŀ
 * Name      : ResetAnimation                                             
 * Kommentar : Animation ausstellen und ersten Animationspointer setzen   
 * Autor     : C.Langanke                                                 
 * Datum     : 24.07.1996                                                 
 * nderung  : 24.07.1996                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : -                                                          
 * Aufgaben  : - deinitialisieren                                         
 * Rckgabe  : APIRET - OS/2 Fehlercode                                   
 *
 */

APIRET
ResetAnimation( PPOINTERLIST ppl, BOOL fSetPointer )
{
  APIRET    rc = NO_ERROR;
  PICONINFO piconinfo;

  do
  {
    // Parameter prfen
    if( ppl == NULL ) {
      rc = ERROR_INVALID_PARAMETER;
      break;
    }

    // Animationsflag lschen
    ppl->fAnimate = FALSE;

    // ersten Pointer aus Liste setzen
    piconinfo = &ppl->iconinfoStatic;
    if( piconinfo->fFormat == 0 ) {
      DEBUGMSG( "warning: no static ptr available" NEWLINE, 0 );
    } else {
      // etwas warten, dann klappt es immer
      DosSleep( 50 );
      if( !WinSetSysPointerData( HWND_DESKTOP,
                                 ppl->ulPtrId,
                                 piconinfo )) {
        rc = ERRORIDERROR( WinGetLastError( WinQueryAnchorBlock( HWND_DESKTOP )));
        //       printf( "error setting static pointer rc=%u/0x%08x" NEWLINE, rc , rc);
        //       BEEP;
      }
    }

    // Arrow-PTR direkt setzen
    if( fSetPointer ) {
      WinSetPointer( HWND_DESKTOP, ppl->hptrStatic );
    }
  } while( FALSE );


  return rc;
}

/*Ŀ
 * Name      : LoadAnimouseModule                                         
 * Kommentar : ldt AnimouseModule (DosLoadModule)                        
 * Autor     : C.Langanke                                                 
 * Datum     : 21.03.1998                                                 
 * nderung  : 21.03.1998                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : ULONG        - Index fr Pointer                           
 *             PPOINTERLIST - Zielvariable fr PointerSet-Daten           
 *             PSZ          - Name der Animationsdatei (voll              
 *                            qualifiziert) oder Set-Name.                
 *             ###                                                        
 * Aufgaben  : - Pointer laden                                            
 * Rckgabe  : BOOL - Flag: Pointerset geladen/nicht geladen              
 *
 */

APIRET
LoadAnimouseModule( PSZ pszModuleName, PHMODULE phmodResource )
{
  APIRET rc = NO_ERROR;
  CHAR   szLoadError[ _MAX_PATH];
  CHAR   szFullname[ _MAX_PATH];

  do
  {
    // is a full qualified pathname given ?
    if( !IsFilenameValid( pszModuleName, FILENAME_CONTAINSFULLNAME, NULL )) {
      // query the full name
      rc = DosQueryPathInfo( pszModuleName,
                             FIL_QUERYFULLNAME,
                             szFullname,
                             sizeof( szFullname ));
      if( rc != NO_ERROR ) {
        break;
      }

      pszModuleName = szFullname;
    }

    // load the module
    rc = DosLoadModule( szLoadError, sizeof( szLoadError ),
                        pszModuleName, phmodResource );
  } while( FALSE );

  return rc;
}

/*Ŀ
 * Name      : QueryFileDetails                                           
 * Kommentar : generic file detail load function                          
 * Autor     : C.Langanke                                                 
 * Datum     : 14.07.1996                                                 
 * nderung  : 14.07.1996                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : ###                                                        
 * Aufgaben  : - Details holen                                            
 * returns   : APIRET - OS/2 error code                                   
 *
 */

APIRET
QueryFileDetails( PSZ pszAnimationFileName, ULONG ulResourceType, ULONG ulInfoLevel,
                  PVOID pvData, ULONG ulBuflen, PULONG pulSize )
{
  APIRET rc = NO_ERROR;

  switch( ulResourceType )
  {
    default:
      rc = ERROR_INVALID_PARAMETER;
      break;

    case RESFILETYPE_DEFAULT:
      if( pvData ) {
        memset( pvData, 0, ulBuflen );
        *pulSize = ulBuflen;
      } else {
        *pulSize = 0;
      }
      rc = NO_ERROR;
      break;

    case RESFILETYPE_POINTER:
    case RESFILETYPE_POINTERSET:
      rc = QueryPointerFileDetails( pszAnimationFileName, ulInfoLevel, pvData, ulBuflen, pulSize );
      break;

    case RESFILETYPE_CURSOR:
    case RESFILETYPE_CURSORSET:
      rc = QueryCursorFileDetails( pszAnimationFileName, ulInfoLevel, pvData, ulBuflen, pulSize );
      break;

    case RESFILETYPE_ANIMOUSE:
    case RESFILETYPE_ANIMOUSESET:
      rc = QueryAnimouseFileDetails( pszAnimationFileName, ulInfoLevel, pvData, ulBuflen, pulSize );
      break;

    case RESFILETYPE_WINANIMATION:
    case RESFILETYPE_WINANIMATIONSET:
      rc = QueryWinAnimationFileDetails( pszAnimationFileName, ulInfoLevel, pvData, ulBuflen, pulSize );
      break;
  }

  return rc;
}

/*Ŀ
 * Name      : LoadPointerAnimation                                       
 * Kommentar : ldt Pointer oder Pointerset Animation                     
 * Autor     : C.Langanke                                                 
 * Datum     : 14.07.1996                                                 
 * nderung  : 14.07.1996                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : ULONG        - Index fr Pointer                           
 *             PPOINTERLIST - Zielvariable fr PointerSet-Daten           
 *             PSZ          - Name der Animationsdatei (voll              
 *                            qualifiziert) oder Set-Name.                
 *             ###                                                        
 * Aufgaben  : - Pointer laden                                            
 * Rckgabe  : BOOL - Flag: Pointerset geladen/nicht geladen              
 *
 */

BOOL
LoadPointerAnimation( ULONG ulPtrIndex, PPOINTERLIST ppl, PSZ pszName,
                      BOOL fLoadSet, BOOL fEnableIfAnimation, BOOL fRefresh )
{
  BOOL         fSuccess      = FALSE;
  BOOL         fLoaded       = FALSE;
  BOOL         fLoadError    = FALSE;
  BOOL         fPointerFound = FALSE;
  BOOL         fSetComplete  = FALSE;
  BOOL         fIsSet        = FALSE;
  BOOL         fIsCursor     = FALSE;
  BOOL         fIsAniMouse   = FALSE;
  BOOL         fIsWinAnim    = FALSE;

  APIRET       rc;
  ULONG        ulFileType    = 0;
  ULONG        ulFirstPtr, ulLastPtr;

  POINTERLIST  plTmp;
  PPOINTERLIST pplTmp = &plTmp;
  PPOINTERLIST pplndx;

  ULONG        i, j, k, l;
  CHAR         szFullName[_MAX_PATH];
  CHAR         szSearchMask[_MAX_PATH];
  CHAR         szStaticName[_MAX_PATH];

  static PSZ   apszSearchFormat[] = { "%s\\%s.and",
                                      "%s\\%s.ani",
                                      "%s\\%s%03u.ptr",
                                      "%s\\%s%03u.cur",
                                      "%s\\%s.ptr",
                                      "%s\\%s.cur" };

  static ULONG afIsCursor[]    = { FALSE, FALSE, FALSE, TRUE,  FALSE, TRUE  };
  static ULONG afIsAniMouse[]  = { TRUE,  FALSE, FALSE, FALSE, FALSE, FALSE };
  static ULONG afIsWinAnim[]   = { FALSE, TRUE,  FALSE, FALSE, FALSE, FALSE };

  static ULONG afIsSet[]       = { FALSE, FALSE, TRUE,  TRUE,  FALSE, FALSE };
  static ULONG afSetComplete[] = { FALSE, FALSE, FALSE, FALSE, TRUE,  TRUE  };

  ULONG        ulFormatCount =  sizeof( apszSearchFormat ) / sizeof( PSZ );

  PSZ          pszUsedSearchFormat;
  ULONG        ulNumerationValue = 0;
  ULONG        ulBootDrive       = 0;
  PVOID        pvSourceInfo      = NULL;
  PSOURCEINFO  psi;
  ULONG        ulSourceInfoLen;

  do {
    // Parameter prfen
    if( ppl == NULL ) {
      break;
    }

    if( pszName == NULL ) {
      ulFileType     = RESFILETYPE_DEFAULT;
      szFullName[ 0] = 0;
    } else {
      if( !QueryResFileType( pszName, &ulFileType, szFullName, sizeof( szFullName ))) {
        break;
      }
    }

    // prfen, ob ein Set geladen werden soll,
    // dann mu der Dateityp stimmen
    if( fLoadSet ) {
      if(( ulFileType == RESFILETYPE_POINTER       ) ||
         ( ulFileType == RESFILETYPE_CURSOR        ) ||
         ( ulFileType == RESFILETYPE_WINANIMATION  )) {
        break;
      }
    }

    // Bootdrive und Maske fr Dateisuche ermitteln
    DosQuerySysInfo( QSV_BOOT_DRIVE, QSV_BOOT_DRIVE, &ulBootDrive, sizeof( ULONG ));

    //
    // jetzt Dateien suchen und eintragen
    //

    // range fr Verarbeitung festlegen
    ulFirstPtr = ulPtrIndex;

    if( fLoadSet ) {
      ulLastPtr  = NUM_OF_SYSCURSORS;
    } else {
      ulLastPtr  = ulPtrIndex + 1;
    }

    for( i = ulFirstPtr, pplndx = ppl; i < ulLastPtr; i++, pplndx++ )
    {
#ifdef SURPRESS_RELOAD
      // ist das richtige schon geladen ?
      if( ulFileType != RESFILETYPE_DEFAULT ) {
        if( strcmp( pplndx->szAnimationName, szFullName ) == 0 ) {
          continue;
        }
      }
#endif

      // temporre Pointerliste initialisieren
      memset( pplTmp, 0, sizeof( POINTERLIST ));
      strcpy( pplTmp->szAnimationName, szFullName );
      strcpy( pplTmp->szAnimationFile, szFullName );
      pplTmp->ulPtrIndex = i;
      pplTmp->ulPtrId    = QueryPointerSysId( i );

      fSetComplete   = FALSE;
      fIsSet         = FALSE;
      fPointerFound  = FALSE;

      for( j = 0; ( j < MAX_COUNT_POINTERS && ( !fSetComplete )); j++ ) {
        switch( ulFileType ) {
          case RESFILETYPE_DEFAULT:
            pplTmp->ulPtrCount    = 1;
            pplTmp->hptr[ j]      = 0;
            fSetComplete          = TRUE;
            fPointerFound         = TRUE;
            break;

          case RESFILETYPE_POINTER:
          case RESFILETYPE_CURSOR:
            if( IsFilenameValid( pplTmp->szAnimationName, FILENAME_CONTAINSNUMERATION, &ulNumerationValue )) {
              // get next file in numeration
              strcpy( szSearchMask, pplTmp->szAnimationName );

              ChangeFilename( szSearchMask, CHANGE_SETNUMERATION,
                              szSearchMask, sizeof( szSearchMask ),
                              NULL, 0, j + ulNumerationValue );

              // does file exist ?
              if( FileExist( szSearchMask, FALSE )) {
                // load file
                fLoaded = ( ulFileType == RESFILETYPE_POINTER ) ?
                          LoadPointerFromPointerFile( szSearchMask,
                                                      &pplTmp->hptr[ j],
                                                      &pplTmp->iconinfo[ j],
                                                      &pplTmp->aulTimer[ j] ) :
                          LoadPointerFromCursorFile( szSearchMask,
                                                     &pplTmp->hptr[ j],
                                                     &pplTmp->iconinfo[ j],
                                                     &pplTmp->aulTimer[ j] );
                if( fLoaded ) {
                  pplTmp->ulPtrCount += 1;
                  fPointerFound      = TRUE;
                } else {
                  fLoadError = TRUE;
                }
              } else {
                ulFileType         =  ( ulFileType == RESFILETYPE_POINTER ) ?
                                     RESFILETYPE_POINTERSET :
                                     RESFILETYPE_CURSORSET;
                fSetComplete  = TRUE;
              }
            }       // if (containsnumeration( pplTmp->szAnimationName))
            else {
              // load file
              fLoaded = ( ulFileType == RESFILETYPE_POINTER ) ?
                        LoadPointerFromPointerFile( pplTmp->szAnimationName,
                                                    &pplTmp->hptr[ 0],
                                                    &pplTmp->iconinfo[ 0],
                                                    &pplTmp->aulTimer[ 0] ) :
                        LoadPointerFromCursorFile( pplTmp->szAnimationName,
                                                   &pplTmp->hptr[ 0],
                                                   &pplTmp->iconinfo[ 0],
                                                   &pplTmp->aulTimer[ 0] );
              if( fLoaded ) {
                pplTmp->ulPtrCount = 1;
                fPointerFound      = TRUE;
              } else {
                fLoadError = TRUE;
              }
              fSetComplete  = TRUE;
            }
            break;

          case RESFILETYPE_POINTERSET:
          case RESFILETYPE_CURSORSET:
          case RESFILETYPE_ANIMOUSESET:
          case RESFILETYPE_WINANIMATIONSET:
            // Pointer- oder Cursor Datei-Maske ermitteln
            rc = ERROR_FILE_NOT_FOUND;
            for( k = 0; (( k < ulFormatCount ) && ( rc != NO_ERROR )); k++ )
            {
              // ist zuvor ein Set gefunden worden ?
              // dann nur noch set-Masken suchen !
              if( fIsSet ) {
                if( !afIsSet[ k] ) {
                  continue;
                }
              }

              // Format bestimmen
              pszUsedSearchFormat = apszSearchFormat[ k];
              sprintf( szSearchMask,
                       pszUsedSearchFormat,
                       szFullName,
                       QueryPointerName( i ),
                       j );

              // Pointer-Datei im Verzeichnis suchen
              rc = ( FileExist( szSearchMask, FALSE ) ? NO_ERROR : ERROR_NO_MORE_FILES );

              // Resource typ bestimmen
              if( rc == NO_ERROR ) {
                // typ herausfinden
                ulFileType = RESFILETYPE_POINTERSET;

                // Ist es ein Cursor ?
                fIsCursor = afIsCursor[ k];
                if( fIsCursor ) {
                  ulFileType = RESFILETYPE_CURSORSET;
                }

                // Ist es eine Win Animation ?
                fIsWinAnim = afIsWinAnim[ k];
                if( fIsWinAnim ) {
                  ulFileType = RESFILETYPE_WINANIMATIONSET;
                }

                // Ist es eine AniMouse Animation ?
                fIsAniMouse = afIsAniMouse[ k];
                if( fIsAniMouse ) {
                  ulFileType = RESFILETYPE_ANIMOUSESET;
                }

                // ist es ein Set ? Dann Suche begrenzen
                fIsSet = afIsSet[ k];
              }

              // ist alles geladen ?
              fSetComplete = afSetComplete[ k];

              // hole statische Pointer
              if(( fIsSet ) &&  ( !fIsAniMouse ) && ( pplTmp->hptrStatic == NULLHANDLE )) {
                pszUsedSearchFormat = apszSearchFormat[ k + 2];
                sprintf( szStaticName,
                         pszUsedSearchFormat,
                         szFullName,
                         QueryPointerName( i ));

                if( fIsCursor ) {
                  LoadPointerFromCursorFile( szStaticName,
                                             &pplTmp->hptrStatic,
                                             &pplTmp->iconinfoStatic,
                                             NULL );
                } else {
                  LoadPointerFromPointerFile( szStaticName,
                                              &pplTmp->hptrStatic,
                                              &pplTmp->iconinfoStatic,
                                              NULL );
                }
              }
            }       // end for (k < ulFormatCount)


            if( rc == NO_ERROR ) {
              // Name dieser Datei merken
              if( pplTmp->ulPtrCount == 0 ) {
                strcpy( pplTmp->szAnimationFile, szSearchMask );
              }

              //
              // Pointer in Liste einsetzen
              //
              if( fIsCursor ) {
                fLoadError += ( !LoadPointerFromCursorFile( szSearchMask,
                                                            &pplTmp->hptr[ j],
                                                            &pplTmp->iconinfo[ j],
                                                            &pplTmp->aulTimer[ j] ));
              } else if( fIsWinAnim ) {
                // beim direkten Laden eines Sets vorherige Fehler verwerfen !
                pplTmp->ulPtrCount = MAX_COUNT_POINTERS;
                fLoadError  = ( !LoadPointerFromWinAnimationFile( szSearchMask,
                                                                  &pplTmp->hptr[ 0],
                                                                  &pplTmp->iconinfo[ 0],
                                                                  &pplTmp->aulTimer[ 0],
                                                                  &pplTmp->ulPtrCount ));
                fSetComplete  = TRUE;
                fPointerFound = !fLoadError;
                break;
              } else if( fIsAniMouse ) {
                // access animouse dll
                if( pplTmp->hmodResource == NULL ) {
                  rc = LoadAnimouseModule( szSearchMask, &pplTmp->hmodResource );
                  if( rc != NO_ERROR ) {
                    fLoadError = TRUE;
                    fSetComplete  = TRUE;
                    break;
                  }
                }

                // beim direkten Laden eines Sets vorherige Fehler verwerfen !
                pplTmp->ulPtrCount = MAX_COUNT_POINTERS;
                fLoadError = ( !LoadPointerFromAnimouseFile( pplTmp->hmodResource,
                                                             i,
                                                             &pplTmp->hptr[ j],
                                                             &pplTmp->iconinfo[ j],
                                                             &pplTmp->hptrStatic,
                                                             &pplTmp->iconinfoStatic,
                                                             &pplTmp->aulTimer[ 0],
                                                             &pplTmp->ulPtrCount ));
                fSetComplete  = TRUE;
                fPointerFound = !fLoadError;
                break;
              } else {
                fLoadError += ( !LoadPointerFromPointerFile( szSearchMask,
                                                             &pplTmp->hptr[ j],
                                                             &pplTmp->iconinfo[ j],
                                                             &pplTmp->aulTimer[ j] ));
              }

              // Variablen weiterzhlen
              pplTmp->ulPtrCount += 1;
              fPointerFound = TRUE;
            } else {
              // ist was gefunden worden ?
              // if ((rc != ERROR_NO_MORE_FILES) || (pplTmp->ulPtrCount == 0))
              //   fLoadError = TRUE;

              fSetComplete = TRUE;
            }
            break;    // case RESFILETYPE_POINTER RESFILETYPE_CURSOR

          case RESFILETYPE_ANIMOUSE:
            // access animouse dll
            if( pplTmp->hmodResource == NULL ) {
              rc = LoadAnimouseModule( pszName, &pplTmp->hmodResource );
              if( rc != NO_ERROR ) {
                fLoadError = TRUE;
                fSetComplete  = TRUE;
                break;
              }
            }

            pplTmp->ulPtrCount = MAX_COUNT_POINTERS;
            fLoadError = ( !LoadPointerFromAnimouseFile( pplTmp->hmodResource,
                                                         i,
                                                         &pplTmp->hptr[ j],
                                                         &pplTmp->iconinfo[ j],
                                                         &pplTmp->hptrStatic,
                                                         &pplTmp->iconinfoStatic,
                                                         &pplTmp->aulTimer[ 0],
                                                         &pplTmp->ulPtrCount ));

            if( !fLoadError ) {
              fPointerFound = (( pplTmp->hptrStatic != NULLHANDLE ) ||
                               ( pplTmp->ulPtrCount != 0 ));
            }
            fSetComplete  = TRUE;
            break;    // case RESFILETYPE_ANIMOUSE

          case RESFILETYPE_ANMFILE:
            pplTmp->ulPtrCount = MAX_COUNT_POINTERS;
            fLoadError = ( !LoadPointerFromAnimouseScriptFile( pplTmp->szAnimationName,
                                                               i,
                                                               &pplTmp->hptr[ j],
                                                               &pplTmp->iconinfo[ j],
                                                               &pplTmp->hptrStatic,
                                                               &pplTmp->iconinfoStatic,
                                                               &pplTmp->aulTimer[ 0],
                                                               &pplTmp->ulPtrCount ));

            if( !fLoadError ) {
              fPointerFound = (( pplTmp->hptrStatic != NULLHANDLE ) ||
                               ( pplTmp->ulPtrCount != 0 ));
            }
            fSetComplete  = TRUE;
            break;    // case RESFILETYPE_ANMFILE

          case RESFILETYPE_WINANIMATION:
            pplTmp->ulPtrCount = MAX_COUNT_POINTERS;
            fLoadError += ( !LoadPointerFromWinAnimationFile( pplTmp->szAnimationName,
                                                              &pplTmp->hptr[ 0],
                                                              &pplTmp->iconinfo[ 0],
                                                              &pplTmp->aulTimer[ 0],
                                                              &pplTmp->ulPtrCount ));
            fPointerFound = !fLoadError;
            fSetComplete  = TRUE;
            break;    // case RESFILETYPE_WINANIMATION
        }
      }

      // Resource-Typ sichern
      pplTmp->ulResourceType = ulFileType;

      // wenn alles ok ist, Pointerliste bernehmen
      if( fPointerFound ) {
        // falls nicht extra geladen,
        // ersten Pointer der Animation als statischen Pointer verwenden
        if( pplTmp->hptrStatic == NULLHANDLE ) {
          pplTmp->fStaticPointerCopied = TRUE;
          pplTmp->hptrStatic = pplTmp->hptr[ 0];
          memcpy( &pplTmp->iconinfoStatic, &pplTmp->iconinfo[ 0], sizeof( ICONINFO ));
        }

        // falls keine animierten Zeiger vorhanden sind,
        // statischen Pointer als ersten Pointer der Animation verwenden
        if(( pplTmp->hptr[ 0] == NULLHANDLE ) && ( pplTmp->ulPtrCount == 0 )) {
          pplTmp->fStaticPointerCopied = TRUE;
          pplTmp->hptr[ 0] = pplTmp->hptrStatic;
          memcpy( &pplTmp->iconinfo[ 0], &pplTmp->iconinfoStatic, sizeof( ICONINFO ));
          pplTmp->ulPtrCount = 1;      // ### ist das notwendig ?
        }

        // ggfs. in Graustufen umrechnen
        if( getBlackWhite()) {
          GrayScalePointer( pplTmp->iconinfoStatic.pIconData, 2, 50 );
          WinDestroyPointer( pplTmp->hptrStatic );
          pplTmp->hptrStatic = CreatePtrFromIconInfo( &pplTmp->iconinfoStatic );

          for( l = 0; l < pplTmp->ulPtrCount; l++ )
          {
            GrayScalePointer( pplTmp->iconinfo[ l].pIconData, 2, 50 );
            WinDestroyPointer( pplTmp->hptr[ l] );
            pplTmp->hptr[ l] = CreatePtrFromIconInfo( &pplTmp->iconinfo[ l] );
          }      /* endfor */
        }

        // ggfs. Animation ausstellen
        if( pplndx->fAnimate ) {
          EnableAnimation( pplndx, FALSE );
        }

        // Beschreibung laden: Gre holen
        if( pplTmp->szAnimationFile[ 0] != 0 ) {
          // Beschreibung laden: Gre holen
          QueryFileDetails( pplTmp->szAnimationFile,  pplTmp->ulResourceType,
                            FI_SOURCEINFO, NULL, 0, &ulSourceInfoLen );

          if( ulSourceInfoLen ) {
            pvSourceInfo = malloc( ulSourceInfoLen );
            if( pvSourceInfo ) {
              // Beschreibung laden: Daten holen und speichern
              QueryFileDetails( pplTmp->szAnimationFile,  pplTmp->ulResourceType, FI_SOURCEINFO, pvSourceInfo, ulSourceInfoLen, &ulSourceInfoLen );
              psi = (PSOURCEINFO)pvSourceInfo;
              if( psi->pszInfoName ) {
                pplTmp->pszInfoName   = strdup( psi->pszInfoName );
              }
              if( psi->pszInfoArtist ) {
                pplTmp->pszInfoArtist = strdup( psi->pszInfoArtist );
              }
              free( pvSourceInfo );
            }
          }
        }

        // aktuelle Liste lschen
        CopyPointerlist( NULL, pplndx, FALSE );

        // temporre Liste kopieren
        CopyPointerlist( pplndx, pplTmp, TRUE );

        // bei mehr als einem Poiner: Animation anstellen
        if( getActivateOnLoad()) {
          if(( fEnableIfAnimation ) && ( pplndx->ulPtrCount > 1 )) {
            EnableAnimation( pplndx, TRUE );
          }
        }
      }
    }

    // Fehler aufgetreten ?
    if( !fLoadError ) {
      fSuccess = TRUE;
    }
  } while( FALSE );

  return fSuccess;
}

/*Ŀ
 * Name      : CopyPointerlist                                            
 * Kommentar : POINTERLIST Struktur kopieren oder "lschen": Ressource    
 *             freigeben.                                                 
 * Autor     : C.Langanke                                                 
 * Datum     : 20.07.1996                                                 
 * nderung  : 20.07.1996                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : PPOINTERLIST - Zeiger auf Ziel-Struktur                    
 *             PPOINTERLIST - Zeiger auf Quell-Struktur oder NULL         
 *             BOOL         - Flag zum Kopieren der Resourcen             
 * Aufgaben  : - Speicher freigeben und Struktur kopieren                 
 * Rckgabe  : -                                                          
 *
 */

BOOL
CopyPointerlist( PPOINTERLIST pplTarget, PPOINTERLIST pplSource, BOOL fCopyResources )
{
  BOOL      fSuccess = FALSE;
  ULONG     i;
  PICONINFO pIconInfo;
  ULONG     ulCopyLen;

  do
  {
    // Parameter prfen
    if( pplSource == NULL ) {
      break;
    }

    // Zeiger drfen nicht gleich sein
    // sonst wird beim Kopieren gelscht
    if( pplSource == pplTarget ) {
      break;
    }

    // wenn nicht kopiert werden soll, Resourcen lschen
    if( pplTarget == NULL ) {
      // statischen Zeiger lschen
      pplSource->fStaticPointerCopied = FALSE;
      if( pplSource->hptrStatic ) {
        WinDestroyPointer( pplSource->hptrStatic );
      }

      // info freigeben
      if( pplSource->pszInfoName ) {
        free( pplSource->pszInfoName );
      }

      if( pplSource->pszInfoArtist ) {
        free( pplSource->pszInfoArtist );
      }

      // ggfs. AniMouse DLL freigeben
      if( pplSource->hmodResource ) {
        DosFreeModule( pplSource->hmodResource );
      }

      // Animation lschen
      for( i = 0; i < pplSource->ulPtrCount; i++ )
      {
        // Pointer freigeben
        if( pplSource->hptr[ i] != 0 ) {
          WinDestroyPointer( pplSource->hptr[ i] );
        }

        // ICONINFO freigeben
        pIconInfo = &pplSource->iconinfo[ i];
        if( pIconInfo->pszFileName ) {
          free( pIconInfo->pszFileName );
        }
        if( pIconInfo->hmod ) {
          DosFreeResource( pIconInfo->pIconData );
        }
        if( pIconInfo->pIconData ) {
          free( pIconInfo->pIconData );
        }

        // timeout lschen
        pplSource->aulTimer[ i] = 0;
      }

      // komplette Struktur zurcksetzen
      memset( pplSource, 0, sizeof( POINTERLIST ));
    } else {
      // Struktur kopieren
      memset( pplTarget, 0, sizeof( POINTERLIST ));

      ulCopyLen = ( fCopyResources ) ?
                  sizeof( POINTERLIST ) : FIELDOFFSET( POINTERLIST, fReplaceActive );
      memcpy( pplTarget, pplSource, ulCopyLen );

      // info kopieren
      if( pplSource->pszInfoName ) {
        pplTarget->pszInfoName   = strdup( pplSource->pszInfoName );
      }
      if( pplSource->pszInfoArtist ) {
        pplTarget->pszInfoArtist = strdup( pplSource->pszInfoArtist );
      }

      // alles ok
      fSuccess = TRUE;
    }
  } while( FALSE );

  return fSuccess;
}


/*Ŀ
 * Name      : IsPointerlistInitialized                                   
 * Kommentar : POINTERLIST Struktur Initialisierung bestimmen             
 * Autor     : C.Langanke                                                 
 * Datum     : 24.07.1996                                                 
 * nderung  : 24.07.1996                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : -                                                          
 * Aufgaben  : - Initialisierung prfen                                   
 * Rckgabe  : BOO -  Flag, ob initialisiert                              
 *
 */

BOOL
IsPointerlistInitialized( VOID ) {
  return fPointerlistInitialized;
}

/*Ŀ
 * Name      : InitializePointerlist                                      
 * Kommentar : POINTERLIST Struktur initialisieren                        
 * Autor     : C.Langanke                                                 
 * Datum     : 24.07.1996                                                 
 * nderung  : 24.07.1996                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : -                                                          
 * Aufgaben  : - Initialisieren                                           
 * Rckgabe  : APIRET - OS/2 Fehlercode                                   
 *
 */

APIRET
InitializePointerlist( HAB hab, HMODULE hmodResource )
{
  APIRET rc = NO_ERROR;
  ULONG  i;
  BOOL   fLoadError = FALSE;
  PPOINTERLIST ppl;

  do
  {
    // ist bereits initialisiert ?
    if( IsPointerlistInitialized()) {
      rc = ERROR_ACCESS_DENIED;
      break;
    }

    // Flag setzen
    fPointerlistInitialized = TRUE;

    // Resourcen fr Container laden
    InitStringResources( hab, hmodResource );

    // Zugriffs-Semaphore erstellen
    rc = DosCreateMutexSem( NULL,
                            &hmtxDataAccess,
                            0,
                            TRUE );         // Thread erst mal blokieren
    if( rc != NO_ERROR ) {
      break;
    }

    if( !DEBUGSETTING( SET_INACTIVEANIMATETHREAD )) {
      // Startup-Semaphore erstellen
      rc = DosCreateEventSem( NULL,
                              &hevStartup,
                              0,
                              FALSE );           // Direkt erst mal blokieren
      if( rc != NO_ERROR ) {
        break;
      }

      // Animations-Thread starten
      tidAnimation = _beginthread( AnimationThread,
                                   NULL,
                                   16384,
                                   &hevStartup );
      rc = ( tidAnimation == -1 );
      if( rc != NO_ERROR ) {
        break;
      }


      // startup abwarten
      rc = DosWaitEventSem( hevStartup, 20000L );
      DosCloseEventSem( hevStartup );
      if( rc != NO_ERROR ) {
        BEEP4;  // waiting for startup sem failed on InitializePointerlist
        break;
      }
    }

    // alle POINTERLISTen initialisieren
    for( i = 0, ppl = QueryPointerlist( i );
         i < NUM_OF_SYSCURSORS;
         i++, ppl++ )
    {
      // Variablen setzen
      memset( ppl, 0, sizeof( POINTERLIST ));
      ppl->ulPtrId          = QueryPointerSysId( i );
    }   // end for (i < NUM_OF_SYSCURSORS)

    // ist ein Fehler aufgetreten ?
    if( fLoadError ) {
      rc = ERROR_GEN_FAILURE;
    }

    // Zugriff freigeben
    rc = RELEASE_DATA_ACCESS();
  } while( FALSE );

  return rc;
}

/*Ŀ
 * Name      : DeinitializePointerlist                                    
 * Kommentar : POINTERLIST Struktur deinitialisieren                      
 * Autor     : C.Langanke                                                 
 * Datum     : 24.07.1996                                                 
 * nderung  : 24.07.1996                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : -                                                          
 * Aufgaben  : - deinitialisieren                                         
 * Rckgabe  : APIRET - OS/2 Fehlercode                                   
 *
 */

APIRET
DeinitializePointerlist( VOID )

{
  APIRET rc = NO_ERROR;
  ULONG  i;
  PPOINTERLIST ppl;

  FUNCENTER();

  do
  {
    // ist noch nicht initialisiert ?
    if( !IsPointerlistInitialized()) {
      rc = ERROR_ACCESS_DENIED;
      break;
    }

    // get data access
    REQUEST_DATA_ACCESS();

    // Flag setzen
    fPointerlistInitialized = FALSE;

    // alle POINTERLISTen deinitialisieren
    for( i = 0, ppl = QueryPointerlist( i );
         i < NUM_OF_SYSCURSORS;
         i++, ppl++ )
    {
      // Animation ausschalten
      EnableAnimation( ppl, FALSE );

      //    // wird bereits in EnableAnimation() durchgefhrt
      //    // Animation zurcksetzen
      //    ResetAnimation( ppl, (i == 0));

      // Resourcen lschen
      CopyPointerlist( NULL, ppl, FALSE );
    }

    // end animation thread, if not yet done
    WinPostMsg( QueryAnimationHwnd(), WM_CLOSE, 0L, 0L );
  } while( FALSE );

  return rc;
}

/*Ŀ
 * Name      : QueryPointerlist                                           
 * Kommentar : Zeiger auf Pointerliste(n) zurckgeben                     
 * Autor     : C.Langanke                                                 
 * Datum     : 24.07.1996                                                 
 * nderung  : 24.07.1996                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : -                                                          
 * Aufgaben  : - deinitialisieren                                         
 * Rckgabe  : HMTX - Handle der Semaphore                                
 *
 */

PPOINTERLIST
QueryPointerlist( ULONG ulIndex )
{
  if( ulIndex > NUM_OF_SYSCURSORS ) {
    return 0;
  } else {
    return &appl[ ulIndex];
  }
}


/*Ŀ
 * Name      : EnableAnimation                                            
 * Kommentar : (de)aktiviert den Animationsthread fr einen Pointer       
 * Autor     : C.Langanke                                                 
 * Datum     : 20.07.1996                                                 
 * nderung  : 20.07.1996                                                 
 * aufgerufen: Window Procedure                                           
 * ruft auf  : -                                                          
 * Eingabe   : PPOINTERLIST - Zeiger auf Pointerliste                     
 * Aufgaben  : - Animation ein/ausstellen                                 
 * Rckgabe  : BOOL - Flag, ob erfolgreich                                
 *
 */

BOOL
EnableAnimation( PPOINTERLIST ppl, BOOL fEnable )

{
  BOOL   fSuccess = FALSE;
  ULONG  ulTimerId;
  HWND   hwndAnimation      = QueryAnimationHwnd();
  HAB    habAnimationThread = QueryAnimationHab();

  do {
    // Parameter prfen
    if( ppl == NULL ) {
      break;
    }

    // ist Animation mglich ?
    if( ppl->ulPtrCount <= 1 ) {
      break;
    }

    // timer Id ermitteln
    ulTimerId =  ( ppl - ( QueryPointerlist( 0 ))) + 1;
    if( ulTimerId > NUM_OF_SYSCURSORS ) {
      break;
    }

    // prfen, ob Thread da ist
    if( tidAnimation == 0 ) {
      // kein Fehler, wenn deaktiviert werden soll
      fSuccess = !fEnable;
      break;
    }

    // prfen, ob Object Fenster noch da ist
    if(( hwndAnimation == NULLHANDLE ) || ( !WinIsWindow( habAnimationThread, hwndAnimation ))) {
      // kein Fehler, wenn deaktiviert werden soll
      fSuccess = !fEnable;
      break;
    }

    // Timer starten / stoppen
    if( DEBUGSETTING( SET_ANIMCMDASYNC )) {
      fSuccess = (BOOL)WinPostMsg( hwndAnimation, WM_USER_ENABLEANIMATION, MPFROMP( ppl ), MPFROMSHORT( fEnable ));
      if( !fSuccess ) {
        DosBeep( 1000, 50 );
        DosBeep( 700, 50 );
      }
    } else {
      fSuccess = (BOOL)WinSendMsg( hwndAnimation, WM_USER_ENABLEANIMATION, MPFROMP( ppl ), MPFROMSHORT( fEnable ));
    }
  } while( FALSE );

  return fSuccess;
}

/*Ŀ
 * Name      : SetNextAnimatedPointer                                     
 * Kommentar : prft, ob die Animation ein. bzw auszuschalten ist und     
 *             ldt ggfs. nchsten Pointer aus dem Set                    
 *             luft bereits mit exclusiv-Rechten auf die Datenstrukturen!
 * Autor     : C.Langanke                                                 
 * Datum     : 07.06.1996                                                 
 * nderung  : 07.06.1996                                                 
 * aufgerufen: diverse                                                    
 * ruft auf  : -                                                          
 * Eingabe   : HAB          - handle anchorblock                          
 *           : PPOINTERLIST - Pointerliste                                
 * Aufgaben  : - Animation aktivieren / deaktivieren                      
 * Rckgabe  : BOOL - Flag: Erfolgsflag                                   
 *
 */

BOOL
SetNextAnimatedPointer( HAB hab, PPOINTERLIST ppl, BOOL fForceSet )

{
  HPOINTER  hptrCurrent;
  BOOL      fSuccess = FALSE;
  PICONINFO piconinfo;
  BOOL      fPointerActive = FALSE;

  do
  {
    // ist der Mauszeiger sichtbar ?
    if( !WinIsPointerVisible( HWND_DESKTOP )) {
      fSuccess = TRUE;
      break;
    }

    // Parameter prfen
    if( ppl == NULL ) {
      break;
    }

    // wenn die Ptr-Liste leer ist oder
    // nur einen Pointer hat, abbrechen
    if( ppl->ulPtrCount <= 1 ) {
      break;
    }

    // prfen, ob dieser Pointer der aktuelle ist
    fPointerActive = IsPointerActive( ppl->ulPtrId,  &hptrCurrent );

    // Animation aktivieren bzw. deaktivieren
    if( !ppl->fReplaceActive ) {
      // falls der pointer des systems aktiv ist, Animation aktivieren
      if( fPointerActive ) {
        ppl->fReplaceActive = TRUE;
      }
    } else { // if (ppl->fReplaceActive)
      // prfen, ob deaktiviert werden mu

      BOOL  fNotOwnPointer = TRUE;
      ULONG i;

      // wenn es nicht der sys ptr ist
      if( !fPointerActive ) {
        // ist es ein eigener pointer ?
        for( i = 0; i < ppl->ulPtrCount; i++ )
        {
          if( hptrCurrent == ppl->hptr[ i] ) {
            fNotOwnPointer = FALSE;
            break;
          }
        }

        if( fNotOwnPointer ) {
          // wurde von jemandem anderen ein Pointer gesetzt:
          // jetzt Animation deaktivieren

          ppl->fReplaceActive = FALSE;
        }
      }
    }

    // ************************************************

    if(( ppl->fReplaceActive ) || ( fForceSet )) {
      // nchsten Pointer in der Animation auswhlen
      ppl->ulPtrIndex++;
      if( ppl->ulPtrIndex >= ppl->ulPtrCount ) {
        ppl->ulPtrIndex = 0;
      }

      if( ppl->hptr[ ppl->ulPtrIndex ] != NULL ) {
        // wait pointer data setzen
        piconinfo = &( ppl->iconinfo[ ppl->ulPtrIndex] );
        fSuccess = WinSetSysPointerData( HWND_DESKTOP, ppl->ulPtrId, piconinfo );
        // set new pointer
        WinSetPointer( HWND_DESKTOP, ppl->hptr[ ppl->ulPtrIndex ] );
      }
    } else {
      // Pointer handle ungltig
      fSuccess = FALSE;
    }
  } while( FALSE );

  return fSuccess;
}

/*Ŀ
 * Name      : QueryAnimate                                               
 * Kommentar : fragt das Animation Flag fr einen oder alle Ptr aktiv ist 
 *             Wenn eine Animation aktiv ist, wird TRUE zurckgegeben     
 * Autor     : C.Langanke                                                 
 * Datum     : 24.07.1996                                                 
 * nderung  : 24.07.1996                                                 
 * aufgerufen: Window Procedure                                           
 * ruft auf  : -                                                          
 * Eingabe   : ULONG - pointer index                                      
 *             BOOL  - query all ptrs                                     
 * Aufgaben  : - Animate abfragen                                         
 * Rckgabe  : BOOL - Flag, ob eine Animation aktiv ist                   
 *
 */

BOOL
QueryAnimate( ULONG ulPointerIndex, BOOL fQueryAll )

{
  ULONG  i;
  BOOL   fAnimate = FALSE;
  PPOINTERLIST ppl;

  ULONG  ulFirstPtr, ulLastPtr;

  do {
    // range fr Verarbeitung festlegen
    ulFirstPtr = ulPointerIndex;

    if( fQueryAll ) {
      ulLastPtr  = NUM_OF_SYSCURSORS;
    } else {
      ulLastPtr  = ulPointerIndex + 1;
    }

    ppl = QueryPointerlist( ulPointerIndex );

    for( i = ulFirstPtr; i < ulLastPtr; i++, ppl++ )
    {
      // abfragen
      if( ppl->fAnimate ) {
        fAnimate = TRUE;
      }
    }
  } while( FALSE );

  return fAnimate;
}

/*Ŀ
 * Name      : QueryDemo                                                  
 * Kommentar : fragt das Demo Flag ab                                     
 * Autor     : C.Langanke                                                 
 * Datum     : 24.07.1996                                                 
 * nderung  : 24.07.1996                                                 
 * aufgerufen: Window Procedure                                           
 * ruft auf  : -                                                          
 * Eingabe   : -                                                          
 * Aufgaben  : - Flag zurckgeben                                         
 * Rckgabe  : BOOL - Flag, ob Demo aktiv ist                             
 *
 */

BOOL QueryDemo() {
  return fDemoActive;
}

/*Ŀ
 * Name      : ToggleDemo                                                 
 * Kommentar : (de-)aktiviert Demo                                        
 * Autor     : C.Langanke                                                 
 * Datum     : 24.07.1996                                                 
 * nderung  : 24.07.1996                                                 
 * aufgerufen: Window Procedure                                           
 * ruft auf  : -                                                          
 * Eingabe   : BOOL        - Enable Flag                                  
 *             PRECORDCORE - Zeiger auf Cnr-Daten                         
 *             PBOOL       - Zeiger auf ForceFlag                         
 * Aufgaben  : - Demo (de-)aktivieren                                     
 * Rckgabe  : BOOL - Flag, ob erfolgreich                                
 *
 */

BOOL
ToggleDemo( HWND hwnd, PRECORDCORE pcnrrec, BOOL fRefresh, PBOOL pfEnable )
{
  BOOL   fSuccess = FALSE;
  ULONG  ulTimerId;

  do {
    // Demo aus-/anschalten
    // ggfs. Flag overrulen
    if( pfEnable ) {
      fDemoActive = *pfEnable;
    } else {
      fDemoActive = !fDemoActive;
    }

    // ggfs. kein Update
    if( hwnd == NULLHANDLE ) {
      break;
    }

    // Timer fr Demo aktivieren/deaktivieren
    switch( fDemoActive )
    {
      case TRUE:
        ulTimerId = WinStartTimer( WinQueryAnchorBlock( hwnd ),
                                   hwnd,
                                   DEMO_TIMER_ID,
                                   getDefaultTimeout());
        fSuccess  = ( ulTimerId != 0 );
        break;

      case FALSE:
        // timer stoppen
        fSuccess = WinStopTimer( WinQueryAnchorBlock( hwnd ), hwnd, DEMO_TIMER_ID );

        // Pointer zurcksetzen
        if( pcnrrec ) {
          ProceedWithDemo( hwnd, pcnrrec, fRefresh, TRUE );
        }
        break;
    }
  } while( FALSE );

  return fSuccess;
}

/*Ŀ
 * Name      : ProceedWithDemo                                            
 * Kommentar : fhrt einen Schritt in der Demo-Animation durch            
 * Autor     : C.Langanke                                                 
 * Datum     : 24.07.1996                                                 
 * nderung  : 24.07.1996                                                 
 * aufgerufen: Window Procedure                                           
 * ruft auf  : -                                                          
 * Eingabe   : HWND        - window handle                                
 *             PRECORDCORE - Zeiger auf Cnr-Daten                         
 *             BOOL        - Flag, ob nur zurckgesetzt werden soll       
 * Aufgaben  : - Demo einen Schritt weiter durchfhren                    
 * Rckgabe  : BOOL - Flag, ob erfolgreich                                
 *
 */

BOOL
ProceedWithDemo( HWND hwnd, PRECORDCORE pcnrrec, BOOL fRefresh, BOOL fReset )
{
  BOOL   fSuccess = FALSE;
  ULONG  i;
  PPOINTERLIST ppl;
  ULONG  ulViewStyle;
  PRECORDCORE rec;

  do {
    // nicht im Default View ausfhren
    QueryContainerView( hwnd, &ulViewStyle );
    if( ulViewStyle & CV_DETAIL ) {
      break;
    }

    // Pointer auswhlen
    ppl = QueryPointerlist( 0 );

    for( i = 0, rec = pcnrrec; i < NUM_OF_SYSCURSORS; i++, ppl++, rec = rec->preccNextRecord )
    {
      BOOL fChanged = FALSE;

      if( fReset ) {
        ppl->ulPtrIndexCnr = 0;
        fRefresh = TRUE;
        fChanged = TRUE;
      } else {
        if( ppl->fAnimate ) {
          // nchsten Zeiger anwhlen
          ppl->ulPtrIndexCnr++;
          if( ppl->ulPtrIndexCnr >= ppl->ulPtrCount ) {
            ppl->ulPtrIndexCnr = 0;
          }
          fChanged = TRUE;
        }
      }
      // Refresh durchfhren
      if( fRefresh && fChanged ) {
        RefreshCnrItem( hwnd, rec, pcnrrec, FALSE );
      }
    }
  } while( FALSE );

  return fSuccess;
}

/*Ŀ
 * Name      : IsPointerActive                                            
 * Kommentar : Prft, ob der Pointer mit der angegebenen ID aktiv ist     
 * Autor     : C.Langanke                                                 
 * Datum     : 24.07.1996                                                 
 * nderung  : 24.07.1996                                                 
 * aufgerufen: Window Procedure                                           
 * ruft auf  : -                                                          
 * Eingabe   : ULONG - Ptr Id                                             
 *             PHPTR - Zeiger auf Handle Variable des aktiven Zeigers     
 * Aufgaben  : - prfen, ob Zeiger aktiv ist                              
 * Rckgabe  : BOOL - Flag, ob aktiv                                      
 *
 */
BOOL
IsPointerActive( ULONG ulPtrId, HPOINTER* phptrCurrent )
{
  HPOINTER hptrSys;
  HPOINTER hptrCurrent;

  // check pointer
  // CHECK ME:
  // DosEnterCritSec();
  hptrSys = WinQuerySysPointer( HWND_DESKTOP, ulPtrId, FALSE );
  hptrCurrent = WinQueryPointer( HWND_DESKTOP );
  // DosExitCritSec();

  if( phptrCurrent != NULL ) {
    *phptrCurrent = hptrCurrent;
  }
  return hptrSys == hptrCurrent;
}

