/************************************************************************
** MODULE INFORMATION **
************************
** FILE NAME:          statc.c
** SYSTEM NAME:        beholder
** MODULE NAME:        stat
** ORIGINAL AUTHOR(S): M.F.B. de Greeve
** VERSION NUMBER:     1.0
** CREATION DATE:      1992/7/13
** DESCRIPTION:        etherStat group: collector
*************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <dnpap.h>
#include <message.h>
#include <mac.h>
#include <timer.h>
#include <protocol.h>

#include "state.h"
#include "statc.h"


static VOID StatTimerCallback (TIMER_DESCR *timer, ULONG now, ETHER_STATS *etherStats);
static VOID StatMacCallback(MAC_COLL *collector, MAC_FRAME *frame);



/*****************************************************************
** NAME:        StatCInit
** SYNOPSIS:    BOOLEAN StatCInit (ETHER_STATS *etherStats)
** PARAMETERS:  pointer to data etherStats entry
** DESCRIPTION: initializes collector.
** REMARKS:     called when status is CREATE_REQUEST: see statm.c
** RETURNS:     TRUE: everything OK
**              FALSE: timer or net not initialized
*******************************************************************/

BOOLEAN StatCInit (ETHER_STATS *etherStats)
{
    LONG source[] = {1,3,6,1,2,1,2,2,1,1,1};
    
    etherStats->SourceLen = sizeof(source)/sizeof(long);
    memcpy (etherStats->Source, source, sizeof(source));

    if ((etherStats->Iface =
        MacIfaceGet((WORD) etherStats->Source[etherStats->SourceLen-1])) == NULL)
    {
        DnpapMessage(DMC_ERROR, STAT_NETINIT, "etherStat: network init");
        return (FALSE);
    }
    return (TRUE);
}

/*****************************************************************
** NAME:        StatCStart
** SYNOPSIS:    BOOLEAN StatCStart (ETHER_STATS *etherStats)
** PARAMETERS:  pointer to data etherStats entry
** DESCRIPTION: starts collector.
** REMARKS:     called when status is VALID: see statm.c
** RETURNS:     TRUE: everything OK
**              FALSE: timer or net not registered
*******************************************************************/

BOOLEAN StatCStart (ETHER_STATS *etherStats)
{
    MAC_STAT netStat;
    
    if ((etherStats->Timer = TimerRegister
        (StatTimerCallback, etherStats, 1000L, TIMER_FOREVER, TIMER_TYPE_SKIP)) == NULL)
    {
        DnpapMessage(DMC_ERROR, STAT_TIMER, "etherStat: timer init");
        return (FALSE);
    }
    
    etherStats->Coll.Rcve       = StatMacCallback;
    etherStats->Coll.specific   = etherStats;
    if (!MacCollRegister(&(etherStats->Coll)))
    {
        DnpapMessage(DMC_ERROR, STAT_NETINIT, "etherStat: network init");
        TimerRemove(etherStats->Timer);
        return (FALSE);
    }

    MacStatistics(etherStats->Iface, &netStat);

    etherStats->OldNetStat[0] = netStat.LostPkts + netStat.DiscardedPkts;
    etherStats->OldNetStat[1] = netStat.BroadcastPkts; 
    etherStats->OldNetStat[2] = netStat.MulticastPkts;
    etherStats->OldNetStat[3] = netStat.CRCAlignErrors;
    etherStats->OldNetStat[4] = netStat.UndersizePkts; 
    etherStats->OldNetStat[5] = netStat.OversizePkts;  
    etherStats->OldNetStat[6] = netStat.Fragments;     
    etherStats->OldNetStat[7] = netStat.Jabbers;       
    etherStats->OldNetStat[8] = netStat.Collisions;
    etherStats->Pkts64Octets = 0; 
    etherStats->Pkts65to127Octets = 0; 
    etherStats->Pkts128to255Octets = 0; 
    etherStats->Pkts256to511Octets = 0;
    etherStats->Pkts512to1023Octets = 0;
    etherStats->Pkts1024to1518Octets = 0;

    return(TRUE);
}

/*****************************************************************
** NAME:        StatCStop 
** SYNOPSIS:    BOOLEAN StatCStop (ETHER_STATS *etherStats)
** PARAMETERS:  pointer to etherStats entry
** DESCRIPTION: stops collector.
** REMARKS:     called when status is INVALID: see statm.c
** RETURNS:     TRUE: timer & net registration removed 
*******************************************************************/

BOOLEAN StatCStop (ETHER_STATS *etherStats)
{
    MacCollRemove(&(etherStats->Coll));
    TimerRemove(etherStats->Timer);
    return(TRUE);
}

/*****************************************************************
** NAME:        StatTimerCallback
** SYNOPSIS:    VOID StatTimerCallback (TIMER_DESCR *timer,
**                  ULONG now, ETHER_STATS *etherStats)
** PARAMETERS:  see Timer-manual
** DESCRIPTION: periodically requests network statistics
** REMARKS:     none
** RETURNS:     VOID
*******************************************************************/

static VOID StatTimerCallback (TIMER_DESCR *timer, ULONG now, ETHER_STATS *etherStats)
{
    MAC_STAT netStat;
    MacStatistics(etherStats->Iface, &netStat);
    etherStats->DropEvents = netStat.LostPkts + netStat.DiscardedPkts - etherStats->OldNetStat[0];
    etherStats->BroadcastPkts = netStat.BroadcastPkts - etherStats->OldNetStat[1]; 
    etherStats->MulticastPkts = netStat.MulticastPkts - etherStats->OldNetStat[2];
    etherStats->CRCAlignErrors = netStat.CRCAlignErrors - etherStats->OldNetStat[3];
    etherStats->UndersizePkts = netStat.UndersizePkts - etherStats->OldNetStat[4]; 
    etherStats->OversizePkts = netStat.OversizePkts - etherStats->OldNetStat[5];  
    etherStats->Fragments = netStat.Fragments - etherStats->OldNetStat[6];     
    etherStats->Jabbers = netStat.Jabbers - etherStats->OldNetStat[7];       
    etherStats->Collisions = netStat.Collisions - etherStats->OldNetStat[8];
}

/*****************************************************************
** NAME:        StatMacCallback
** PARAMETERS:  see Net-manual
** DESCRIPTION: analyses packets received from net
** REMARKS:     none
** RETURNS:     VOID
*******************************************************************/

static VOID StatMacCallback(MAC_COLL *collector, MAC_FRAME *frame)
{
    PROT_OBJ Interface = {1, {1,2}};
    PROT_OBJ Size = {1, {1,4}};
    PROT_PKT Pkt;
    LWORD size;
    ETHER_STATS *etherStats;

    etherStats = collector->specific;

    if (ProtFrame(&Pkt, frame) == FALSE)
        return;
    if (ProtGetField(&Pkt,&Interface) == TRUE &&
        Interface.Syntax.LngInt == etherStats->Source[etherStats->SourceLen-1])
    {
        etherStats->Pkts++;
        if (ProtGetField(&Pkt,&Size) == TRUE)
        {
            size = Size.Syntax.LngUns + 4L;
            etherStats->Octets += size;
            if (size == 64)
                etherStats->Pkts64Octets++;
            if (size >= 65)
            {
                if (size <= 127)
                    etherStats->Pkts65to127Octets++;
                else if (size <= 255)
                    etherStats->Pkts128to255Octets++; 
                else if (size <= 511)
                    etherStats->Pkts256to511Octets++;
                else if (size <= 1023)
                    etherStats->Pkts512to1023Octets++;
                else if (size <= 1518)
                    etherStats->Pkts1024to1518Octets++;
            }
        }
    }
    ProtFree (&Pkt);
    return;
}                       
