/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       high.c
**     SYSTEM   NAME:       Config
**     ORIGINAL AUTHOR(S):  Dirk Wisse
**     VERSION  NUMBER:     0.1
**     CREATION DATE:       1992/7/28
**
** DESCRIPTION:             Configuration storage/retreival.
**                          High level interface.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#if ! defined(PRD)
static char _config_hdr[] =
"$Header$";
#endif
#include <stdlib.h>
#include <stdio.h>
#include <memory.h>
#include <string.h>
#include <dnpap.h>

#include "config.h"



/* static prototypes of help functions */

static BOOLEAN       Base(BYTE *name, BYTE *base, int sizeBase);
static CONFIG_ENTRY *Create(BYTE *name);
static CONFIG_ENTRY *Find(BYTE *name, USHORT type);


/* global variables */

#ifndef CONFIGROOT
#define CONFIGROOT  DNPAPBASE
#endif

#ifdef UNIX
#define PATHSEPSTR  ":"
#endif

#ifdef OS2
#define PATHSEPSTR  ";"
#endif

static BYTE *configPath;   
static BYTE staticPath[256];

static BYTE *configRoot;   
static BYTE *configHome;   

BYTE configBase[256];       /*  not static so that utilities like cfg2ini & ini2cfg can access it  */


/* interface functions */

/**************************************************************
** NAME:        ConfigInit     
** SYNOPSIS:    BOOLEAN ConfigInit(int argc, char **argv)
** DESCRIPTION: Initializes the module.
**              Gets the HOME environment variable and
**              the base name of the executable. These
**              are used in ConfigLoad to load the
**              standard configuration files.
** RETURNS:     BOOLEAN success.
** SEE ALSO:    ConfigLoad
**************************************************************/
BOOLEAN ConfigInit(int argc, char **argv)
{
    if ((configPath = getenv("CONFIGPATH")) == NULL)    
    {
        configPath = staticPath;
        configPath[0] = '\0';
                                                
        strcpy(configPath, ".");
        
        if ((configHome = getenv("CONFIGHOME")) == NULL)
            configHome = getenv("HOME");
            
        if ((configRoot = getenv("CONFIGROOT")) == NULL)
            configRoot = CONFIGROOT;
            
        if (configHome != NULL)            
        {
            strcat(configPath, PATHSEPSTR);
            strcat(configPath, configHome);
        }
        if (configRoot != NULL)            
        {
            strcat(configPath, PATHSEPSTR);
            strcat(configPath, configRoot);
        }
    }
        
    Base(argv[0], configBase, sizeof(configBase));
    
    return TRUE;
}


/**************************************************************
** NAME:        ConfigLoad       
** SYNOPSIS:    BOOLEAN ConfigLoad(void)
** DESCRIPTION: Loads the standard config files.
**              \global.ini
**              \<base>.ini
**              <home>\global.ini
**              <home>\<base>.ini
**              .\global.ini
**              .\<base>.ini
** RETURNS:     BOOLEAN success.
** SEE ALSO:    ConfigInit, ConfigSave
**************************************************************/
BOOLEAN ConfigLoad(void)
{
    BYTE name[256];             
    BYTE path[256];
    CHAR *pp;    
    USHORT level = 0;
                   
    strcpy(path, configPath);   

    pp = strtok(path, PATHSEPSTR);
    while (pp != NULL)
    {
        sprintf(name, "%s/%s.ini", pp, configBase);
        if (ConfigRead(name, level) == TRUE)
            level++;
        sprintf(name, "%s/global.ini", pp);
        if (ConfigRead(name, level) == TRUE)
            level++;
        pp = strtok(NULL, PATHSEPSTR);
    }
    
    return TRUE;
}


/**************************************************************
** NAME:        ConfigSave   
** SYNOPSIS:    BOOLEAN ConfigSave(void)
** DESCRIPTION: Saves the changed variables to ".\<base>.ini"
** RETURNS:     BOOLEAN success.
** SEE ALSO:    ConfigLoad
**************************************************************/
BOOLEAN ConfigSave(void)
{
    BYTE name[256];

    sprintf(name, "%s.ini", configBase);
    return ConfigWrite(name, 0);
}



/* Support for the different types */


/**************************************************************
** NAME:        ConfigSetBoolean
** SYNOPSIS:    BOOLEAN ConfigSetBoolean(BYTE *name, BOOLEAN b)
** DESCRIPTION: Sets variable <name> to <b>.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigSetBoolean(BYTE *name, BOOLEAN b)
{
    CONFIG_ENTRY *e;

    e=Create(name);
    if (e==0)
        return FALSE;
    return ConfigSetValueBoolean(e,b);
}


/**************************************************************
** NAME:        ConfigGetBoolean
** SYNOPSIS:    BOOLEAN ConfigGetBoolean(BYTE *name, BOOLEAN *b)
** DESCRIPTION: Gets variable <name> in <b>.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigGetBoolean(BYTE *name, BOOLEAN *b)
{
    CONFIG_ENTRY *e;

    e=Find(name, CONFIG_TYPE_BOOLEAN);
    if (e==0)
        return FALSE;
    return ConfigGetValueBoolean(e, b);
}


/**************************************************************
** NAME:        ConfigSetShort  
** SYNOPSIS:    BOOLEAN ConfigSetShort(BYTE *name, short s)
** DESCRIPTION: Sets variable <name> to <s>.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigSetShort(BYTE *name, short s)
{
    CONFIG_ENTRY *e;

    e=Create(name);
    if (e==0)
        return FALSE;
    return ConfigSetValueShort(e,s);
}


/**************************************************************
** NAME:        ConfigGetShort  
** SYNOPSIS:    BOOLEAN ConfigGetShort(BYTE *name, short *s)
** DESCRIPTION: Gets variable <name> in <s>.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigGetShort(BYTE *name, short *s)
{
    CONFIG_ENTRY *e;

    e=Find(name, CONFIG_TYPE_SHORT);
    if (e==0)
        return FALSE;
    return ConfigGetValueShort(e, s);
}


/**************************************************************
** NAME:        ConfigSetLong  
** SYNOPSIS:    BOOLEAN ConfigSetLong(BYTE *name, LONG l)
** DESCRIPTION: Sets variable <name> to <l>.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigSetLong(BYTE *name, LONG l)
{
    CONFIG_ENTRY *e;

    e=Create(name);
    if (e==0)
        return FALSE;
    return ConfigSetValueLong(e,l);
}


/**************************************************************
** NAME:        ConfigGetLong  
** SYNOPSIS:    BOOLEAN ConfigGetLong(BYTE *name, LONG *l)
** DESCRIPTION: Gets variable <name> in <l>.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigGetLong(BYTE *name, LONG *l)
{
    CONFIG_ENTRY *e;

    e=Find(name, CONFIG_TYPE_LONG);
    if (e==0)
        return FALSE;
    return ConfigGetValueLong(e, l);
}


/**************************************************************
** NAME:        ConfigSetIPAddr
** SYNOPSIS:    BOOLEAN ConfigSetIPAddr(BYTE *name, IPADDR a)
** DESCRIPTION: Sets variable <name> to <a>.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigSetIPAddr(BYTE *name, IPADDR a)
{
    CONFIG_ENTRY *e;

    e=Create(name);
    if (e==0)
        return FALSE;
    return ConfigSetValueIPAddr(e,a);
}


/**************************************************************
** NAME:        ConfigGetIPAddr 
** SYNOPSIS:    BOOLEAN ConfigGetIPAddr(BYTE *name, IPADDR *a)
** DESCRIPTION: Gets variable <name> in <a>.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigGetIPAddr(BYTE *name, IPADDR *a)
{
    CONFIG_ENTRY *e;

    e=Find(name, CONFIG_TYPE_IPADDR);
    if (e==0)
        return FALSE;
    return ConfigGetValueIPAddr(e, a);
}


/**************************************************************
** NAME:        ConfigSetDouble  
** SYNOPSIS:    BOOLEAN ConfigSetDouble(BYTE *name, double d)
** DESCRIPTION: Sets variable <name> to <d>.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigSetDouble(BYTE *name, double d)
{
    CONFIG_ENTRY *e;

    e=Create(name);
    if (e==0)
        return FALSE;
    return ConfigSetValueDouble(e,d);
}


/**************************************************************
** NAME:        ConfigGetDouble  
** SYNOPSIS:    BOOLEAN ConfigGetDouble(BYTE *name, double *d)
** DESCRIPTION: Gets variable <name> in <d>.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigGetDouble(BYTE *name, double *d)
{
    CONFIG_ENTRY *e;

    e=Find(name, CONFIG_TYPE_DOUBLE);
    if (e==0)
        return FALSE;
    return ConfigGetValueDouble(e, d);
}


/**************************************************************
** NAME:        ConfigSetString  
** SYNOPSIS:    BOOLEAN ConfigSetString(BYTE *name, char *s)
** DESCRIPTION: Sets variable <name> to <s>.
**              The string is copied so you can destroy s.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigSetString(BYTE *name, char *s)
{
    CONFIG_ENTRY *e;

    e=Create(name);
    if (e==0)
        return FALSE;
    return ConfigSetValueString(e,s);
}


/**************************************************************
** NAME:        ConfigGetString  
** SYNOPSIS:    BOOLEAN ConfigGetString(BYTE *name, char **s)
** DESCRIPTION: Gets variable <name> in <s>.
**              You get a pointer to the string in s.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigGetString(BYTE *name, char **s)
{
    CONFIG_ENTRY *e;

    e=Find(name, CONFIG_TYPE_STRING);
    if (e==0)
        return FALSE;
    return ConfigGetValueString(e,s);
}


/**************************************************************
** NAME:        ConfigSetBytes  
** SYNOPSIS:    BOOLEAN ConfigSetBytes(BYTE *name, BYTE *b,
**                                          USHORT n)
** DESCRIPTION: Sets variable <name> to <b> with length <n>.
**              The buffer is copied so you can destroy b.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigSetBytes(BYTE *name, BYTE *b, USHORT n)
{
    CONFIG_ENTRY *e;

    e=Create(name);
    if (e==0)
        return FALSE;
    return ConfigSetValueBytes(e,b,n);
}


/**************************************************************
** NAME:        ConfigGetBytes  
** SYNOPSIS:    BOOLEAN ConfigGetBytes(BYTE *name, BYTE **b,
**                                          USHORT *n)
** DESCRIPTION: Gets variable <name> in <b>, length in <n>.
**              You get a pointer to the buffer.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigGetBytes(BYTE *name, BYTE **b, USHORT *n)
{
    CONFIG_ENTRY *e;

    e=Find(name, CONFIG_TYPE_BYTES);
    if (e==0)
        return FALSE;
    return ConfigGetValueBytes(e,b,n);
}


/**************************************************************
** NAME:        ConfigSetLongBuf
** SYNOPSIS:    BOOLEAN ConfigSetLongBuf(BYTE *name, LONG *b,
**                                          USHORT n)
** DESCRIPTION: Sets variable <name> to <b> with length <n>.
**              The buffer is copied so you can destroy b.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigSetLongBuf(BYTE *name, LONG *b, USHORT n)
{
    CONFIG_ENTRY *e;

    e=Create(name);
    if (e==0)
        return FALSE;
    return ConfigSetValueLongBuf(e,b,n);
}


/**************************************************************
** NAME:        ConfigGetLongBuf
** SYNOPSIS:    BOOLEAN ConfigGetLongBuf(BYTE *name, LONG **b,
**                                          USHORT *n)
** DESCRIPTION: Gets variable <name> in <b>, length in <n>.
**              You get a pointer to the buffer.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigGetLongBuf(BYTE *name, LONG **b, USHORT *n)
{
    CONFIG_ENTRY *e;

    e=Find(name, CONFIG_TYPE_LONGBUF);
    if (e==0)
        return FALSE;
    return ConfigGetValueLongBuf(e,b,n);
}



/* static help functions */
/*
static BOOLEAN HighRead(BYTE *name, USHORT level)
{
    FILE *f;
    
    f=fopen(name,"rb");
    if (f==0)
        return FALSE;
    if (!ConfigRead(f,level))
        return FALSE;
    fclose(f);
    return TRUE;
}
*/
/*    
static BOOLEAN Write(BYTE *name, USHORT level)
{
    FILE *f;
    
    f=fopen(name,"wb");
    if (f==0)
        return FALSE;
    if (!ConfigWrite(f,level))
        return FALSE;
    fclose(f);
    return TRUE;
}
*/

static CONFIG_ENTRY *Create(BYTE *name)
{
    CONFIG_ENTRY *e;
    
    e=ConfigFind(name);
    if (e==0)
        e=ConfigCreate(name);
    if (e==0)
        return 0;
    ConfigSetLevel(e,0);
    return e;
}


static CONFIG_ENTRY *Find(BYTE *name, USHORT type)
{
    CONFIG_ENTRY *e;
    
    e=ConfigFind(name);
    if (e==0)
        return FALSE;
    if (e->type!=type)
        return FALSE;
    return e;
}


static BOOLEAN Base(BYTE *name, BYTE *base, int sizeBase)
{
    int i,n,l;

    l=strlen(name);
    i=l;
    while (i>=0 && name[i]!='.')
        i--;
    if (i<=0)
        i=l;
    n=0;
    while (i>=0 && name[i]!='\\' &&  name[i]!='/')
    {
        i--;
        n++;
    }
    i++;
    n--;
    if (n>=sizeBase)
        return FALSE;
    memcpy(base,name+i,n);
    base[n]='\0';
    /* printf("%s\n",base); */		/* JvO */
    return TRUE;
}
