/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       low.c
**     SYSTEM   NAME:       Config
**     ORIGINAL AUTHOR(S):  Dirk Wisse
**     VERSION  NUMBER:     0.1
**     CREATION DATE:       1992/7/28
**
** DESCRIPTION:             Configuration storage/retreival.
**                          Low level interface.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <memory.h>
#include <string.h>
#include <dnpap.h>

#include "config.h"



/* static prototypes of help functions */

static USHORT Hash(BYTE *name);
static BOOLEAN Clear(CONFIG_ENTRY *e);
static BOOLEAN Write(FILE *f, CONFIG_ENTRY *e);
static BOOLEAN LowRead(FILE *f, CONFIG_ENTRY *e);


/* global variables */

#define CONFIG_HASH_SIZE  99
CONFIG_ENTRY *configHash[CONFIG_HASH_SIZE];
CONFIG_ENTRY *configList=0;



/* interface functions */

/**************************************************************
** NAME:        ConfigFlush(void)     
** SYNOPSIS:    BOOLEAN ConfigFlush(void)
** DESCRIPTION: Flushes all the variables.
** RETURNS:     BOOLEAN success.
**************************************************************/
BOOLEAN ConfigFlush(void)
{
    CONFIG_ENTRY *e;

    e = configList;
    while (e != 0)
    {
        Clear(e);
        DnpapFree(e);
        e = e->next;
    }
    return TRUE;
}


/**************************************************************
** NAME:        ConfigFirst     
** SYNOPSIS:    BOOLEAN ConfigFirst(void)
** DESCRIPTION: Returns lex-first entry.
** RETURNS:     CONFIG_ENTRY * pointer to first entry.
**************************************************************/
CONFIG_ENTRY *ConfigFirst(void)
{
    return configList;
}


/**************************************************************
** NAME:        ConfigNext     
** SYNOPSIS:    CONFIG_ENTRY *ConfigNext(CONFIG_ENTRY *e)
** DESCRIPTION: Returns lex-next entry after <e>.
** RETURNS:     CONFIG_ENTRY * pointer to entry after <e>.
**************************************************************/
CONFIG_ENTRY *ConfigNext(CONFIG_ENTRY *e)
{
    return e->next;
}


/**************************************************************
** NAME:        ConfigFind 
** SYNOPSIS:    CONFIG_ENTRY *ConfigFind(BYTE *name)
** DESCRIPTION: Returns entry with <name>.
** RETURNS:     CONFIG_ENTRY * pointer to entry with <name>.
**************************************************************/
CONFIG_ENTRY *ConfigFind(BYTE *name)
{
    CONFIG_ENTRY *e;
    USHORT i;
    
    i = Hash(name);
    for (e=configHash[i]; e!=0; e=e->nextHash)
    {
        if (strcmp(e->name, name)==0)
            break;
    }
    return e;
}


/**************************************************************
** NAME:        ConfigCreate     
** SYNOPSIS:    CONFIG_ENTRY *ConfigCreate(BYTE *name)
** DESCRIPTION: Creates entry with <name>.
**              If entry already exists the existing entry
**              is returned.
** RETURNS:     CONFIG_ENTRY * pointer to entry with <name>.
**************************************************************/
CONFIG_ENTRY *ConfigCreate(BYTE *name)
{
    CONFIG_ENTRY *e,**p;
    USHORT i;
    
    e = ConfigFind(name);
    if (e!=0)
        return e;
    
    e = DnpapMalloc(sizeof(CONFIG_ENTRY));
    if (e==0)
        return 0;
    memset(e,0,sizeof(CONFIG_ENTRY));

    e->name = DnpapMalloc(strlen(name)+1);
    if (e->name==0)
        return 0;
    strcpy(e->name,name);
    
    for (p=&configList; *p!=0; p=&(*p)->next)
    {
        if (strcmp((*p)->name,name)>=0)
            break;
    }
    e->next = *p;
    *p = e;
    
    i = Hash(name);
    e->nextHash = configHash[i]; 
    configHash[i] = e;

    e->type=CONFIG_TYPE_NULL;
    e->level=0;

    return e;
}


/**************************************************************
** NAME:        ConfigRead     
** SYNOPSIS:    BOOLEAN ConfigRead(char *file, USHORT level)
** DESCRIPTION: Reads entries from <file> with level <l>.
**              The level determines if existing entries
**              are overwritten. The lower the level the
**              higher the priority.
** RETURNS:     BOOLEAN successful.
**************************************************************/
BOOLEAN ConfigRead(char *file, USHORT level)
{
    CONFIG_ENTRY *e;
    CONFIG_ENTRY junk;
    BYTE name[100];
    USHORT n,l;
    FILE   *f;

    f=fopen(file,"rb");
    if (f==NULL)
        return FALSE;

    while(fread(&n,sizeof(USHORT),1,f)==1)
    {
        if (n>=sizeof(name))
        {
            fclose(f);
            return FALSE;
        }
        if (fread(name,n,1,f)!=1)
        {
            fclose(f);
            return FALSE;
        }
        name[n]='\0';
        e=ConfigFind(name);
        if (e!=0)
        {
            if (!ConfigGetLevel(e,&l))
            {
                fclose(f);
                return FALSE;
            }
            if (l<level)
                e=&junk;
        }
        else
        {
            e=ConfigCreate(name);
            if (e==0)
            {
                fclose(f);
                return FALSE;
            }
        }
        if (!ConfigSetLevel(e,level))
            return FALSE;
        if (!LowRead(f,e))
        {
            fclose(f);
            return FALSE;
        }
        if (e==&junk)
        {
            ConfigClear(e);           /* JUCK !!!  DMW & RK */
        }
    }
    fclose(f);
    return TRUE;		/* JvO */
}


/**************************************************************
** NAME:        ConfigWrite     
** SYNOPSIS:    BOOLEAN ConfigWrite(char *file, USHORT level)
** DESCRIPTION: Writes entries with level <= <l> to file <f>.
**              The level determines if an entry is written
**              to file.
** RETURNS:     BOOLEAN successful.
**************************************************************/
BOOLEAN ConfigWrite(char *file, USHORT level)
{
    CONFIG_ENTRY *e;
    BYTE *name;
    USHORT n,l;
    FILE *f;
    
    f=fopen(file,"wb");
    if (f==NULL)
        return FALSE;

    e = ConfigFirst();
    while(e!=0)
    {
        if (!ConfigGetLevel(e,&l))
        {
            fclose(f);
            return FALSE;
        }
        if (l <= level)
        {
            if (!ConfigGetName(e,&name))
                return FALSE;
            n=strlen(name);
            if (fwrite(&n,sizeof(USHORT),1,f)!=1)
            {
                fclose(f);
                return FALSE;
            }
            if (fwrite(name,n,1,f)!=1)
            {
                fclose(f);
                return FALSE;
            }
            if (!Write(f,e))
            {
                fclose(f);
                return FALSE;
            }
        }
        e = ConfigNext(e);
    }
    fclose(f);
    return TRUE;
}


/**************************************************************
** NAME:        ConfigGetName   
** SYNOPSIS:    BOOLEAN ConfigGetName(CONFIG_ENTRY *e,
**                                              BYTE **name)
** DESCRIPTION: Returns the name of <e> in <name>.
** RETURNS:     BOOLEAN successful.
**************************************************************/
BOOLEAN ConfigGetName(CONFIG_ENTRY *e, BYTE **name)
{
    *name=e->name;
    return TRUE;
}


/**************************************************************
** NAME:        ConfigGetType   
** SYNOPSIS:    BOOLEAN ConfigGetType(CONFIG_ENTRY *e,
**                                              USHORT *type)
** DESCRIPTION: Returns the type of <e> in <type>.
** RETURNS:     BOOLEAN successful.
**************************************************************/
BOOLEAN ConfigGetType(CONFIG_ENTRY *e, USHORT *type)
{
    *type=e->type;
    return TRUE;
}


/**************************************************************
** NAME:        ConfigGetLevel   
** SYNOPSIS:    BOOLEAN ConfigGetLevel(CONFIG_ENTRY *e,
**                                            USHORT *level)
** DESCRIPTION: Returns the level of <e> in <level>.
** RETURNS:     BOOLEAN successful.
**************************************************************/
BOOLEAN ConfigGetLevel(CONFIG_ENTRY *e, USHORT *level)
{
    *level=e->level;
    return TRUE;
}


/**************************************************************
** NAME:        ConfigSetLevel   
** SYNOPSIS:    BOOLEAN ConfigSetLevel(CONFIG_ENTRY *e,
**                                            USHORT *level)
** DESCRIPTION: Sets the level of <e> to <level>.
** RETURNS:     BOOLEAN successful.
**************************************************************/
BOOLEAN ConfigSetLevel(CONFIG_ENTRY *e, USHORT level)
{
    e->level=level;
    return TRUE;
}


/**************************************************************
** NAME:        ConfigClear
** SYNOPSIS:    BOOLEAN ConfigClear(CONFIG_ENTRY *e)
** DESCRIPTION: Clears the entry.
**              Sets the type to CONFIG_TYPE_NULL.
** RETURNS:     BOOLEAN successful.
**************************************************************/
BOOLEAN ConfigClear(CONFIG_ENTRY *e)
{
    return Clear(e);
}



/* static help functions */

static USHORT Hash(BYTE *name)
{
    USHORT hash;

    hash = 0;
    while (*name)
        hash=(hash*3)+*name++;			/* AK */
    hash %= CONFIG_HASH_SIZE;

    return hash;
}



/* change the following functions if you want to add a type */


static BOOLEAN Clear(CONFIG_ENTRY *e)
{
    switch(e->type)
    {
        case CONFIG_TYPE_NULL:
            break;
        case CONFIG_TYPE_BOOLEAN:
            if (!ConfigDelValueBoolean(e))
                return FALSE;
            break;
        case CONFIG_TYPE_SHORT:
            if (!ConfigDelValueShort(e))
                return FALSE;
            break;
        case CONFIG_TYPE_LONG:
            if (!ConfigDelValueLong(e))
                return FALSE;
            break;
        case CONFIG_TYPE_IPADDR:
            if (!ConfigDelValueIPAddr(e))
                return FALSE;
            break;
        case CONFIG_TYPE_DOUBLE:
            if (!ConfigDelValueDouble(e))
                return FALSE;
            break;
        case CONFIG_TYPE_STRING:
            if (!ConfigDelValueString(e))
                return FALSE;
            break;
        case CONFIG_TYPE_BYTES:
            if (!ConfigDelValueBytes(e))
                return FALSE;
            break;
        case CONFIG_TYPE_LONGBUF:
            if (!ConfigDelValueLongBuf(e))
                return FALSE;
            break;
    }
    e->type=CONFIG_TYPE_NULL;
    return TRUE;
}


static BOOLEAN LowRead(FILE *f, CONFIG_ENTRY *e)
{
    USHORT type;
    
    if (fread(&type,sizeof(USHORT),1,f)!=1)
    {
        fclose(f);
        return FALSE;
    }
    switch(type)
    {
        case CONFIG_TYPE_NULL:
            break;
        case CONFIG_TYPE_BOOLEAN:
        {
            BOOLEAN b;

            if (fread(&b,sizeof(BOOLEAN),1,f)!=1)
                return FALSE;
            if (e!=0 && !ConfigSetValueBoolean(e,b))
                return FALSE;
            break;
        }
        case CONFIG_TYPE_SHORT:
        {
            short s;

            if (fread(&s,sizeof(short),1,f)!=1)
                return FALSE;
            if (e!=0 && !ConfigSetValueShort(e,s))
                return FALSE;
            break;
        }
        case CONFIG_TYPE_LONG:
        {
            LONG l;

            if (fread(&l,sizeof(LONG),1,f)!=1)
                return FALSE;
            if (e!=0 && !ConfigSetValueLong(e,l))
                return FALSE;
            break;
        }
        case CONFIG_TYPE_IPADDR:
        {
            IPADDR a;

            if (fread(&a,sizeof(IPADDR),1,f)!=1)
                return FALSE;
            if (e!=0 && !ConfigSetValueIPAddr(e,a))
                return FALSE;
            break;
        }
        case CONFIG_TYPE_DOUBLE:
        {
            double d;

            if (fread(&d,sizeof(double),1,f)!=1)
                return FALSE;
            if (e!=0 && !ConfigSetValueDouble(e,d))
                return FALSE;
            break;
        }
        case CONFIG_TYPE_STRING:
        {
            USHORT n;
            BYTE   s[100];

            if (fread(&n,sizeof(USHORT),1,f)!=1)
                return FALSE;
            if (n >= sizeof(s))
                return FALSE;
            if (n > 0 && fread(s,n,1,f)!=1)
                return FALSE;
            s[n]='\0';
            if (e!=0 && !ConfigSetValueString(e,s))
                return FALSE;
            break;
        }
        case CONFIG_TYPE_BYTES:
        {
            USHORT n;
            BYTE   b[100];

            if (fread(&n,sizeof(USHORT),1,f)!=1)
                return FALSE;
            if (n > sizeof(b))
                return FALSE;
            if (n > 0 && fread(b,n,1,f)!=1)
                return FALSE;
            if (e!=0 && !ConfigSetValueBytes(e,b,n))
                return FALSE;
            break;
        }
        case CONFIG_TYPE_LONGBUF:
        {
            USHORT n;
            LONG   b[100];

            if (fread(&n,sizeof(USHORT),1,f)!=1)
                return FALSE;
            if (n > sizeof(b))
                return FALSE;
            if (n > 0 && fread(b,n*sizeof(LONG),1,f)!=1)
                return FALSE;
            if (e!=0 && !ConfigSetValueLongBuf(e,b,n))
                return FALSE;
            break;
        }
    }
    return TRUE;
}


static BOOLEAN Write(FILE *f, CONFIG_ENTRY *e)
{
    USHORT type;
    
    if (!ConfigGetType(e,&type))
        return FALSE;
    if (fwrite(&type,sizeof(USHORT),1,f)!=1)
    {
        fclose(f);
        return FALSE;
    }
    switch(type)
    {
        case CONFIG_TYPE_NULL:
            break;
        case CONFIG_TYPE_BOOLEAN:
        {
            BOOLEAN b;

            if (!ConfigGetValueBoolean(e,&b))
                return FALSE;
            if (fwrite(&b,sizeof(BOOLEAN),1,f)!=1)
                return FALSE;
            break;
        }
        case CONFIG_TYPE_SHORT:
        {
            short s;

            if (!ConfigGetValueShort(e,&s))
                return FALSE;
            if (fwrite(&s,sizeof(short),1,f)!=1)
                return FALSE;
            break;
        }
        case CONFIG_TYPE_LONG:
        {
            LONG l;

            if (!ConfigGetValueLong(e,&l))
                return FALSE;
            if (fwrite(&l,sizeof(LONG),1,f)!=1)
                return FALSE;
            break;
        }
        case CONFIG_TYPE_IPADDR:
        {
            IPADDR a;

            if (!ConfigGetValueIPAddr(e,&a))
                return FALSE;
            if (fwrite(&a,sizeof(IPADDR),1,f)!=1)
                return FALSE;
            break;
        }
        case CONFIG_TYPE_DOUBLE:
        {
            double d;

            if (!ConfigGetValueDouble(e,&d))
                return FALSE;
            if (fwrite(&d,sizeof(double),1,f)!=1)
                return FALSE;
            break;
        }
        case CONFIG_TYPE_STRING:
        {
            USHORT n;
            BYTE   *s;

            if (!ConfigGetValueString(e,(char **) &s)) 	/* JvO */
                return FALSE;
            n=strlen(s);
            if (fwrite(&n,sizeof(USHORT),1,f)!=1)
                return FALSE;
            if (n > 0 && fwrite(s,n,1,f)!=1)
                return FALSE;
            break;
        }
        case CONFIG_TYPE_BYTES:
        {
            USHORT n;
            BYTE   *b;

            if (!ConfigGetValueBytes(e,&b,&n))
                return FALSE;
            if (fwrite(&n,sizeof(USHORT),1,f)!=1)
                return FALSE;
            if (n > 0 && fwrite(b,n,1,f)!=1)
                return FALSE;
            break;
        }
        case CONFIG_TYPE_LONGBUF:
        {
            USHORT n;
            LONG   *b;

            if (!ConfigGetValueLongBuf(e,&b,&n))
                return FALSE;
            if (fwrite(&n,sizeof(USHORT),1,f)!=1)
                return FALSE;
            if (n > 0 && fwrite(b,n*sizeof(LONG),1,f)!=1)
                return FALSE;
            break;
        }
    }
    return TRUE;
}
