/**********************************************************************
** MODULE INFORMATION*
**********************
**      FILE     NAME:       MSGTRANS.C
**      SYSTEM   NAME:       TRANSCRIPT
**      ORIGINAL AUTHOR(S):  Alfred Kayser
**      VERSION  NUMBER:     1.0
**      CREATION DATE:       91/09/16
**
** DESCRIPTION: Transcript functions.
**
** CONFIG:      transcript.medium.a = file:beholder.log
**              transcript.medium.b = vip:logwin
**              transcript.trans.jan   = a      ; These make output to Jan goto a
**              transcript.trans.piet  = b      ; These make output to Piet goto b
**              transcript.trans.klaas = a,b    ; This makes output to Klaas goto a and b
**              
**              ;special startup transscript
**              transcript.startup = std:out
**               
**              logwin.vip.title = "LogWindow..."
**              logwin.vip.location = 10x10 
**              logwin.vip.size = 40x5
**                                                     
**              load.error.trans.debug = piet    
**              load.error.level.debug = 2
**              
**              error.level.error = 4               
**              error.trans.error = Jan
**              error.level.debug = -1
**              error.level.log = -1
***********************************************************************
** CHANGES INFORMATION **
*************************
** REVISION: $Revision: $
** AUTHOR:   $Author:  $
** DATE:     $Date: $
** LOG:      $Log: $
**
**********************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <dnpap.h>
#include <config.h>
#include "message.h"
                   
/*--------------------Local Defines-------------------------*/

#define ERROR(n,s)       DnpapMessage(DMC_ERROR,TRANS_ERROR+n, s)
#define ERROR1(n,s,a)    DnpapMessage(DMC_ERROR,TRANS_ERROR+n, s, a)
#define ERROR2(n,s,a,b)  DnpapMessage(DMC_ERROR,TRANS_ERROR+n, s, a, b)

#define MAXDEVICES 32
#define MAXTRANS   32
#define MAXMEDIA   32
#define MAXATTACH  8


/*--------------------Local Typedefs------------------------*/

/* MEDIUM holds an handle of an opened device */    
typedef struct _medium MEDIUM;
    struct _medium
    {
        CONST char  *name;
        void        *handle;
        LONG         refCount;
        TRANSDEVICE *device;
        char        *devArgs;
    };                                                                                        

/* TRANS holds a set of media */
    struct _trans       
    {              
        int         mediaCount;
        MEDIUM     *media[MAXATTACH];
    };


/*--------------Local Prototypes--------------------------*/

PRIVAT void    TransExit          __((void));
PRIVAT void    TransConfigMedium  __((CONST char *treename));
PRIVAT MEDIUM *TransConnectMedium __((CONST char *));
PRIVAT int     TransFindDevice    __((CONST char *));
PRIVAT MEDIUM *TransFindMedium    __((CONST char *));

PRIVAT void  MediumDisconnect __((MEDIUM *));
PRIVAT void  MediumClose      __((MEDIUM *));
PRIVAT void  MediumFlush      __((MEDIUM *));
PRIVAT void  MediumMessage    __((MEDIUM *, CONST char *));

PRIVAT CONFIG_ENTRY *MyConfigTreeFind __((CONST BYTE *name));
PRIVAT CONFIG_ENTRY *MyConfigTreeNext __((CONFIG_ENTRY *ce, CONST BYTE *name));


/*--------------Local Global Vars------------------------*/

PRIVAT TRANSDEVICE deviceList[MAXDEVICES];
PRIVAT int         deviceCount=0;
PRIVAT MEDIUM      mediumList[MAXMEDIA];
PRIVAT int         mediumCount=0;
PRIVAT MEDIUM *    defaultMedium=NULL;



/**************************************************************
** NAME:        TransInit
** SYNOPSIS:    EXPORT void
**              TransInit(CONST char *defDev, va_list vp);
** DESCRIPTION: Initializes the transcript module.
**              <vp> points to a va_arg list of TRANSDEVICE
**              pointers which ends with a NULL pointer.
**              These arguments can be MSGFILE, MSGSTDIO or
**              MSGVIP (if VIP is used).
**              <defDev> can contain a medium name, to be
**              opened, when 'message.startup' can't be found.
** EXAMPLE:     TransInit("stdio:err", MSGFILE, MSGSTDIO, NULL);
** RETURNS:     void
**************************************************************/
EXPORT void
TransInit(CONST char *defDevice)
{                                   
    char *startDev;
    
    /* Make sure that our exit function will be called at exit */
    DnpapAtExit(TransExit);

    /* Get the Startup transcript */
    if (ConfigGetString("message.startup", &startDev))
        TransCreateMedium("startup", startDev);
										  
	/* Try to connect to default medium */
    defaultMedium = TransConnectMedium("startup");

    if (!defaultMedium) /* Oops, failed to open default version */
    	if (defDevice && strchr(defDevice,':'))
        {
    		TransCreateMedium("startup", defDevice);
            defaultMedium = TransConnectMedium("startup");
        }
		
    /* Then define some media */
    TransConfigMedium("message.medium.");
    
    /* Redir errors from transcript modules to defined transcripts */
    MessageConfig(TRANS_ERROR,"transcript"); 
}


/**************************************************************
** NAME:        TransExit
** SYNOPSIS:    PRIVAT void
**              TransExit()
** DESCRIPTION: Closes the transcript system down. This
**              function is called automatically by the
**              DnpapExit function.
** RETURNS:     void
**************************************************************/
PRIVAT void
TransExit()
{
    int i;
    for (i=0;i<mediumCount;i++)
        MediumClose(&mediumList[i]);
}


/**************************************************************
** NAME:        TransOpen                                 [API]
** SYNOPSIS:    EXPORT TRANSCRIPT *
**              TransOpen(CONST char *mediaList)
** DESCRIPTION: Opens a transcript to several media.
**              The media is specified in a string, seperated
**				with ','s. Ie. 
**				  trans = TransOpen("stdio:out,file:test.log");
**				P.s. no spaces are allowed!
** RETURNS:     void
**************************************************************/
EXPORT TRANSCRIPT *
TransOpen(CONST char *mediaList)
{
    static char medium[255];
    char *p;
    MEDIUM *mp;
    TRANSCRIPT *tp;

    if (!mediaList) return NULL;
    tp = (TRANSCRIPT *)DnpapMalloc(sizeof(TRANSCRIPT));
    if (!tp)
        return NULL;
    tp->mediaCount=0;
    while(mediaList)
    {
        p=(BYTE*)strchr(mediaList,',');
        if (p)
        {
            memcpy(medium, mediaList, p-mediaList);
            medium[p-mediaList]='\0';
            mp=TransConnectMedium(medium);
            mediaList=p+1;
        }
        else
        {
            mp=TransConnectMedium(mediaList);
            mediaList=NULL;
        }
        if (mp)
        {
            if (tp->mediaCount<=MAXATTACH)
                tp->media[tp->mediaCount++]=mp;
            else
                ERROR(1,"Too many media in transcript");
        }
    }
    return tp;                                                 
}


/**************************************************************
** NAME:        TransMessage                              [API]
** SYNOPSIS:    EXPORT void 
**              TransMessage(TRANSCRIPT *tnr,
**                           CONST char *format VAR_PROTO)
** DESCRIPTION: Send the message <format> to the transcript.
**              If <tnr> is NULL, then the startup transcript
**              will be used.
** RETURNS:     void
**************************************************************/
EXPORT void 
TransMessage(TRANSCRIPT *tp, CONST char *format VAR_PROTO)
{
    PRIVAT char formatBuf[1024];
    va_list vp;
    int i;
    
    VAR_START(vp,format);
    vsprintf(formatBuf,format,vp);
    va_end(vp);
    if (tp==NULL) /* if no transcript was found, use default medium */
        MediumMessage(defaultMedium, formatBuf);
    else
        for (i=0;i<tp->mediaCount;i++)
            MediumMessage(tp->media[i], formatBuf);            
}

     
/**************************************************************
** NAME:        TransFlush                                [API]
** SYNOPSIS:    EXPORT void 
**              TransFlush(TRANSCRIPT *tp)
** DESCRIPTION: Flushes the transcript. It makes sure that
**              all bytes are flushed to the device.
** RETURNS:     void
**************************************************************/
EXPORT void 
TransFlush(TRANSCRIPT *tp)
{
    int i;
    
    if (tp==NULL)
        MediumFlush(defaultMedium);
    else
        for (i=0;i<tp->mediaCount;i++)
            MediumFlush(tp->media[i]);
}

                                                          
/**************************************************************
** NAME:        TransClose                                [API]
** SYNOPSIS:    EXPORT void 
**              TransClose(TRANSCRIPT *tp)
** DESCRIPTION: Closes the transcript. If the ref. count
**              of the used medium reached 0, then the medium
**              will be closed.
** RETURNS:     void
**************************************************************/
EXPORT void 
TransClose(TRANSCRIPT *tp)
{
    int i;
    
    if (tp!=NULL)
    {
        for (i=0;i<tp->mediaCount;i++)
            MediumDisconnect(tp->media[i]);
        DnpapFree(tp);
    }
}


/**************************************************************
** NAME:        MediumMessage                             
** SYNOPSIS:    PRIVAT void      
**              MediumMessage(MEDIUM *mp, CONST char *msg)
** DESCRIPTION: Puts the message in the medium <mp>.
** RETURNS:     void
**************************************************************/
PRIVAT void      
MediumMessage(MEDIUM *mp, CONST char *msg)
{
    if (mp && mp->handle) 
        mp->device->put(mp->handle,msg);
}                    


/**************************************************************
** NAME:        MediumFlush
** SYNOPSIS:    PRIVAT void        
**              MediumFlush(MEDIUM *mp)
** DESCRIPTION: Flushes the medium <mp>.
** RETURNS:     void
**************************************************************/
PRIVAT void        
MediumFlush(MEDIUM *mp)
{
    if (mp && mp->handle)
        mp->device->flush(mp->handle);
}                    


/**************************************************************
** NAME:        MediumDisconnect
** SYNOPSIS:    PRIVAT void
**              MediumDisconnect(MEDIUM *mp)
** DESCRIPTION: Disconnects the medium <mp>.
** RETURNS:     void
**************************************************************/
PRIVAT void
MediumDisconnect(MEDIUM *mp)
{
    if (mp && mp->refCount)
    {
        mp->refCount--;
        if (mp->refCount==0) MediumClose(mp);
    }
}


/**************************************************************
** NAME:        MediumClose
** SYNOPSIS:    PRIVAT void
**              MediumClose(MEDIUM *mp)
** DESCRIPTION: Closes the medium <mp>.
** RETURNS:     void
**************************************************************/
PRIVAT void
MediumClose(MEDIUM *mp)
{
    if (mp && mp->handle)
    {
        mp->device->close(mp->handle);
        mp->handle=0;
    }
}                    

                                                               
/**************************************************************
** NAME:        TransAddDevice                            [API]
** SYNOPSIS:    EXPORT void
**              TransAddDevice(TRANSDEVICE *td)
** DESCRIPTION: Adds a device to the transcript system.
**              The <td> structure has the following layout:
**          		{
**          			CONST char *name;
**          			TRANSDOPEN  open;
**          			TRANSDPUT   put;
**          			TRANSDFLUSH flush;
**          			TRANSDCLOSE close;
**          		};
**              The <name> should be unique. The <open>, <put>,
**              <flush> and <close> elements are functions
**              which are called by the transcript library and
**              they have the following prototypes:
**              void *open  (const char *);  
**              void  put   (void *, const char *);
**              void  flush (void *);
**              void  close (void *);
**
** RETURNS:     void
**************************************************************/
EXPORT void
TransAddDevice(TRANSDEVICE *td)
{                                   
    if (!td) return;
    if (deviceCount==MAXDEVICES)
        ERROR(2, "Too many transcript devices");
    else
    {
        if (TransFindDevice(td->name)>0)
            ERROR1(3,"Transcript device <%s> is already defined", td->name);
        else
        {
            deviceList[deviceCount++]=*td;
            if (td->open==NULL)  td->open  = messageNullDev.open;
            if (td->put==NULL)   td->put   = messageNullDev.put;
            if (td->flush==NULL) td->flush = messageNullDev.flush;
            if (td->close==NULL) td->close = messageNullDev.close;
        }
    }
}


PRIVAT void 
TransConfigMedium(CONST char *treename)
{                   
    CONFIG_ENTRY *node;
    BYTE *label;
    char *device;
    int len=strlen(treename);
    
/* Config: TRANSCRIPT.MEDIUM.NAME = DEVICE:OPENARGS */
    for (node=MyConfigTreeFind(treename);node;node=MyConfigTreeNext(node, treename))
    {            
        if (!ConfigGetName(node, &label))
        {
            ERROR(4,"Failed to get name of config variable");
            continue;
        }
        if (!ConfigGetString(label, &device))
        {
            ERROR1(5,"Failed to get string of config variable: %s", label);
            continue;    /* Try next config var */
        }
        TransCreateMedium(label+len, device);
    }             
}


EXPORT void
TransCreateMedium(CONST char *logname, CONST char *devargs)
{
    char *args, *device, buf[200];
    int dev, len;

    device=(char*)devargs;
    args=(char*)strchr(devargs,':');
    if (args)
    {
        memcpy(buf,device,args-devargs);
        buf[args-devargs]='\0';
        device=buf;
        args++;
    }
    dev = TransFindDevice(device);
    if (dev<=0)
        ERROR1(6, "Can't find device %s", device);
    else
    {
        if (TransFindMedium(logname)!=NULL)
            ERROR1(7,"Medium %s is allready defined", logname);
        else if (mediumCount==MAXMEDIA)
            ERROR(8,"Too many media opened");
        else        
        {
            mediumList[mediumCount].name     = logname;
            mediumList[mediumCount].device   = &deviceList[dev-1];
            mediumList[mediumCount].handle   = NULL;  
            mediumList[mediumCount].refCount = 0;
            mediumList[mediumCount].devArgs  = NULL;
            if (args)
            {
                len=strlen(args)+1;
                mediumList[mediumCount].devArgs=DnpapMalloc(len);
                memcpy(mediumList[mediumCount].devArgs, args, len);
            }
            mediumCount++;
        }
    }
}


PRIVAT MEDIUM *
TransConnectMedium(CONST char *medium)
{
    MEDIUM *mp;
    
    mp=TransFindMedium(medium);
    if (!mp) return NULL;
    if (mp->refCount==0)
    {
        mp->handle=mp->device->open(mp->devArgs);
        if (mp->handle==NULL)
            ERROR2(9,"Open device '%s:%s' failed", mp->device->name, mp->devArgs);
    }
    mp->refCount++;
    return mp;
}


PRIVAT int 
TransFindDevice(CONST char *name)
{
    int i;

    for (i=0;i<deviceCount;i++)
        if (stricmp(deviceList[i].name,name)==0)
            return i+1;
    return 0;
}


PRIVAT MEDIUM *
TransFindMedium(CONST char *name)
{
    int i;

    for (i=0;i<mediumCount;i++)
        if (stricmp(mediumList[i].name,name)==0)
            return &mediumList[i];
    return NULL;
}


PRIVAT CONFIG_ENTRY *
MyConfigTreeFind(CONST BYTE *name)
{
    CONFIG_ENTRY *ce;
    BYTE *keyname;                    
    int match, len=strlen(name);
    
    ce=ConfigFirst();
    while (ce)
    {
        if (ConfigGetName(ce, &keyname))
        {
            if (*keyname>*name) break;
            if (*keyname==*name) 
            {
                match=memcmp(keyname, name, len);
                if (match==0) return ce;
                if (match>0) break;
            }
        }
        ce=ConfigNext(ce);
    }                      
    return NULL;
}
                               

PRIVAT CONFIG_ENTRY *
MyConfigTreeNext(CONFIG_ENTRY *ce, CONST BYTE *name)
{
    BYTE *keyname;
            
    while (ce=ConfigNext(ce))
        if (ConfigGetName(ce, &keyname))
            return (memcmp(keyname, name, strlen(name))==0)?ce:NULL;
    return NULL;
}



