/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       vipgraph.c
**     SYSTEM   NAME:       VIP
**     ORIGINAL AUTHOR(S):  Alfred Kayser
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       1992/5/29
**
** DESCRIPTION: Graph Module.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#define LIBRARY
#include "vipinc.h"

PRIVAT void VipUpdateGraph(VIPINFO *wip, HPS hps, BOOLEAN all);
PRIVAT char graphBuf[255];

IMPORT LONG vipColors[VIP_COLORS];


/**************************************************************
** NAME:        VipOpenGraph                              [API]
** SYNOPSIS:    VIPINFO * VipOpenGraph(VIPINFO *parent,
**                  int x, int y, int w, int h)
** DESCRIPTION: Opens a graph window. This window can contain
**              a number of datasets. Each dataset has its own
**              dimensions. It is advisable to use the same
**              x dimension for all datasets.
** RETURNS:     window pointer, or
**              NULL, when out of memory.
**************************************************************/
VIPINFO *
VipOpenGraph(VIPINFO *parent, int x, int y, int w, int h)
{
    VIPINFO *wip;
    struct _graphdata *data;
    int i;

    if (!(data = VipMalloc(sizeof(struct _graphdata))))
        return NULL;
    if ( !(wip = VipOpenSimple(parent, x,y,w,h)))
    {
        VipFree(data);
        return NULL;
    }
    
    for (i=0;i<MAXSETS;i++)
    {
        data->datasets[i].values=NULL;
        data->datasets[i].legend=NULL;
    }
    data->xstripes = 5;
    data->ystripes = 5;

    wip->graphdata=data;
    wip->type=T_GRAPH;
    wip->border=2;
    wip->btype=VIP_DEPTH;

    /* Change Update function to a graph draw function... */
    wip->update=VipUpdateGraph;
    return wip;
}


/**************************************************************
** NAME:        VipSetGraphValue                          [API]
** SYNOPSIS:    void VipSetGraphValue(VIPINFO *wip,
**                    int set, int pos, LONG value)
** DESCRIPTION: Sets a value of the graph.
**              As with all VipSet functions the corresponding
**              view is not updated until a 'VipUpdate' or
**              'VipShow' forces a redraw or refresh.
** RETURNS:     void
**************************************************************/
void
VipSetGraphValue(VIPINFO *wip, int set, int pos, LONG value)
{
    TYPETEST(wip,T_GRAPH,return);
    if (set<0 || set>=MAXSETS
     || pos<0 || pos>=GRAPHSET(wip, set).size
     || GRAPHSET(wip,set).values==NULL)
        return;
    GRAPHSET(wip,set).values[pos]=value;
    GRAPHSET(wip,set).last=pos;
}


/**************************************************************
** NAME:        VipSetGraphOffset                         [API]
** SYNOPSIS:    void VipSetGraphOffset(VIPINFO *wip,
**                  int set, int offset)
** DESCRIPTION: Sets the start position of the graph.
**              This function makes it possible to scroll a
**              graph by incrementing the starting offset.
**              But it is beter to use VipSetGraphLast for
**              this purpose.
** RETURNS:     void
**************************************************************/
void
VipSetGraphOffset(VIPINFO *wip, int set, int pos)
{
    TYPETEST(wip,T_GRAPH,return);
    if (set<0 || set>=MAXSETS
     || pos<0 || pos>=GRAPHSET(wip, set).size)
        return;
    GRAPHSET(wip,set).offset=pos;
}


/**************************************************************
** NAME:        VipSetGraphLast                          [API]
** SYNOPSIS:    void VipSetGraphLast(VIPINFO *wip,
**                    int set, LONG value)
** DESCRIPTION: Sets the last value of the graph.
**              As with all VipSet functions the corresponding
**              view is not updated until a 'VipUpdate' or
**              'VipShow' forces a redraw or refresh.
**              This function adds this value to the dataset.
**              The graph is automatically scrolled if needed.
**              The graph is updated in such a way that at
**              least half of the dataset is shown, and that
**              there is minimum of redrawing...
** RETURNS:     void
**************************************************************/
void
VipSetGraphLast(VIPINFO *wip, int set, LONG value)
{
    int pos;
    TYPETEST(wip,T_GRAPH,return);
    if (set<0 || set>=MAXSETS
      || GRAPHSET(wip,set).values==NULL)
        return;
    pos=GRAPHSET(wip,set).last+1;
    if (pos>=GRAPHSET(wip,set).size) pos=0;
    GRAPHSET(wip,set).last=pos;
    GRAPHSET(wip,set).values[pos]=value;
    if (pos==GRAPHSET(wip,set).offset)
    {
        pos+=GRAPHSET(wip,set).size/2;
        if (pos>=GRAPHSET(wip,set).size)
            pos-=GRAPHSET(wip,set).size;
        GRAPHSET(wip,set).offset=pos;
        SREDRAW(wip);
    }
}


/**************************************************************
** NAME:        VipSetGraphStripes                        [API]
** SYNOPSIS:    void VipSetGraphStripes(VIPINFO *wip,
**                    int x, int y);
** DESCRIPTION: Sets the stripes of the graph.
** RETURNS:     void
**************************************************************/
void
VipSetGraphStripes(VIPINFO *wip, int x, int y)
{
    TYPETEST(wip,T_GRAPH,return);
    GRAPHDATA(wip, xstripes) = x;
    GRAPHDATA(wip, ystripes) = y;
}


/**************************************************************
** NAME:        VipSetGraphDataSet                        [API]
** SYNOPSIS:    BOOLEAN VipSetGraphDataSet(VIPINFO *wip,
**                  int set, int xsize, LONG ymin,
**                  LONG ymax, char *legend, LONG color);
** DESCRIPTION: Initializes a dataset of the graph.
**              When <legend> is NULL, no legend label will
**              will be created. 
** RETURNS:     FALSE, when allocation failed
**              TRUE, when graphdataset is initialized.
**************************************************************/
BOOLEAN
VipSetGraphDataSet(VIPINFO *wip,
    int set, int xsize, LONG ymin,
    LONG ymax, CONST char *legend, int color)
{
    GRAPHDSET *gds;
    int i;

    TYPETEST(wip,T_GRAPH,return FALSE);
    if (set<0 || set>=MAXSETS) return FALSE;
    gds = &GRAPHSET(wip,set);
    gds -> min=ymin;
    gds -> max=ymax;
    gds -> size=xsize;
    gds -> offset=0;
    gds -> last=0;
    if (gds -> values)
        VipFree(gds -> values);
    gds -> values=VipMalloc(xsize * sizeof(LONG));
    if (!gds -> values)
        return FALSE;
    for (i=0;i<xsize;i++)
        gds -> values[i]=0;
    gds -> color=color;
    gds -> label=NULL;
    VipString(&(gds -> legend), legend);
    if (gds -> legend)
    {
        gds -> label=VipOpenLabel(wip,
            (int)(set/4)*200, (int)(set%4)*200, 0, 0);
        if (gds -> label)
        {
            VipSetForeground(gds->label, color);
            VipSetTitle(gds->label, legend);
        }

    }
    return TRUE;
}


PRIVAT void
VipUpdateGraph(VIPINFO *wip, HPS hps, BOOLEAN all)
{                       
    int n, i, j, m;
    int st;
    POINTL pp;
    LONG *pts;
    LONG min, range;
    GRAPHDSET *gds;
    int WID,HID;

    WID = (int)wip->cx-2*wip->border-1;
    HID = (int)wip->cy-2*wip->border-1;
    if (all || QREDRAW(wip))
    {
        RREDRAW(wip);
        VipBorder(wip, hps);

        GpiSetColor(hps, FOREGROUND2(wip));
        st = GRAPHDATA(wip,xstripes);
        for (i=1; i<st; i++) 
        {
            MOVE(wip->border+i*WID/st, wip->border);
            LINE(wip->border+i*WID/st, wip->border+HID);
        }
        st = GRAPHDATA(wip,ystripes);
        for (i=1; i<st; i++) 
        {
            MOVE(wip->border,     wip->border+i*HID/st);
            LINE(wip->border+WID, wip->border+i*HID/st);
        }

        for (n=0; n<MAXSETS; n++)
        {
            gds = &GRAPHSET(wip,n);
            m = gds->last - gds->offset;

            if (!gds->values || m==0) continue;

            GpiSetColor(hps, vipColors[gds->color]);
            st = gds->size;
            pts = gds->values;
            min = gds->min;
            range = gds->max - min;
            j=gds->offset;

            if (m<0) m+=st;

            MOVE(wip->border, wip->border+((pts[j]-min)*HID)/range);
            for (i=1; i<=m;i++)
            {
                if (++j>=st) j=0;
                LINE(wip->border+(i*WID)/(st-1), wip->border+((pts[j]-min)*HID)/range);
            }
            if (gds->label)
            {
                gds->label->font = wip->font;
                if (gds->last>=0)
                    sprintf(graphBuf,"%s: %ld", gds->legend, pts[gds->last]);
                else
                    sprintf(graphBuf,"%s: 0", gds->legend);
                VipSetTitle(gds->label, graphBuf);
                WinSendMsg(gds->label->win, UM_SCALE,
                    (MPARAM)TRUE, NULL);
            }
        }
    }
    else
    {
        for (n=0; n<MAXSETS; n++)
        {
            gds = &GRAPHSET(wip,n);
            m=gds->last-gds->offset;

            if (!gds->values || m==0) continue;
            GpiSetColor(hps, vipColors[gds->color]);
            st = gds->size;
            pts = gds->values;
            min = gds->min;
            range = gds->max - min;
            j=gds->last-1;
            i=j+gds->offset;

            if (m<0) m+=st;

            MOVE(wip->border+(i*WID)/(st-1), wip->border+((pts[j]-min)*HID)/range);
            j++;
            i++;
            LINE(wip->border+(i*WID)/(st-1), wip->border+((pts[j]-min)*HID)/range);
            if (gds->label)
            {
                if (gds->last>=0)
                    sprintf(graphBuf,"%s: %ld", gds->legend, pts[gds->last]);
                else
                    sprintf(graphBuf,"%s: 0", gds->legend);
                VipSetTitle(gds->label, graphBuf);
                WinSendMsg(gds->label->win, UM_SCALE,
                    (MPARAM)TRUE, NULL);
            }
        }
    }
}
