/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       vipinput.c
**     SYSTEM   NAME:       VIP
**     ORIGINAL AUTHOR(S):  Alfred Kayser
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       1992/5/29
**
** DESCRIPTION: Static Input Module.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#include "vipinc.h"
#include <ctype.h>
#include <malloc.h>

PRIVAT LONG InputHandler(VIPINFO *wip, USHORT msg, MPARAM mp1, MPARAM mp2);
PRIVAT void InputUpdate(VIPINFO *wip, HPS hps, BOOLEAN all);
PRIVAT BOOLEAN InputKey(VIPINFO *wip, VOID *arg, int keyCode);
PRIVAT BOOLEAN InputAdjust(VIPINFO *wip, SWP *pswp);
PRIVAT BOOLEAN Insert(VIPINFO *wip, int key);


/**************************************************************
** NAME:        VipOpenInput                              [API]
** SYNOPSIS:    VIPINFO *VipOpenInput(VIPINFO *parent,
**                  int x, int y, int w, int h)
** DESCRIPTION: Opens a input window. This window can be moved
**              by dragging it. When <w> or <h> the width
**              resp. height are resized to fit the title
**              exactly.
** RETURNS:     void
**************************************************************/
EXPORT VIPINFO *
VipOpenInput(VIPINFO *parent, int x, int y, int w, int h)
{
    VIPINFO *wip;
    
    /* Open a simple window and change its defaults, and behaviour */
    if (!parent)
    {
        VipError(VIPERR_PARENT,"VipOpenInput must have a parent window");
        return NULL;
    }
    if (!(wip = VipOpenSimple(parent, x,y,w,h)))
        return NULL;
    wip->update=InputUpdate;
    wip->handler=InputHandler;
    wip->adjust=InputAdjust;
    wip->border=2;
    wip->btype=VIP_DEPTH;
    wip->active=0;
    wip->title=NULL;
    VipSetBackground(wip, VIP_WHITE);
    VipSetKeyHandler(wip, InputKey, NULL);
    return wip;
}


/**************************************************************
** NAME:        VipSetInputBuffer                         [API]
** SYNOPSIS:    void
**              VipSetInputBuffer(wip, char *buf, int len)
** DESCRIPTION: Sets the input buffer of the input window.
**              <buf> points to the buffer space.
**              <len> is the nr. of bytes in <buf>.
** EXAMPLE:     char buf[200];
**              VipSetInputBuffer(wip, buf, sizeof(buf));
** RETURNS:     void
**************************************************************/
EXPORT void
VipSetInputBuffer(VIPINFO *wip, char *buf, int len)
{
    VIPTEST(wip, return);
    wip->title=buf;
    wip->active=len;
}


PRIVAT BOOLEAN
InputKey(VIPINFO *wip, VOID *arg, int keyCode)
{
    BOOLEAN changed=FALSE;
    int keyFlags=VIPKEY_FLAGS(keyCode);

    if (keyFlags&VIP_KEYRELEASE) return TRUE;  /* Don't act on key releases */

    keyCode=VIPKEY_CODE(keyCode);

    if (keyCode&VIP_KEYSPECIAL)
    {
        switch(keyCode)
        {
        case VIPKEY_DELETE:
        case VIPKEY_BACKSPACE:
            if (wip->title)
            {
                wip->title[strlen(wip->title)-1]='\0';
                changed=TRUE;
            }
            break;
        case VIPKEY_SPACE:
            changed=Insert(wip, ' ');
            break;
        }
    }
    else
        changed=Insert(wip, keyCode);
    if (changed) VipUpdate(wip, 0);
    return changed;
}


PRIVAT BOOLEAN
Insert(VIPINFO *wip, int key)
{
    if (wip->title && wip->active==0)
        wip->active=strlen(wip->title)+1;

    if (wip->title)
    {
        int l=strlen(wip->title);
        if (l+2>wip->active)
            VipBell();
        else
        {
            wip->title[l]=(char)key;
            wip->title[l+1]='\0';
        }
    }
    else
        VipBell();
    return TRUE;
}


LONG
InputHandler(VIPINFO *wip, USHORT msg, MPARAM mp1, MPARAM mp2)
{
	switch(msg)
	{
	case WM_MOUSEMOVE:
#ifndef OS2EMX
        WinSetPointer(HWND_DESKTOP,
            WinQuerySysPointer(HWND_DESKTOP, SPTR_TEXT, FALSE));
#endif
        if (wip->win)
            WinSetFocus(HWND_DESKTOP, wip->win); 
        return TRUE;
	}
	return VipSimpleHandler(wip, msg, mp1, mp2);
}

PRIVAT void
InputUpdate(VIPINFO *wip, HPS hps, BOOLEAN all)
{
    RECTL rect;
    USHORT flags=DT_VCENTER|DT_CENTER;

    if (all)
        VipBorder(wip, hps);
    else
        flags |= DT_ERASERECT;
    if (wip->title)
    {
        rect.yBottom = rect.xLeft=wip->border;
        rect.xRight = wip->cx - wip->border;
        rect.yTop = wip->cy - wip->border;
        VipUseFont(hps, wip->font);
        WinDrawText(hps, -1, wip->title, &rect,
            FOREGROUND(wip), BACKGROUND(wip), flags);
        VipReleaseFont(hps, wip->font);
    }
}


PRIVAT BOOLEAN
InputAdjust(VIPINFO *wip, SWP *pswp)
{
    RECTL tr;
    HPS hps;
    BOOLEAN changed=FALSE;
    
    if ((wip->w==0) && wip->title)
    {
        tr.xLeft=tr.yBottom=0;
        tr.xRight=tr.yTop=1000;
        hps=WinGetPS(wip->win);
        VipUseFont(hps, wip->font);
        WinDrawText(hps, -1, wip->title, &tr, 0L,0L,
            DT_LEFT|DT_BOTTOM|DT_QUERYEXTENT|DT_TEXTATTRS);
        VipReleaseFont(hps, wip->font);
        WinReleasePS(hps);

        /* Reserver space for border and margin */
        tr.xRight+=2+2*wip->border;

        /* We allow the input to be slightly bigger */
        /* So only when tr is much bigger then current size, increase size */
        if (tr.xRight<pswp->cx-20 || tr.xRight>pswp->cx)
        {
            pswp->cx=(SHORT)tr.xRight;
            changed=TRUE;
        }
    }
    if (wip->h==0)
    {
        int fh=VipQueryFontHeight(wip->font);
        if (pswp->cy!=fh)
        {
            pswp->cy=fh;
            changed=TRUE;
        }
    }
    return changed;
}
