/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       viplabel.c
**     SYSTEM   NAME:       VIP
**     ORIGINAL AUTHOR(S):  Alfred Kayser
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       1992/5/29
**
** DESCRIPTION: Static Label Module.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#include "vipinc.h"

EXPORT LONG VipLabelHandler(VIPINFO *wip, USHORT msg, MPARAM mp1, MPARAM mp2);
PRIVAT void LabelUpdate(VIPINFO *wip, HPS hps, BOOLEAN all);
PRIVAT void LabelMove(VIPINFO *win, LONG x, LONG y);
PRIVAT BOOLEAN LabelAdjust(VIPINFO *wip, SWP *pswp);


/**************************************************************
** NAME:        VipOpenLabel                              [API]
** SYNOPSIS:    VIPINFO *VipOpenLabel(VIPINFO *parent,
**                  int x, int y, int w, int h)
** DESCRIPTION: Opens a label window. This window can be moved
**              by dragging it. When <w> or <h> the width
**              resp. height are resized to fit the title
**              exactly.
** RETURNS:     void
**************************************************************/
VIPINFO *
VipOpenLabel(VIPINFO *parent, int x, int y, int w, int h)
{
    VIPINFO *wip;
    
    /* Open a simple window and change its defaults, and behaviour */
    if (!parent)
    {
        VipError(VIPERR_PARENT,"VipOpenLabel must have a parent window");
        return NULL;
    }
    if (!(wip = VipOpenSimple(parent, x,y,w,h)))
        return NULL;
    wip->update=LabelUpdate;
    wip->handler=VipLabelHandler;
    wip->adjust=LabelAdjust;
    wip->border=1;
    wip->btype=VIP_RISE;
    return wip;
}


EXPORT LONG
VipLabelHandler(VIPINFO *wip, USHORT msg, MPARAM mp1, MPARAM mp2)
{
    static SHORT sx, sy;
    static SWP swp;
    static POINTL pp;
    static BOOLEAN moved=FALSE, moving=FALSE;
    HPS hps;

	switch(msg)
	{
	case WM_BUTTON2DOWN:
        WinSetFocus(HWND_DESKTOP, wip->win); 
        WinSetCapture(HWND_DESKTOP, wip->win);
        sx = SHORT1FROMMP(mp1);  /* Position in menu window */
        sy = SHORT2FROMMP(mp1);
        WinQueryWindowPos(wip->win, &swp);
        moved=FALSE;
		moving=TRUE;
		return TRUE;

	case WM_MOUSEMOVE:
		if (moving)
        {
            hps=WinGetScreenPS(HWND_DESKTOP);
            if (moved)
            {
                GpiSetMix(hps, FM_INVERT);
                GpiMove(hps, &pp);
                pp.x -= swp.cx-1;
                pp.y -= swp.cy-1;
                GpiBox(hps,DRO_OUTLINE, &pp, 0, 0);
            }
            moved=TRUE;
            VipPullMouseMoves();

            pp.x = swp.x + (SHORT1FROMMP(mp1) - sx);
            pp.y = swp.y + (SHORT2FROMMP(mp1) - sy);
            if (wip->parent)
            {
                LONG b=wip->parent->border;
                if (pp.x<b) pp.x=b;
                if (pp.y<b) pp.y=b;
                if (pp.x+swp.cx>=wip->parent->cx - b) pp.x=wip->parent->cx - swp.cx - b;
                if (pp.y+swp.cy>=wip->parent->cy - b) pp.y=wip->parent->cy - swp.cy -b;
                WinMapWindowPoints(wip->parent->win, HWND_DESKTOP, &pp, 1);
            }
            GpiSetMix(hps, FM_INVERT);
            GpiMove(hps, &pp);
            pp.x += swp.cx-1;
            pp.y += swp.cy-1;
            GpiBox(hps,DRO_OUTLINE, &pp, 0, 0);

            WinReleasePS(hps);
        }
		break;

	case WM_BUTTON2UP:
        if (moved)
        {
            hps=WinGetScreenPS(HWND_DESKTOP);
            GpiMove(hps, &pp);
            pp.x -= swp.cx-1;
            pp.y -= swp.cy-1;
            GpiSetMix(hps, FM_INVERT);
            GpiBox(hps,DRO_OUTLINE, &pp, 0, 0);
            WinReleasePS(hps);
        }
		moved=moving=FALSE;
        LabelMove(wip, SHORT1FROMMP(mp1) - sx,
                       SHORT2FROMMP(mp1) - sy);
        WinSetCapture(HWND_DESKTOP, NULL);
		return TRUE;
	}
	return VipSimpleHandler(wip, msg, mp1, mp2);
}


PRIVAT void
LabelMove(VIPINFO *wip, LONG x, LONG y)
{
    SWP swp;

    WinQueryWindowPos(wip->win, &swp);
    swp.x += x;
    swp.y += y;
    if (wip->parent)
    {
        int w=(int)wip->parent->cx;
        int h=(int)wip->parent->cy;
        int b=wip->parent->border;
        if (swp.x<b) swp.x=b;
        if (swp.y<b) swp.y=b;
        if (swp.x + swp.cx>w-b) swp.x=w-swp.cx-b;
        if (swp.y + swp.cy>h-b) swp.y=h-swp.cy-b;
    }
    VipMoveWindow(wip, swp.x, swp.y);
}


PRIVAT void
LabelUpdate(VIPINFO *wip, HPS hps, BOOLEAN all)
{
    RECTL rect;
    USHORT flags=DT_VCENTER|DT_CENTER;

    if (all)
        VipBorder(wip, hps);
    else
        flags |= DT_ERASERECT;
    if (wip->title)
    {
        rect.yBottom = rect.xLeft=wip->border;
        rect.xRight = wip->cx - wip->border;
        rect.yTop = wip->cy - wip->border;
        VipUseFont(hps, wip->font);
        WinDrawText(hps, -1, wip->title, &rect,
            FOREGROUND(wip), BACKGROUND(wip), flags);
        VipReleaseFont(hps, wip->font);
    }
}


PRIVAT BOOLEAN
LabelAdjust(VIPINFO *wip, SWP *pswp)
{
    RECTL tr;
    HPS hps;
    
    if (wip->w!=0 && wip->h!=0) return FALSE;

    tr.xLeft=tr.yBottom=0;
    if (wip->title)
    {
        tr.xRight=tr.yTop=1000;
        hps=WinGetPS(wip->win);
        VipUseFont(hps, wip->font);
        WinDrawText(hps, -1, wip->title, &tr, 0L,0L,
            DT_LEFT|DT_BOTTOM|DT_QUERYEXTENT|DT_TEXTATTRS);
        VipReleaseFont(hps, wip->font);
        WinReleasePS(hps);

        /* Reserver space for border and margin */
        tr.xRight+=2+(wip->border*2);
        tr.yTop+=2*wip->border;
    }
    else
        tr.xRight=tr.yTop=10;

    /* We allow the label to be slightly bigger */
    /* So only when tr is much bigger then current size, increase size */
    if (tr.xRight>wip->cx    || tr.yTop>wip->cy ||
        tr.xRight<wip->cx-20 || tr.yTop<wip->cy-4)
    {
        if (wip->w==0) pswp->cx=(SHORT)tr.xRight;
        if (wip->h==0) pswp->cy=(SHORT)tr.yTop;
        return TRUE;
    }
    return FALSE;
}
