/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       viptimer.c
**     SYSTEM   NAME:       VIP
**     ORIGINAL AUTHOR(S):  Alfred Kayser
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       1992/5/29
**
** DESCRIPTION: Module with Timer Handling Functions
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#define INCL_DOSINFOSEG
#include <time.h>
#include "vipinc.h"

IMPORT HAB vipHab;

EXPORT VIPTIMER *vipTimers=NULL;

PRIVAT USHORT interval;
PRIVAT USHORT timerId=0;

PRIVAT VOID ChangeTimer(ULONG msecs);


/**************************************************************
** NAME:        VipGetMiSecs
** SYNOPSIS:    EXPORT ULONG VipGetMiSecs()
** DESCRIPTION: Returns the number of milliseconds since system
**              startup.
** RETURNS:     milliseconds since startup.
**************************************************************/
EXPORT ULONG VipGetMiSecs()
{
#ifdef OS2EMX
    ULONG t=clock()*(1000/CLOCKS_PER_SEC);
    return t;
#else
    static PGINFOSEG pgis;
    static SEL selGlobalSeg, selLocalSeg;
    static timerInit=FALSE;
    if (!timerInit)
    {
        timerInit=TRUE;
        DosGetInfoSeg(&selGlobalSeg, &selLocalSeg);
        pgis = MAKEPGINFOSEG(selGlobalSeg);
    }
    return pgis->msecs;
#endif
}


/**************************************************************
** NAME:        VipCreateTimer
** SYNOPSIS:    EXPORT VIPTIMER *VipCreateTimer
**                  (ULONG msecs, VIP_TIMEOUT fun, VOID *arg)
** DESCRIPTION: Creates a timer. The timer started at creation
**              time, and calls <fun> every <msecs>.
**              Synopsis of VIP_TIMEOUT:
**                void fun(VIPTIMER *timer, VOID *arg);
**              So <fun> is called with the <timer> pointer
**              and the <arg> pointer.
** RETURNS:     The pointer to the timer instance.
**              NULL, if timer creation failed.
**************************************************************/
EXPORT VIPTIMER *
VipCreateTimer(ULONG msecs, VIP_TIMEOUT fun, VOID *arg)
{
    VIPTIMER *pt;

    if (msecs<=100) msecs=100;
    if (!timerId)
    {
        interval=32000;
        ChangeTimer(msecs);
        if (!timerId)
        {
            DnpapMessage(DMC_ERROR,VIPERR_TIMER,
                "Failed to start timer (too many timers?)");
            return NULL;
        }
    }
    else
        ChangeTimer(msecs);

    pt = VipMalloc(sizeof(VIPTIMER));
    if (!pt) return NULL;
    pt->alarm    = VipGetMiSecs()+msecs;
    pt->msecs    = msecs;
    pt->next     = vipTimers;
    pt->callback = fun;
    pt->arg      = arg;

    vipTimers=pt;
    return pt;
}


/**************************************************************
** NAME:        VipDestroyTimer
** SYNOPSIS:    EXPORT BOOLEAN
**              VipDestroyTimer(VIPTIMER *timer)
** DESCRIPTION: Destroys the timer. The timer instance will be
**              freed.
** RETURNS:     TRUE, okay.
**              FALSE, timer not found in timer list.
**************************************************************/
EXPORT BOOLEAN
VipDestroyTimer(VIPTIMER *pt)
{
    VIPTIMER **pl;
    ULONG msecs=32000L;
    BOOLEAN found=FALSE;

    for (pl=&vipTimers;*pl;pl=&((*pl)->next))
    {
        if (*pl==pt)
        {
            *pl=pt->next;
            DnpapFree(pt);
            found=TRUE;
            break;
        }
        else
            if (pt->msecs<msecs)
                msecs=pt->msecs;
    }
    if (!found)
        DnpapMessage(DMC_ERROR, VIPERR_TIMER,
            "VipDestroyTimer: timer not found!");
    else
    {
        if (vipTimers==NULL)
        {
            WinStopTimer(vipHab, NULL, timerId);
            timerId=0;
        }
        else if (interval<msecs)
        {
            /* interval is smaller than smallest, so adjust interval */
            interval=32000;
            ChangeTimer(msecs);
        }
    }
    return found;
}


PRIVAT VOID
ChangeTimer(ULONG msecs)
{
    msecs/=2;
    while (interval>msecs) interval/=2;
    timerId = WinStartTimer(vipHab, NULL, timerId, interval);
}
