/*
 * local - Local Commands for Shifty Terminal Emulator
 */

#define	INCL_BASE

#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#include <process.h>
#include "shifty.h"

/*
 * Common Keyboard Functions
 */

/*
 * g e t k e y  - Get One Character from KBD
 */

USHORT	getkey(VOID)
{
    KBDKEYINFO	key ;
    USHORT	code ;

    KbdCharIn(&key, 0, 0) ;
    code = key.chChar     ;
    return(code) ;
}
	
/*
 * g e t l i n  - Get One Line
 */

#define	isdelim(x)	( ispunct((x)) || isspace((x)) )

UCHAR *getlin(UCHAR *lin, INT len)
{
    INT	    i    ;
    USHORT  code, row, col ;
    UCHAR   attr, cell[2]  ;

    VioGetCurPos(&row, &col, 0) ;
    attr = DispAttr[7] ;
    cell[0] = ' '   ;
    cell[1] = attr  ;
    i = strlen(lin) ;

    VioWrtCharStrAtt(lin, i, row, col, &attr, 0) ;
    VioWrtNCell(cell, (ScrCol - (col + i)), row, (col + i), 0) ;
    VioSetCurPos(row, (col + i), 0) ;

    while (i < len) {
	code = getkey() ;

	if ((code == 0x03) || (code == 0x07) || (code == 0x1b)) {
	    lin[i] = '\0' ;
	    return(NULL) ;
	} else if ((code == 0x0a) || (code == 0x0d)) {
	    lin[i] = '\0' ;
	    break ;
	} else if ((code == 0x08) || (code == 0x7f)) {
	    if (i > 0) {
		i -= 1 ; lin[i] = '\0' ;
	    }
	} else if ((code == 0x15) || (code == 0x18)) {
	    while (i > 0) {
		i -= 1 ; lin[i] = '\0' ;
	    }
	} else if (code == 0x17) {
	    if ((i > 0) && ispunct(lin[i - 1])) {
		while ((i > 0) && ispunct(lin[i - 1])) {
	            i -= 1 ; lin[i] = '\0' ;
		}
	    } else if ((i > 0) && isspace(lin[i - 1])) {
		while ((i > 0) && isspace(lin[i - 1])) {
	            i -= 1 ; lin[i] = '\0' ;
		}
	    }
	    while ((i > 0) && (! isdelim(lin[i - 1]))) {
		i -= 1 ; lin[i] = '\0' ;
	    }
	} else if (code < 0x20) {
	    ;
	} else if (code < 0x7f) {
	    lin[i++] = (UCHAR) code ; lin[i] = '\0' ;
	}
	VioWrtCharStrAtt(lin, i, row, col, &attr, 0) ;
	VioWrtNCell(cell, (ScrCol - (col + i)), row, (col + i), 0) ;
	VioSetCurPos(row, (col + i), 0) ;
    }
    return(lin) ;
}

/*
 * p r o m p t  -  prompt message on bottom line
 */

VOID	prompt(UCHAR *msg)
{
    USHORT  lin, col ;
    UCHAR   attr, cell[2] ;
	
    attr = DispAttr[7] ;
    cell[0] = ' '  ;
    cell[1] = attr ;

    lin = ScrLin - 1  ;
    col = strlen(msg) ;
	
    VioWrtCharStrAtt(msg, col, lin, 0, &attr, 0) ;
    VioWrtNCell(cell, (ScrCol - col), lin, col, 0)   ;
    VioSetCurPos(lin, col, 0) ;
}

/*
 * k e y w a i t - wait for any key
 */

VOID    keywait(VOID)
{
    KBDKEYINFO	key ;

    KbdCharIn(&key, 0, 0) ;
}

/*
 * k e y c h e c k
 *	Check ATTN key
 */

BOOL    keycheck(VOID)
{
    KBDKEYINFO	key   ;
	
    KbdCharIn(&key, 1, 0) ;
	
    if (key.fbStatus == 0) {
	return(FALSE) ;
    } else if (key.chChar != KeyAttn) {
	return(FALSE) ;
    } else {
	return(TRUE) ;	
    }
}

/*
 * Wildcard Expanding
 */

#define	MAXARG	1024

USHORT	Xargc = 0      ;
UCHAR	*Xargv[MAXARG] ;

/*
 * a r g i n i t - setup argument parsing
 */

VOID	arginit(VOID)
{
    int	i ;
	
    for (i = 0 ; i < MAXARG ; i++) {
	if (Xargv[i]) {
	    free(Xargv[i]) ;
	}
	Xargv[i] = NULL ;
    }
    Xargc = 0 ;
}

/*
 * a r g e x p a n d  -  expand wildcard (if any)
 */

INT     argexpand(UCHAR *name)
{
    APIRET      stat ;
    ULONG       count, found ;
    HDIR	handle = -1     ;
    ULONG	attrib = 0x0017 ;
    FILEFINDBUF3    buff ;
    ULONG	    leng = sizeof(FILEFINDBUF3) ;

    /*
     * Find matching files
     */
	
    found = 0 ; count = 1 ;
	
    if ((stat = DosFindFirst(name, &handle, attrib, &buff, leng, &count, 1)) != 0) {
        return(0) ;
    }
    while ((stat == 0) && (Xargc < MAXARG)) {
        if (buff.achName[0] != '.') {	    
            Xargv[Xargc++] = strdup(buff.achName) ;
            found += 1 ;
        }
	count = 1  ;
	stat = DosFindNext(handle, &buff, leng, &count) ;
    }
    DosFindClose(handle) ;
    return(found) ;
}

/*
 * g e t C u r D i r  - get current working directory
 */

INT 	getCurDir(UCHAR *buff, USHORT leng)
{
    APIRET  stat ;
    ULONG   len  ;
    ULONG   dnum ;
    ULONG   dmap ;
	
    if ((stat = DosQueryCurrentDisk(&dnum, &dmap)) != 0) {
	buff[0] = '\0' ;
	return(0) ;
    }
    buff[0] = (UCHAR) (dnum - 1 + 'A') ;
    buff[1] = ':'  ;
    buff[2] = '\\' ;
    len = leng - 3 ;

    if ((stat = DosQueryCurrentDir(0, &buff[3], &len)) != 0) {
	return(0) ;
    }
    buff[3 + len] = '\0' ;
    strlwr(buff) ;
    return(1)    ;
}

/*
 * Local Functions
 */

/*
 * s h o w H e l p  -  Help for emulator
 */

static	UCHAR	*helpmsgs[] = {
"                                                                 " ,
"      ?  This Message               C  Close the connection      " ,
"      B  Send a BREAK signal        0  Send NULL code            " ,
"      U  Upload File                D  Download File             " ,
"      %  Change Directory           L  List Directory            " ,
"      !  Execute a Command          P  Invoke Shell              " ,
"      S  Save current screen (S - Append : s - Update)           " ,
"      Typing Attention Key will send it to the host              " ,
"                                                                 " ,
NULL
} ;

static	UCHAR	help_prompt[] = " Command> " ;

USHORT	showHelp(UCHAR *title)
{
    USHORT  i, len, lin, col ;
    UCHAR   attr ;
    USHORT  key  ;

    scrSave() ;

    attr = DispAttr[7] ;

    len = strlen(title) ;
    col = (ScrCol - len) / 2 ;
    VioWrtCharStrAtt(title, len, 0, col, &attr, 0) ;
	
    for (i = 0, lin = 1 ; helpmsgs[i] ; i++, lin++) {
	len = strlen(helpmsgs[i]) ;
	col = (ScrCol - len) / 2 ;
	VioWrtCharStrAtt(helpmsgs[i], len, lin, col, &attr, 0) ;
    }

    prompt(help_prompt) ;

    while ((key = getkey()) == '?') {
	;
    }	
    scrRest() ;
	
    return(key) ;
}

/*
 * s h o w S t a t u s
 *      show communication status
 */

VOID    showStatus(VOID)
{
    scrSave() ;

    if (comDevice != NULL) {
	(*comDevice->comStat) () ;
    }
    keywait() ;

    scrRest() ;
}

/*
 * d i r C h a n g e  - change working drive and directory
 */

VOID    dirChange(VOID)
{
    APIRET  stat ;
    ULONG   dnum ;
    UCHAR   *path, *getlin() ;

static	UCHAR	cdir[128]  ;
static	UCHAR	ndir[128]  ;

    /*
     * Get Current Directory
     */
	
    getCurDir(cdir, 128)  ;
    strcpy(ndir, cdir) ;

    /*
     * Where to go
     */
	
    scrSave() ;

    prompt("Directory to ") ;

    if (getlin(ndir, 128) == NULL) {
	scrRest() ;
	return ;
    }
    if (stricmp(ndir, cdir) == 0) {
	scrRest() ;
	return ;
    }
	
    /*
     * Change current drive if specified
     */

    path = ndir ;
    if ((strlen(ndir) >= 2) && (ndir[1] == ':')) {
	dnum = toupper(ndir[0]) - 'A' + 1 ;
	if ((stat = DosSetDefaultDisk(dnum)) != 0) {
	    sprintf(cdir, "failed %d", stat) ;
	    prompt(cdir) ;
	    keywait() ;
	    scrRest() ;
	    return ;
    	}
	path = &ndir[2] ;
    }
	
    /*
     * Change working directory if specified
     */

    if (strlen(path) > 0) {
	if ((stat = DosSetCurrentDir(path)) != 0) {
	    sprintf(cdir, "failed %d <%s>", stat, ndir) ;
	    prompt(cdir) ;
	    keywait() ;
	    scrRest() ;
	    return ;
	}
    }
    scrRest() ;
}

/*
 * d i r L i s t i n g - list directory contents
 */

static	UCHAR	listfiller[] = "               " ;

static  int     listsort(const void *p1, const void *p2)
{
    return(stricmp(*(char **) p1, *(char **) p2)) ;
}

VOID    dirListing(VOID)
{
    INT     i, listed ;
    UCHAR   mesg[128] ;
    UCHAR   path[128] ;
    FILESTATUS3 infoBuff ;
    ULONG       infoLeng ;

    if (getCurDir(path, 128) == 0) {
	return ;
    }
    sprintf(mesg, "Contents of %s", path) ;

    scrSave() ;

    /*
     * list up files
     */

    arginit() ;

    if (argexpand("*.*") == 0) {
	strcat(mesg, " empty!!") ;
	prompt(mesg) ;
	keywait()    ;
	scrRest()    ;
	return ;
    }

    qsort(Xargv, Xargc, sizeof(UCHAR *), listsort) ;

    /*
     * display file list
     */

    prompt(mesg) ;
    printf("\n") ;

    for (i = 0, listed = 0 ; i < Xargc ; i++) {

	if (((listed + 1) * strlen(listfiller)) >= ScrCol) {
	    printf("\n") ;
	    listed = 0   ;
	}
	infoLeng = sizeof(infoBuff) ;
	if (DosQueryPathInfo(Xargv[i], 1, &infoBuff, infoLeng) != 0) {
	    continue ;
	} else if (infoBuff.attrFile & FILE_DIRECTORY) {
	    printf("%s/", Xargv[i]) ;
	    printf("%s",  &listfiller[strlen(Xargv[i]) + 1]) ;
	    listed += 1 ;
	} else {
	    printf("%s", Xargv[i]) ;
	    printf("%s",  &listfiller[strlen(Xargv[i])]) ;
	    listed += 1 ;
	}
    }
    if (listed != 0) {
	printf("\n") ;
    }
    arginit() ;
    keywait() ; 
    scrRest() ;
}

/*
 * e x e c C o m m a n d  - execute single command
 */

static	UCHAR	cmd_buff[128] ;

VOID    execCommand(VOID)
{
    USHORT  prty ;

    scrSave() ;
    prompt("Execute Command : ") ;
	
    if ((getlin(cmd_buff, 128) == NULL) || (*cmd_buff == '\0')) {
	scrRest() ;
	return ;
    }

    printf("\n") ; system(cmd_buff) ; printf("\n") ;

    keywait() ;

    scrRest() ;
}

/*
 * e x e c S h e l l  -  Start Local Command Interpreter
 */

VOID    execShell(VOID)
{
    UCHAR   *p ;
    USHORT  prty ;

    if ((p = getenv("SHELL")) != NULL) {
	;
    } else if ((p = getenv("COMSPEC")) != NULL) {
	;
    } else {
	p = "CMD.EXE" ;
    }

    scrSave() ;

    spawnlp(P_WAIT, p, p, NULL) ;

    scrRest() ;
}

/*
 * Supporting File Transfer Protocols
 */

void    kermitSend(void) ;
void    kermitRecv(void) ;
void    xmodemSend(void) ;
void    xmodemRecv(void) ;
void    rawSend(void) ;
void    rawRecv(void) ;

/*
 * u p L o a d i n g  -  select FTP protocol and start uploading
 */

VOID upLoading(VOID)
{
    USHORT  code ;

    scrSave() ;
	
    prompt("Upload with Kermit/Xmodem/Raw ?") ;
    code = getkey() ;

    scrRest()  ;

    switch (code) {
    case 'k' :
    case 'K' :
	kermitSend() ;
	break ;
    case 'x' :
    case 'X' :
	xmodemSend() ;
	break ;
    case 'r' :
    case 'R' :
	rawSend() ;
	break ;
    default  :
	break ;
    }
}

/*
 * d n L o a d i n g  -  select FTP protocol and start downloading
 */

VOID dnLoading(VOID)
{
    USHORT  code ;

    scrSave() ;
	
    prompt("Download with Kermit/Xmodem/Raw ?") ;
    code = getkey() ;
	
    scrRest() ;

    switch (code) {
    case 'k' :
    case 'K' :
	kermitRecv() ;
	break ;
    case 'x' :
    case 'X' :
	xmodemRecv() ;
	break ;
    case 'r' :
    case 'R' :
	rawRecv() ;
	break ;
    default  :
	break ;
    }
}

/*
 * s a v e S c r e e n  -  Save current screen to file (shifty.scr)
 */

static	char	save_name[] = "shifty.scr" ;
static	char	save_buff[256] ;

VOID	saveScreen(BOOL append)
{
    FILE    *fp ;
    USHORT  len ;
    int	    lin, col ;

    scrSave() ;

    if (append) {
	fp = fopen(save_name, "a") ;
    } else {
	fp = fopen(save_name, "w") ;
    }
    if (fp == NULL) {
	prompt("Cannot open save file") ;
	keywait() ;
	scrRest() ;
	return ;
    }

    for (lin = 0 ; lin < ScrLin ; lin++) {
	len = ScrCol ;
	VioReadCharStr(save_buff, &len, lin, 0, 0) ;
	save_buff[len] = '\0' ;
	for (col = len - 1 ; col >= 0 ; col--) {
            if (isspace(save_buff[col])) {
		save_buff[col] = '\0' ;
	    } else {
		break ;
	    }
	}
	fputs(save_buff, fp) ;
	fputs("\n", fp) ;
    }
    fclose(fp) ;

    scrRest() ;
}
