/*
 * vt100emu.c - VT100 emulation for Shifty Term.
 */

#define	INCL_BASE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <os2.h>
#include "shifty.h"
#include "screen.h"

/*
 * Exported VT100 Emulations
 */

VOID    emuReset(VOID) ;

VOID    dispSJIS(UCHAR *buff, INT len) ;
VOID    dispEUC(UCHAR *buff, INT len)  ;
VOID    dispJIS7(UCHAR *buff, INT len) ;
VOID    dispJIS8(UCHAR *buff, INT len) ;

VOID    keySJIS(USHORT ch, USHORT sc, USHORT stat) ;
VOID    keyEUC(USHORT ch, USHORT sc, USHORT stat)  ;
VOID    keyJIS7(USHORT ch, USHORT sc, USHORT stat) ;
VOID    keyJIS8(USHORT ch, USHORT sc, USHORT stat) ;

EMUREC  vt100sjis = {
    "VT100 SJIS",
    "VT100",
    emuReset,
    dispSJIS,
    keySJIS
} ;

EMUREC  vt100euc  = {
    "VT100 EUC",
    "VT100",
    emuReset,
    dispEUC,
    keyEUC
} ;

EMUREC  vt100jis7 = {
    "VT100 JIS7",
    "VT100",
    emuReset,
    dispJIS7,
    keyJIS7
} ;

EMUREC  vt100jis8 = {
    "VT100 JIS8",
    "VT100",
    emuReset,
    dispJIS8,
    keyJIS8
} ;

/*
 * Buffers for Key Emulation
 */

static  UCHAR   keyKI[] = { 0x1b, '$', '@', 0 } ;
static  UCHAR   keyKO[] = { 0x1b, '(', 'J', 0 } ;
static  UCHAR   keySI[] = { 0x0e, 0 } ;
static  UCHAR   keySO[] = { 0x0f, 0 } ;
 
static  UCHAR   keycsi[3] = { 0x1b, '[', 0 } ;
static  UCHAR   keyalt[3] = { 0x1b, 'O', 0 } ;
static  UCHAR   keybuf[6] = { 0, 0, 0, 0, 0, 0 } ;

VOID    emuExtendedKey(USHORT sc)
{
    switch(sc) {
    case 0x03 :     /* CTRL-@   */
        keybuf[0] = '\0' ;
	(*comDevice->comSend) (keybuf, 1) ;
	break ;
    case 0x3b :     /* PF1      */
        keyalt[2] = 'P' ;
	(*comDevice->comSend) (keyalt, 3) ;
	break ;
    case 0x3c :     /* PF2      */
        keyalt[2] = 'Q' ;
	(*comDevice->comSend) (keyalt, 3) ;
	break ;
    case 0x3d :     /* PF3      */
        keyalt[2] = 'R' ;
	(*comDevice->comSend) (keyalt, 3) ;
	break ;
    case 0x3e :     /* PF4      */
        keyalt[2] = 'S' ;
	(*comDevice->comSend) (keyalt, 3) ;
	break ;
    case 0x47 :     /* HOME     */
        keycsi[2] = 'H' ;
	(*comDevice->comSend) (keycsi, 3) ;
	break ;
    case 0x48 :     /* Arrow UP */
        keyalt[2] = 'A' ;
	(*comDevice->comSend) (keyalt, 3) ;
	break ;
    case 0x4b :     /* Arrow LF */
        keyalt[2] = 'D' ;
	(*comDevice->comSend) (keyalt, 3) ;
	break ;
    case 0x4d :     /* Arrow RT */
        keyalt[2] = 'C' ;
	(*comDevice->comSend) (keyalt, 3) ;
	break ;
    case 0x50 :     /* Arrow DN */
        keyalt[2] = 'B' ;
	(*comDevice->comSend) (keyalt, 3) ;
	break ;
    case 0x53 :     /* DELETE   */
        keybuf[0] = (UCHAR) 0x7f ;
	(*comDevice->comSend) (keybuf, 1) ;
	break ;
    default :
        break ;
    }
}

/*
 * Status for Emulation
 */
 
#define	SQPLAIN	    0	/* not in control sequence	*/
#define	SQESCS	    1	/* prefixed by ESC character	*/
#define	SQESCM	    2	/* Multi-byte escape sequnce	*/
#define	SQCTRL	    3	/* control sequence		*/
#define	SQKANJI	    4	/* KANJI Output sequence	*/
#define SQKANA      5   /* KANA  Output sequence        */
#define	SQSHIFT	    6   /* SHIFTED to KANA (KBD-CTRL)   */

static	INT	CtrlStat = SQPLAIN ;
static	INT	LastCode = 0       ;

static  INT     KeyStat  = SQPLAIN ;
static	INT	KeyMode  = SQPLAIN ;
static  INT     KeyLast  = 0       ;

#define	CSROMAN     0   /* JIS Roman    */
#define CSKANJI     1   /* JIS Kanji    */

static  INT     CharSetGL = CSROMAN ;
static  INT     CharSetGR = CSROMAN ;
static  BOOL    CharShift = FALSE   ;

VOID    emuReset(VOID)
{
    CtrlStat = SQPLAIN ;
    LastCode = 0       ;
    KeyStat  = SQPLAIN ;
    KeyMode  = SQPLAIN ;
    KeyLast  = 0       ;

    CharSetGL = CSROMAN ;
    CharSetGR = CSROMAN ;
    CharShift = FALSE   ;
}

/*
 * Control Sequence Parsing Sunroutines
 */

static	void    c0ctrl(USHORT c) ;  /* C0 Control Char      */
static	void    c1ctrl(USHORT c) ;  /* C1 Control Char      */
static	void    seqesc(USHORT c) ;  /* Follow ESC           */
static	void    seqmlt(USHORT c) ;  /* Multi Byte ESC SEQ   */
static	void    seqctl(USHORT c) ;  /* Control SEQ          */
static	void    escsfp(USHORT c) ;  /* do ESC Fp Controls   */
static	void    escsfs(USHORT c) ;  /* do ESC Fs Controls   */
static	void    escmlt(USHORT c) ;  /* do MultiByte ESC     */
static	void    esccsi(USHORT c) ;  /* do CSI controls      */

/*
 * Kanji Conversions between JIS & SJIS
 */

VOID    jis2sjis(USHORT c1, USHORT c2, PUCHAR buff)
{
    if (c1 & 1) c2 += 0x1f         ;
    else        c2 += 0x7d         ; 
    if (c2 >= 0x7f) c2++           ;
    c1 = ((c1 - 0x21) >> 1) + 0x81 ;
    if (c1 > 0x9f) c1 += 0x40      ;
    buff[0] = (UCHAR) (c1 & 0xff)  ;
    buff[1] = (UCHAR) (c2 & 0xff)  ;
}

VOID    sjis2jis(USHORT c1, USHORT c2, PUCHAR buff)
{
    c1 -= (c1 >= 0xa0) ? 0xc1 : 0x81 ;
    if (c2 >= 0x9f) {
        c1 = c1 + c1 + 0x22 ;
        c2 -= 0x7e ;
    } else {
        c1 = c1 + c1 + 0x21 ;
        c2 -= (c2 <= 0x7e) ? 0x1f : 0x20 ;
    }
    buff[0] = (UCHAR) (c1 & 0xff) ;
    buff[1] = (UCHAR) (c2 & 0xff) ;
}

/*
 * SJIS Emulation  -  Parsing Input Data Stream Based on SHIFT-JIS.
 *
 * (1) KANJI code
 *	SHIFT-JIS code is mainly based on representation of the KANJI
 *	code, with 8 bit data stream.  Some codes are interpreted as
 *	first byte of the KANJI code, and follows KANJI second byte.
 *	KANJI second byte may be anything including 00..FF.
 *
 * (2) Control Characters
 *	In SHIFT-JIS code, only C0 effective in not KANJI prefixed status.
 *	All control sequences are encoded with C0 control character and
 *	normal ASCII (7bit) code.
 *
 * (3) Escape Sequence
 *	To expand device controls, ISO introduce some control
 *	sequences.  They are prefixed by ESC or CSI and followed
 *	by paratmeter, intermediate character, and terminal 
 *	character.  Following escape sequences are available.
 *
 * (4) 2 byte escape sequence
 *	ESC Fs sequence - (terminate with 60..7e)	
 *	ESC Fp sequence - (terminate with 30..3f)
 *	ESC Fe sequqnce - (terminate with 40..5f)
 *
 * (5) Multi-byte escape sequence
 *	ESC I ... F
 *		Here, I is intermediate character, in (20..2f),
 *		and F is terminator in (30..7f). 
 *
 * (6) Control Sequence
 *	CSI P1..Pn I1..In F or ESC [ P1..Pn I1..In F
 *		Here, Pi is parameter string in (30..3f).
 *		Ii is imtermediate character in (20..2f).
 *		F is the terminator in range (40..7f).  
 *		Function will determinated with I's and F.
 *
 */

VOID    dispSJIS(UCHAR *buff, INT len)
{
    INT	    i ;
    USHORT  c ;

    for (i = 0 ; i < len ; i++) {
        c = (USHORT) buff[i] ;
		
        /*
         * Meaning of all input character will depend on current
         * control sequence.
         */

        switch(CtrlStat) {

        case SQPLAIN :
            if ((c >= 0x20) && (c <= 0x7e)) {
	        doANK(c) ;
	    } else if ((c < 0x20) || (c == 0x7f)) {
	        c0ctrl(c) ;
	    } else if (c < 0x81) {
	        c0ctrl(0x00) ;	/* Same as NULL */
	    } else if (c <= 0xa0) {
		LastCode = c ;
		CtrlStat = SQKANJI ;
	    } else if (c <= 0xdf) {
	        doANK(c) ;
	    } else if (c < 0xfc) {
	        LastCode = c ;
		CtrlStat = SQKANJI ;
	    }
	    break ;

        case SQKANJI  :
	    doKANJI(LastCode, c) ;
	    CtrlStat = SQPLAIN ;
	    break ;

	case SQESCS  :		/* Prefixed by ESC		*/
	    seqesc(c) ;
	    break ;
	case SQESCM   :		/* In Multi-byte ESC Seq.	*/
	    seqmlt(c) ;
	    break ;
	case SQCTRL   :		/* Control sequence		*/
	    seqctl(c) ;
	    break ;
	default :
	    break ;
	}
    }
}

VOID    keySJIS(USHORT ch, USHORT sc, USHORT stat)
{
    /*
     * No conversion required for SJIS emulation
     */    
    if ((ch == 0x20) && ((stat & 0x0504) != 0)) {   /* CTRL-SPACE   */
        keybuf[0] = '\0' ;
	(*comDevice->comSend) (keybuf, 1) ;
	return ;
    }
    if ((ch != 0x00 && ch != 0xe0) || (sc == 0)) {  /* Normal Key   */
    	keybuf[0] = (UCHAR) ch ;
	(*comDevice->comSend) (keybuf, 1) ;
        return ;
    }
    emuExtendedKey(sc) ;
}

/*
 * EUC Emulation  -  Parsing Input Data Stream Based on EUC
 *
 * (1) KANJI code
 *      EUC code uses 0x80 OR'ed KANJI code. So range 0xA1-0xDE represent
 *      KANJI code.
 *
 * (2) Control Characters
 *	In EUC code, C0 and C1 are both effective.  Special handling is
 *      required for SS2 C1 character which represend 7 bit KANA code.
 *
 * (3) Escape Sequence
 *	Escape sequences are same as SHIST-JIS.
 */

VOID    dispEUC(UCHAR *buff, INT len)
{
    INT	    i ;
    USHORT  c ;
    UCHAR   kbuff[4] ;

    for (i = 0 ; i < len ; i++) {
        c = (USHORT) buff[i] ;
		
        /*
         * Meaning of all input character will depend on current
         * control sequence.
         */

        switch(CtrlStat) {

        case SQPLAIN :
            if ((c >= 0x20) && (c <= 0x7e)) {
	        doANK(c) ;
	    } else if ((c < 0x20) || (c == 0x7f)) {
	        c0ctrl(c) ;
	    } else if (c < 0x81) {
	        c0ctrl(0x00) ;	/* Same as NULL */
	    } else if (c == 0x8e) {
	        CtrlStat = SQKANA ;
	    } else if (c <= 0xa0) {
	        c1ctrl(c) ;
	    } else {
                LastCode = c ;
		CtrlStat = SQKANJI ;
	    }
	    break ;

        case SQKANJI :
            if (c >= 0xa0) {
	        jis2sjis((LastCode & 0x7f), (c & 0x7f), kbuff) ;
	        doKANJI((USHORT) kbuff[0], (USHORT) kbuff[1]) ;
	    } else {
	        doANK(' ') ; doANK(' ') ;
	    }
	    LastCode = 0 ;
	    CtrlStat = SQPLAIN ;
	    break ;
	    
        case SQKANA  :
	    if (c < 0x80) {
	        doANK(c + 0x80) ;
	    } else {
	        doANK(c) ;
	    }
	    CtrlStat = SQPLAIN ;
	    break ;
	    
	case SQESCS  :		/* Prefixed by ESC		*/
	    seqesc(c) ;
	    break ;
	case SQESCM   :		/* In Multi-byte ESC Seq.	*/
	    seqmlt(c) ;
	    break ;
	case SQCTRL   :		/* Control sequence		*/
	    seqctl(c) ;
	    break ;
	default :
	    break ;
	}
    }
}

VOID    keyEUC(USHORT ch, USHORT sc, USHORT stat)
{
    INT     len ;
    
    if ((ch == 0x20) && ((stat & 0x0504) != 0)) {   /* CTRL-SPACE   */
        keybuf[0] = '\0' ;
	(*comDevice->comSend) (keybuf, 1) ;
	return ;
    }
    if ((ch != 0x00 && ch != 0xe0) || (sc == 0)) {  /* Normal Key   */
    	if (KeyStat == SQKANJI) {
	    sjis2jis(KeyLast, ch, keybuf) ;
	    keybuf[0] |= 0x80 ; keybuf[1] |= 0x80 ; len = 2 ;
	    KeyLast = 0       ;
	    KeyStat = SQPLAIN ;
    	} else if (ch <= 0x1f) {
	    keybuf[0] = (UCHAR) ch ; len = 1 ;
	} else if (ch <= 0x7f) {
	    keybuf[0] = (UCHAR) ch ; len = 1 ;
	} else if (ch >= 0xa0 && ch <= 0xdf) {
	    keybuf[0] = (UCHAR) 0x8e ;          /* SS2  */
	    keybuf[1] = (UCHAR) ch   ;
	    len = 2 ;
        } else {
	    KeyLast = (UCHAR) ch ; len = 0 ;
	    KeyStat = SQKANJI ;
        }
	if (len > 0) {		
	    (*comDevice->comSend) (keybuf, len) ;
	}
        return ;
    }
    emuExtendedKey(sc) ;
}

/*
 * JIS7  -  Parsing Input Data Stream Based on JIS 7 bit Kanji Code
 *
 * (1) KANJI code
 *      JIS 7bit code uses normal ASCII, KANA and KANJI with special
 *      introducers. These introducer is based on Control Sequences.
 *
 * (2) Control Characters
 *      In JIS 7 bit code, only C0 is effective. All recv'ed code
 *      should be masked with 0x7f.
 *
 * (3) Escape Sequence
 *	Escape sequence is generally same as SJIS or EUC, but it includes
 *      code designation sequence, which control ASCII, KANA, KANJI
 *      transition.
 */

VOID    dispJIS7(UCHAR *buff, INT len)
{
    INT	    i ;
    USHORT  c ;
    UCHAR   kbuff[4] ;

    for (i = 0 ; i < len ; i++) {
        c = (USHORT) buff[i] & 0x7f ;
		
        /*
         * Meaning of all input character will depend on current
         * control sequence.
         */

        switch(CtrlStat) {

        case SQPLAIN :
            if (c < 0x20 || c > 0x7e) {
	        c0ctrl(c) ;
	    } else if (CharSetGL == CSROMAN) {
	        if (CharShift == FALSE) {
		    doANK(c) ;
		} else {
		    doANK(c + 0x80) ;
		}
	    } else if (CharSetGL == CSKANJI) {
	        if (LastCode == 0) {
                    LastCode = c ;
                } else {
		    jis2sjis(LastCode, c, kbuff) ;
		    doKANJI((USHORT) kbuff[0], (USHORT) kbuff[1]) ;
		    LastCode = 0 ;
		}
	    }
    	    break ;

	case SQESCS  :		/* Prefixed by ESC		*/
	    seqesc(c) ;
	    break ;
	case SQESCM   :		/* In Multi-byte ESC Seq.	*/
	    seqmlt(c) ;
	    break ;
	case SQCTRL   :		/* Control sequence		*/
	    seqctl(c) ;
	    break ;
	default :
	    break ;
	}
    }
}

VOID    keyJIS7(USHORT ch, USHORT sc, USHORT stat)
{
    INT     len ;
    
    if ((ch == 0x20) && ((stat & 0x0504) != 0)) {   /* CTRL-SPACE   */
        keybuf[0] = '\0' ;
	(*comDevice->comSend) (keybuf, 1) ;
	return ;
    }
    if ((ch != 0x00 && ch != 0xe0) || (sc == 0)) {  /* Normal Key   */
    	if (KeyStat == SQKANJI) {
	    if (KeyMode == SQKANA) {
	        (*comDevice->comSend) (keySO, strlen(keySO)) ;
	    }
	    if (KeyMode != SQKANJI) {
	        (*comDevice->comSend) (keyKI, strlen(keyKI)) ;
	    }
	    KeyMode = SQKANJI ;
	    sjis2jis(KeyLast, ch, keybuf) ;
            len = 2 ;
	    KeyLast = 0 ;
	    KeyStat = SQPLAIN ;
	} else if (ch <= 0x1f || ch == 0x7f) {
	    if (KeyMode == SQKANA) {
	        (*comDevice->comSend) (keySO, strlen(keySO)) ;
	    } else if (KeyMode == SQKANJI) {
	        (*comDevice->comSend) (keyKO, strlen(keyKO)) ;
	    }
	    KeyMode = SQPLAIN ;
	    keybuf[0] = (UCHAR) ch ; len = 1 ;
	} else if (ch <= 0x7e) {
	    if (KeyMode == SQKANA) {
	        (*comDevice->comSend) (keySO, strlen(keySO)) ;
	    } else if (KeyMode == SQKANJI) {
	        (*comDevice->comSend) (keyKO, strlen(keyKO)) ;
	    }
            KeyMode = SQPLAIN ;
            keybuf[0] = (UCHAR) ch ; len = 1 ;
	} else if ((ch >= 0xa0) && (ch <= 0xdf)) {
	    if (KeyMode == SQKANJI)  {
	        (*comDevice->comSend) (keyKO, strlen(keyKO)) ;
	    }
	    if (KeyMode != SQKANA) {
	        (*comDevice->comSend) (keySI, strlen(keySI)) ;
	    }
	    KeyMode = SQKANA ;
	    keybuf[0] = (UCHAR) (ch - 0x80) ; len = 1 ;
        } else {
	    KeyLast = ch ; len = 0 ;
	    KeyStat = SQKANJI ;
        }
	if (len > 0) {		
	    (*comDevice->comSend) (keybuf, len) ;
	}
        return ;
    }
    emuExtendedKey(sc) ;
}

/*
 * JIS8  -  Parsing Input Data Stream Based on JIS 8 bit Kanji Code
 *
 * (1) KANJI code
 *      JIS 8bit code uses normal ASCII, KANA and KANJI with special
 *      introducers. These introducer is based on Control Sequences.
 *      Code with 8bit on belongs KANA or KANJI depend on introducers.
 *
 * (2) Control Characters
 *      In JIS 8 bit code, C0/C1 are both effective. 
 *
 * (3) Escape Sequence
 *	Escape sequence is generally same as SJIS or EUC, but it includes
 *      code designation sequence, which control ASCII, KANA, KANJI
 *      transition.
 */

VOID    dispJIS8(UCHAR *buff, INT len)
{
    INT	    i ;
    USHORT  c ;
    UCHAR   kbuff[4] ;

    for (i = 0 ; i < len ; i++) {
        c = (USHORT) buff[i] ;
		
        /*
         * Meaning of all input character will depend on current
         * control sequence.
         */

        switch(CtrlStat) {

        case SQPLAIN :
	    if (c < 0x20 || c == 0x7f) {
	        c0ctrl(c) ;
	    } else if (c >= 0x80 && c < 0xa0) {
	        c1ctrl(c) ;
	    } else if (c >= 0x20 && c <= 0x7e) {
	        if (CharSetGL == CSROMAN) {
		    if (CharShift == FALSE) {
		        doANK(c) ;
		    } else {
		        doANK(c + 0x80) ;
		    }
		} else if (CharSetGL == CSKANJI) {
		    if (LastCode == 0) {
		        LastCode = c ;
		    } else {
		        jis2sjis(LastCode, c, kbuff) ;
		        doKANJI((USHORT) kbuff[0], (USHORT) kbuff[1]) ;
		        LastCode = 0 ;
		    }
		}
	    } else if (c >= 0xa0 && c <= 0xfe) {
	        if (CharSetGR == CSROMAN) {
		    if (CharShift == FALSE) {
		        doANK(c) ;
		    } else {
		        doANK(c - 0x80) ;
		    }
		} else if (CharSetGR == CSKANJI) {
		    if (LastCode == 0) {
		        LastCode = c ;
		    } else {
		        jis2sjis((LastCode & 0x7f), (c & 0x7f), kbuff) ;
		        doKANJI((USHORT) kbuff[0], (USHORT) kbuff[1]) ;
		        LastCode = 0 ;
		    }
		}
	    }
	    break ;

	case SQESCS  :		/* Prefixed by ESC		*/
	    seqesc(c) ;
	    break ;
	case SQESCM   :		/* In Multi-byte ESC Seq.	*/
	    seqmlt(c) ;
	    break ;
	case SQCTRL   :		/* Control sequence		*/
	    seqctl(c) ;
	    break ;
	default :
	    break ;
	}
    }
}

VOID    keyJIS8(USHORT ch, USHORT sc, USHORT stat)
{
    INT     len ;
    
    if ((ch == 0x20) && ((stat & 0x0504) != 0)) {   /* CTRL-SPACE   */
        keybuf[0] = '\0' ;
	(*comDevice->comSend) (keybuf, 1) ;
	return ;
    }
    if ((ch != 0x00 && ch != 0xe0) || (sc == 0)) {  /* Normal Key   */
    	if (KeyStat == SQKANJI) {
	    if (KeyMode == SQKANA) {
	        (*comDevice->comSend) (keySO, strlen(keySO)) ;
	    }
	    if (KeyMode != SQKANJI) {
	        (*comDevice->comSend) (keyKI, strlen(keyKI)) ;
	    }
	    KeyMode = SQKANJI ;
	    sjis2jis(KeyLast, ch, keybuf) ;
            len = 2 ;
	    KeyLast = 0 ;
	    KeyStat = SQPLAIN ;
	} else if (ch <= 0x1f || ch == 0x7f) {
	    keybuf[0] = (UCHAR) ch ; len = 1 ;
	} else if (ch <= 0x7e) {
	    if (KeyMode == SQSHIFT) {
	        (*comDevice->comSend) (keySO, strlen(keySO)) ;
	    } else if (KeyMode == SQKANJI) {
	        (*comDevice->comSend) (keyKO, strlen(keyKO)) ;
	    }
            KeyMode = SQPLAIN ;
            keybuf[0] = (UCHAR) ch ; len = 1 ;
	} else if ((ch >= 0xa0) && (ch <= 0xdf)) {
	    if (KeyMode == SQKANJI)  {
	        (*comDevice->comSend) (keyKO, strlen(keyKO)) ;
	    }
	    if (KeyMode != SQKANA) {
	        (*comDevice->comSend) (keySI, strlen(keySI)) ;
	    }
	    KeyMode = SQKANA ;
	    keybuf[0] = (UCHAR) (ch - 0x80) ; len = 1 ;
        } else {
	    KeyLast = ch ; len = 0 ;
	    KeyStat = SQKANJI ;
        }
	if (len > 0) {		
	    (*comDevice->comSend) (keybuf, len) ;
	}
        return ;
    }
    emuExtendedKey(sc) ;
}

/*
 * m u l t i n  -  process for starting Multi-byte Escape Sequence
 */

static	void    multin(USHORT start)
{
    INT	    i ;
    INT	    *p ;

    for (i = 0, p = InterV ; i < MAXPARM ; i++, p++) {
	*p = 0 ;
    }
    InterC = 0 ;
    InterF = 1 ;
    InterV[InterC++] = start ;	/* 'start' is the 1st I character   */
    CtrlStat = SQESCM ;
}

/*
 * c t r l i n  -  process for starting Control Sequence
 */

static	void ctrlin(void)
{
    INT	    i ;
    INT	    *p, *q ;

    for (i = 0, p = ParamV, q = InterV ; i < MAXPARM ; i++, p++, q++) {
	*p = 0 ;
	*q = 0 ;
    }

    ParamC = 0 ;
    ParamF = 0 ;
    ParamQ = 0 ;
    InterC = 0 ;
    InterF = 0 ;

    CtrlStat = SQCTRL ;
}

/*
 * c 0 c t r l  -  Process C0 Control Characters
 *	Effectives
 *		BEL   -  ring bell
 *		BS    -  non destructive back space
 *		HT    -  horizontal TAB (non-destructive)
 *		LF    -  Line Feed
 *		CR    -  Carrige Return
 */

static	void    c0ctrl(USHORT c)
{
    switch(c) {
	case 0x07 : doBEL() ; break ;
	case 0x08 : doBS()  ; break ;
	case 0x09 : doHT()  ; break ;
	case 0x0a : doLF()  ; break ;
	case 0x0d : doCR()  ; break ;
	case 0x0e : CharShift = TRUE  ; break ;
	case 0x0f : CharShift = FALSE ; break ;
	case 0x1b : CtrlStat = SQESCS ; break ;
	default   : break ;
    }

    if (CtrlStat != SQESCS) {
	CtrlStat = SQPLAIN ;
    }
}

/*
 * c 1 c t r l  -  Process C1 Control Characters
 *	Except SS2 for EUC 7 bit KANA prefix
 */

static	void    c1ctrl(USHORT c)
{
    c -= 0x40 ;
    switch(c) {
	case '[' : ctrlin()  ; break ;
	case 'D' : doINDEX() ; break ;
	case 'E' : doINDEX() ; break ;
	case 'M' : doREVIN() ; break ;
	default : break ;
    }
	
    if (CtrlStat != SQCTRL) {
	CtrlStat = SQPLAIN ;
    }
}

/*
 * s e q e s c  -  Process charcters after ESC
 */

static	void    seqesc(USHORT c)
{
    if (c < 0x20) {
	c0ctrl(c) ;
	return ;
    }

    /*
     * ESC followed by 20..2f introduces multi-byte escape sequence
     */
    
    if (c < 0x30) {
	multin(c) ;
	return ;
    }

    /*
     * ESC followed by 30..3f terminates ESC Fp sequence.
     *	It request function determinated with terminator.
     */

    if (c < 0x40) {
	escsfp(c) ;
	return ;
    }
	
    /*
     * ESC followed by 40..5f terminate ESCs Fe sequence.
     *	It request controls defined in C1 Control Characters.
     */

    if (c < 0x60) {
	c1ctrl(c + 0x40) ;
	return ;
    }
	
    /*
     * ESC followed by 60..7e terminates ESC Fs sequnce.
     *	It request function determinated with terminator.
     */

    if (c < 0x7f) {
	escsfs(c) ;
	return ;
    }

    /*
     * return to PLAIN sequence for undefined characters
     */

    CtrlStat = SQPLAIN ;
}

/*
 * s e q m l t  -  process characters in Multi-Byte Escape Sequence
 */

static	void    seqmlt(USHORT c)
{
    if (c < 0x20) {
	c0ctrl(c) ;
	return ;
    }

    /*
     * 20..2f is intermediate character
     */

    if (c < 0x30) {
	InterV[InterC] = c ;
	InterF = 1 ;
	if ((InterC += 1) >= MAXPARM) {
            CtrlStat = SQPLAIN ;
	}
	return ;
    }
	
    /*
     * Characters in range 30..7f terminate this sequence
     */

    if (c < 0x80) {
	escmlt(c) ;
	return ;
    }

    /*
     * Return to plain for undefined characters
     */

    CtrlStat = SQPLAIN ;
}

/*
 * s e q c t l  -  process characters in Control Sequnce
 */

static	void    seqctl(USHORT c)
{
    if (c < 0x20) {
	c0ctrl(c) ;
	return ;
    }

    /*
     * 20..2f is intermediate character
     */

    if (c < 0x30) {
	InterV[InterC] = c ;
	InterF = 1 ;
	if ((InterC += 1) >= MAXPARM) {
	    CtrlStat = SQPLAIN ;
	}
	return ;
    }

    /*
     * Characters in range 30..3f is paramater string
     */

    if (c < 0x40) {
	ParamF = 1 ;
	if (c <= 0x39) {
	    ParamV[ParamC] *= 10 ;
	    ParamV[ParamC] += (c - '0') ;
	} else if (c == 0x3b) {
	    if ((ParamC += 1) >= MAXPARM) {
		CtrlStat = SQPLAIN ;
	    }
	} else if (c == 0x3f) {
	    ParamQ = 1 ;
	}
	return ;
    }

    /*
     * Characters in range 40..7f is terminator
     */

    if (c < 0x80) {
	if (ParamF) {
	    if ((ParamC += 1) >= MAXPARM) {
		ParamC = MAXPARM - 1 ;
	    }
	}
	esccsi(c) ;
	return ;
    }
	
    /*
     * return to PLAIN for undefined characters
     */
	
    CtrlStat = SQPLAIN ;
}


/*
 * e s c s f p  -  Dispatch to ESC Fp Controls
 *	Here terminator is in range 30..3f
 */

static	void    escsfp(USHORT c)
{
    switch (c) {
	case '7' : doSAVE()    ; break ;
	case '8' : doRESTORE() ; break ;
	default  : break ;
    }
    CtrlStat = SQPLAIN ;
}

/*
 * e s c s f s  -  Dispatch to ESC Fs Controls
 *	Here, terminator is in range 60..7f
 */

static	void    escsfs(USHORT c)
{
    if (c == 'c') {
	doRESET() ;
    }
    CtrlStat = SQPLAIN ;
}

/*
 * e s c m l t  -  Dispatch to Multi-byte ESC controls
 */

static	void    escmlt(USHORT c)
{
    if (InterV[0]  == '$') {
        CharSetGL = CSKANJI ;
    } else if (InterV[0] == '(') {
        CharSetGL = CSROMAN ;
    }
    CtrlStat  = SQPLAIN ;
}

/*
 * e s c c s i  -  Dispatch to CSI Controls
 *	Here, terminator is in range 40..7f
 */

static	void    esccsi(USHORT c)
{
    switch(c) {
	case 'A' : doCURUP()   ; break ;
	case 'B' : doCURDN()   ; break ;
	case 'C' : doCURFW()   ; break ;
	case 'D' : doCURBK()   ; break ;
	case 'f' :
	case 'H' : doCURPOS()  ; break ;
	case 'J' : doERDSP()   ; break ;
	case 'K' : doERLIN()   ; break ;
	case 'L' : doINSERT()  ; break ;
	case 'M' : doDELETE()  ; break ;
	case 'S' : doSAVE()    ; break ;
	case 'h' : doRSTMD()   ; break ;
	case 'l' : doSETMD()   ; break ;
	case 'm' : doATTR()    ; break ;
	case 'n' : doREPST()   ; break ;
	case 'r' : doREGION()  ; break ;
	case 'u' : doRESTORE() ; break ;
	default  : break ;
    }
    CtrlStat = SQPLAIN ;
}
