/*
 * xmodem - supports XMODEM file transfer protocol
 */

#define	INCL_BASE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <os2.h>
#include "shifty.h"

/*
 * Control Characters
 */

#define	XMODEM_SOH	0x01
#define	XMODEM_EOT	0x04
#define	XMODEM_ACK	0x06
#define	XMODEM_NAK	0x15
#define	XMODEM_CAN	0x18

/*
 * Constants
 */

#define	XMODEM_BLK	128	/* XMODEM Block Size	*/
#define	XMODEM_RETRY	10	/* Max Retry		*/

#define	RetryOver(x)	(((x) += 1) >= XMODEM_RETRY)

/*
 * Global Variables (used for reduce memory space)
 */

static  FILE	*XmdmFile ;		/* I/O File Pointer		*/
static  long	XmdmBlock ;		/* Transfering Block Number	*/
static  int	XmdmRetry ;		/* Retry Counter		*/
static  char	XmdmBuff[XMODEM_BLK]  ;	/* I/O Buffer			*/
static  int	XmdmWait  = 10 ;	/* I/O wait multiplier		*/
static  char	XmdmStat[128]  ;

/*
 * c o m _ g e t _ c h a r  - get a character from comm. port
 */

static int com_get_char(int *ch, int wait)
{
    UCHAR   buf ;
    USHORT  bytes ;

    wait *= XmdmWait ;

    for (  ; wait > 0 ; wait--) {
	if ((bytes = queGet(&buf, 1)) > 0) {
	    *ch = buf ;
	    return(0) ;
	}
    }
    return(-1) ;
}

/*
 * c o m _ p u t _ c h a r  - put a character to comm. port
 */

static int com_put_char(int ch)
{
    UCHAR	buf ;

    buf = (UCHAR) (ch & 0xff) ;
    (*comDevice->comSend) (&buf, 1)  ;
    return(0) ;
}

/*
 * c o m _ f l u s h  -  flush out incoming characters
 */

static int com_flush(int tout)
{
    unsigned int    ch ;
    int		    i  ;

    for (i = 0 ; i < 100 ; i++) {
	if (com_get_char(&ch, tout)) {
	    break ;
	}
    }
    return(0) ;
}

/*
 * f a i l e d  - show why failed
 */

static void failed(char *s)
{
    strcat(XmdmStat, s) ;
    prompt(XmdmStat) ;
}

/*
 * p u t _ p a c k e t  -  send XMODEM data packet
 */

static int put_packet(long block, char *data)
{
    unsigned int    ch, chk ;
    int		    i   ;

    /*
     * send packet header
     */

    com_put_char(XMODEM_SOH) ;

    /*
     * send block number
     */
	
    ch = (UCHAR) (block & 0xff) ;
    com_put_char(ch) ;
    com_put_char(255 - ch) ;

    /*
     * send data part
     */

    chk = 0 ;

    for (i = 0 ; i < XMODEM_BLK ; i++) {
	ch = *data++ ;
	chk = ((chk + ch) & 0xff) ;
	com_put_char(ch) ;
    }

    /*
     * send checksum
     */

    ch = (chk & 0xff) ;
    com_put_char(ch)  ;

    return(0) ;
}

/*
 * g e t _ p a c k e t  -  receive XMODEM data packet
 */

static int get_packet(long *block, char *data, int *stat)
{
    unsigned int    ch, chk, pblk, nblk ;
    int		    i, retry ;

    /*
     * look for packet header
     */

    for (retry = 0 ; retry < XMODEM_RETRY ; retry++) {
	if (com_get_char(&ch, 10)) {
	    continue ;
	}
	if (ch == XMODEM_EOT) {
	    *stat = XMODEM_EOT ;
	    return(0) ;
	}
	if (ch == XMODEM_SOH) {
	    *stat = XMODEM_SOH ;
	    break ;
	}
    }
    if (retry >= XMODEM_RETRY) {
	failed("no packet header") ;
	return(-1) ;
    }
	
    /*
     * get block number
     */
	
    if (com_get_char(&ch, XMODEM_RETRY)) {
	failed("no block number") ;
	return(-1) ;
    }
    pblk = ch ;

    if (com_get_char(&ch, 1)) {
	failed("no block number") ;
	return(-1) ;
    }
    nblk = ch ;

    if ((pblk + nblk) != 0xff) {
	failed("invalid block number") ;
	com_flush(10) ;
	return(-1) ;
    }
	
    /*
     * receive data part
     */

    chk = 0 ;

    for (i = 0 ; i < XMODEM_BLK ; i++) {
	if (com_get_char(&ch, 1)) {
	    failed("timed-out on data receiving") ;
	    return(-1) ;
	}
	chk = ((chk + (USHORT) ch) & 0xff) ;
	*data++ = (char) ch ;
    }

    /*
     * receive checksum and compare
     */

    if (com_get_char(&ch, 1)) {
	failed("no block check") ;
	return(-1) ;
    }
    if (ch != chk) {
	failed("block check") ;
	return(-1) ;
    }

    *block = pblk ;
    return(0) ;
}

/*
 * XMODEM File Sending Protocol
 */

#define	FSEND_QUIT	0	/* quit sending		*/
#define	FSEND_ABORT	1	/* abort sending	*/
#define	FSEND_ERROR	2	/* raised an error	*/
#define	FSEND_INIT	3	/* wait start		*/
#define	FSEND_WAIT	4	/* wait acknowledge	*/
#define	FSEND_AGAIN	5	/* send block again	*/
#define	FSEND_NEXT	6	/* send next block	*/
#define	FSEND_LAST	7	/* send EOT and wait	*/

/*
 * f s e n d _ i n i t  -  wait start
 *	get NAK --> FSEND_NEXT,  start sending
 */

static int fsend_init(void)
{
    int	    c ;

    if (com_get_char(&c, 10)) {
	return((RetryOver(XmdmRetry)) ? FSEND_ERROR : FSEND_INIT) ;
    }
    if (c != XMODEM_NAK) {
	return((RetryOver(XmdmRetry)) ? FSEND_ERROR : FSEND_INIT) ;
    }
    XmdmRetry = 0 ;
    return(FSEND_NEXT) ;
}

/*
 * f s e n d _ w a i t  -  wait block acknowldge
 *	get ACK --> FSEND_NEXT,  send next block
 *	get NAK --> FSEND_AGAIN, send same block
 *	others  --> FSEND_ERROR, quit sending
 */

static int fsend_wait(void)
{
    int	    c ;

    if (com_get_char(&c, 10)) {
	return((RetryOver(XmdmRetry)) ? FSEND_ERROR : FSEND_WAIT) ;
    }
    if ((c != XMODEM_ACK) && (c != XMODEM_NAK)) {
	return((RetryOver(XmdmRetry)) ? FSEND_ERROR : FSEND_WAIT) ;
    }
    XmdmRetry = 0 ;
    return((c == XMODEM_ACK) ? FSEND_NEXT : FSEND_AGAIN) ;
}

/*
 * f s e n d _ a g a i n  -  send same block again
 *	send error --> FSEND_ERROR
 *	others     --> FSEND_WAIT
 */

static int fsend_again(void)
{
    if (put_packet(XmdmBlock, XmdmBuff)) {
	return(FSEND_ERROR) ;
    }
    return(FSEND_WAIT) ;
}

/*
 * f s e n d _ n e x t  -  send next block
 *	EOF        --> FSEND_LAST
 *	send error --> FSEND_ERROR
 *	others     --> FSEND_WAIT
 */

static int fsend_next(void)
{
    int	    cnt ;

    if ((cnt = fread(XmdmBuff, 1, XMODEM_BLK, XmdmFile)) <= 0) {
	return(FSEND_LAST) ;
    }
    while (cnt < XMODEM_BLK) {
	XmdmBuff[cnt++] = 0 ;
    }
    if (put_packet((XmdmBlock += 1), XmdmBuff)) {
	return(FSEND_ERROR) ;
    }
    return(FSEND_WAIT) ;
}

/*
 * f s e n d _ l a s t  -  send EOT and wait acknowledge
 */

static int fsend_last(void)
{
    int	    c ;

    com_put_char(XMODEM_EOT) ;

    if (com_get_char(&c, 10)) {
	return(RetryOver(XmdmRetry) ? FSEND_ERROR : FSEND_LAST) ;
    }
    if (c != XMODEM_ACK) {
	return(RetryOver(XmdmRetry) ? FSEND_ERROR : FSEND_LAST) ;
    }
    return(FSEND_QUIT) ;
}

/*
 * f s e n d  -  send a file
 */

static int fsend(char *name)
{
    int	    stat = FSEND_INIT ;

    if ((XmdmFile = fopen(name, "rb")) == NULL) {
	sprintf(XmdmStat, "failed to open %s\n", name) ;
	prompt(XmdmStat) ;
	return(FSEND_ABORT) ;
    }
    XmdmBlock = 0L ;
    XmdmRetry = 0  ;
    XmdmStat[0] = '\0' ;

    while ((stat != FSEND_QUIT) && (stat != FSEND_ABORT)) {

	if (keycheck()) {
	    stat = FSEND_ABORT ;
	    break ;
	}
	sprintf(XmdmStat, " Sending %s block #%ld, retry %d ",
					name, XmdmBlock, XmdmRetry) ;
	prompt(XmdmStat) ;

	switch(stat) {

	case FSEND_INIT :	/* wait start		*/
	    stat = fsend_init() ;
	    break ;
	case FSEND_WAIT :	/* wait acknowledge	*/
	    stat = fsend_wait() ;
	    break ;
	case FSEND_AGAIN :	/* send block again	*/
	    stat = fsend_again() ;
	    break ;
	case FSEND_NEXT :	/* send next block	*/
	    stat = fsend_next() ;
	    break ;
	case FSEND_LAST :	/* terminating....	*/
	    stat = fsend_last() ;
	    break ;
	default :
	    failed("retry over") ;
	    com_flush(1) ;
	    stat = FSEND_ABORT ;
	    break ;
	}
    }
    fclose(XmdmFile) ;
    return((stat == FSEND_QUIT) ? 0 : stat) ;
}

/*
 * XMODEM File Receiving Protocol
 */

#define	FRECV_QUIT	0	/* quit receiving	*/
#define	FRECV_ABORT	1	/* abort receiving	*/
#define	FRECV_ERROR	2	/* raised an error	*/
#define	FRECV_INIT	3	/* start receiving	*/
#define	FRECV_WAIT	4	/* receive packet	*/
#define	FRECV_AGAIN	5	/* NAK to sender	*/
#define	FRECV_NEXT	6	/* ACK to sender	*/
#define	FRECV_LAST	7	/* terminating..	*/

/*
 * f r e c v _ i n i t  -  start receiving
 *	send NAK and --> FRECV_WAIT
 */

static int frecv_init(void)
{
    if (com_put_char(XMODEM_NAK)) {
	return(FRECV_ERROR) ;
    }
    return(FRECV_WAIT) ;
}

/*
 * f r e c v _ w a i t  -  receive packet
 *	error   --> FRECV_AGAIN or FRECV_ERROR
 *	got SOH --> FRECV_NEXT, get next block
 *	got EOT --> FRECV_LAST, ACK and terminate
 */

static int frecv_wait(void)
{
    int	    type  ;
    long    block ;

    if (get_packet(&block, XmdmBuff, &type)) {
	return((RetryOver(XmdmRetry)) ? FRECV_ERROR : FRECV_AGAIN) ;
    }
    if (type == XMODEM_EOT) {
	return(FRECV_LAST) ;
    }
    if (block == XmdmBlock) {
	return(FRECV_NEXT) ;
    }
    if (block == ((XmdmBlock + 1) & 0xff)) {
	fwrite(XmdmBuff, 1, XMODEM_BLK, XmdmFile) ;
	XmdmBlock += 1 ;
	XmdmRetry = 0  ;
	return(FRECV_NEXT) ;
    }
    return((RetryOver(XmdmRetry)) ? FRECV_ERROR : FRECV_AGAIN) ;
}

/*
 * f r e c v _ a g a i n  -  NAK and wait again
 */

static int frecv_again(void)
{
    com_flush(1) ;
    com_put_char(XMODEM_NAK) ;
    return(FRECV_WAIT) ;
}

/*
 * f r e c v _ n e x t  -  ACK and wait next packet
 */

static int frecv_next(void)
{
    com_put_char(XMODEM_ACK) ;
    return(FRECV_WAIT) ;
}

/*
 * f r e c v _ l a s t  -  ACT to EOT and terminats
 */

static int frecv_last(void)
{
    com_flush(1) ;
    com_put_char(XMODEM_ACK) ;

    return(FRECV_QUIT) ;
}

/*
 * f r e c v  -  receive a file
 */

static int frecv(char *name)
{
    int	    stat = FRECV_INIT ;

    if ((XmdmFile = fopen(name, "wb")) == NULL) {
	sprintf(XmdmStat, "failed to open %s\n", name) ;
	prompt(XmdmStat) ;
	return(FRECV_ABORT) ;
    }
    XmdmBlock = 0L ;
    XmdmRetry = 0  ;
    XmdmStat[0] = '\0' ;

    while ((stat != FRECV_QUIT) && (stat != FRECV_ABORT)) {

	if (keycheck()) {
	    stat = FRECV_ABORT ;
	    break ;
    	}
	sprintf(XmdmStat, " Receiving %s block #%ld, retry %d ",
					name, XmdmBlock, XmdmRetry) ;
	prompt(XmdmStat) ;

	switch(stat) {

	case FRECV_INIT :	/* start receiving	*/
	    stat = frecv_init() ;
	    break ;
	case FRECV_WAIT :	/* receive packet	*/
	    stat = frecv_wait() ;
	    break ;
	case FRECV_AGAIN :	/* NAK to sender	*/
	    stat = frecv_again() ;
	    break ;
	case FRECV_NEXT :	/* ACK to sender	*/
	    stat = frecv_next() ;
	    break ;
	case FRECV_LAST :	/* terminating....	*/
	    stat = frecv_last() ;
	    break ;
	default :
	    failed("retry over") ;
	    com_flush(1) ;
	    stat = FRECV_ABORT ;
	    break ;
	}
    }
    fclose(XmdmFile) ;
    return((stat == FRECV_QUIT) ? 0 : stat) ;
}

/*
 * x m o d e m S e n d  /  x m o d e m R e c v
 *
 * 	Start XMODEM file transfer
 */

static	UCHAR	filename[64] ;

void    xmodemSend(void)
{
    USHORT  stat ;
    UCHAR   *fname, *getlin() ;

    scrSave() ;

    prompt("XMODEM Send File Name ? ") ;

    if ((fname = getlin(filename, 64)) == NULL) {
	scrRest() ;
	return ;
    }
    if (*filename == '\0') {
    	scrRest() ;
    	return ;
    }
	
    if ((stat = fsend(fname)) != 0) {
	DosBeep(440, 50) ;
	strcat(XmdmStat, " failed ") ;
    } else {
	DosBeep(880, 50) ;
	strcat(XmdmStat, " done ") ;
    }
    prompt(XmdmStat) ;

    keywait() ;
    scrRest() ;
}

void xmodemRecv(void)
{
    USHORT  stat ;
    UCHAR   *fname, *getlin() ;

    scrSave() ;

    prompt("XMODEM Recv File Name ? ") ;

    if ((fname = getlin(filename, 64)) == NULL) {
	scrRest() ;
	return ;
    }
    if (*filename == '\0') {
	scrRest() ;
	return ;
    }
	
    if ((stat = frecv(fname)) != 0) {
	DosBeep(440, 50) ;
	strcat(XmdmStat, " failed ") ;
    } else {
	DosBeep(880, 50) ;
	strcat(XmdmStat, " done ") ;
    }
    prompt(XmdmStat) ;

    keywait() ;
    scrRest() ;
}

