/****************************************************************************/
/*                                                                          */
/*  Script to create a maintenance partition or a bootable CD/DVD           */
/*                                                                          */
/*  Written by Hayo Baan (hayo.baan@xs4all.nl)                              */
/*    http://hrbaan.home.xs4all.nl/                                         */
/*                                                                          */
/*  Contributions by Doug Bissett & Lothar Soens                            */
/*                                                                          */
/*  Please see booAble.html for instructions and version history            */
/*                                                                          */
/*  Version 6.10.3 (05/02/2016)                                             */
/*                                                                          */
/****************************************************************************/

'@echo off'

SIGNAL ON SYNTAX
SIGNAL ON HALT

/* Make sure environment is local */
rc=setLocal()

/* Initialization */
rc=initialize(ARG(1))

/* Create bootAble CD/DVD / Maintenance partition */
rc=createBootAble()

/* Clean up and exit */
DONERC=cleanUp()

if isSet(PAUSEBEFOREEXIT) then do
    call lineout 'STDERR', 'Press any key to continue'
    rc=sysGetKey('NOECHO')
end

rc=endLocal()
exit DONERC


/****************************************************************************/
/**** Messaging Functions ***************************************************/
/****************************************************************************/

/* ------------------------------------------------------------------------ */
/* Display progress when in given verbose mode                              */
/* ARG1: Verbose mode required for showing the text                         */
/* ARG2: Text to show                                                       */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
progress: procedure expose VERBOSE
    if bitMask(VERBOSE, ARG(1)) then say ARG(2)
    return 0


/* ------------------------------------------------------------------------ */
/* Write an error message to stderr and exit program                        */
/* ARG1: Error text                                                         */
/* ARG2: Do not pause before exiting                                        */
/* ARG3: Query if program should be terminated after error                  */
/* ARG4: Do not query to save asked questions                               */
/* RET:  - (program is aborted)                                             */
/* ------------------------------------------------------------------------ */
error: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    call errorMessage ARG(1), ARG(2)
    if ARG(3)<>'' then do
        if yesNoQuit('Abort program?','Y','NOCLEAN')='No' then return
    end
    if ARG(4)='' then do
        PAUSEAFTERQUESTIONS='NO'
        rc=writeConfig();
    end
    rc=cleanUp()
    if ARG(2)='' & ARG(3)='' then do
        call lineout 'STDERR', 'Press any key to continue'
        rc=sysGetKey('NOECHO')
    end
    rc=endLocal()
    exit 1


/* ------------------------------------------------------------------------ */
/* Write an error message to stderr                                         */
/* ARG1: Error text                                                         */
/* ARG2: Do not show ERROR in front of text                                 */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
errorMessage: procedure
    call beep 440,150
    call lineout 'STDERR', ifClause(ARG(2)='','ERROR: ')||ARG(1)
    return 0


/* ------------------------------------------------------------------------ */
/* Write a warning message to stderr                                        */
/* ARG1: Warning text                                                       */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
warningMessage: procedure
    call lineout 'STDERR', 'WARNING:' ARG(1)
    return 0


/****************************************************************************/
/**** Generic Admin Functions ***********************************************/
/****************************************************************************/

/* ------------------------------------------------------------------------ */
/* Return true if the values have a matching bitmask                        */
/* ARG1: Bitmask 1                                                          */
/* ARG2: Bitmask 2                                                          */
/* RET:  True (1) or False (0)                                              */
/* ------------------------------------------------------------------------ */
bitMask: procedure
    if datatype(ARG(1))<>'NUM' then
        return 0
    else
        return c2d(bitAnd(d2c(ARG(1)),d2c(ARG(2))))>0


/* ------------------------------------------------------------------------ */
/* Cleanup                                                                  */
/* ARG: None                                                                */
/* RET: rc                                                                  */
/* ------------------------------------------------------------------------ */
cleanUp: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    if \isSet(CLEANUP) then return 0
    if strip(REXXEXIT_CLEANUP)<>'' then
        rc=runREXX(REXXEXIT_CLEANUP, 'running REXXEXIT_CLEANUP', 'REXXEXIT_CLEANUP')

    rc=cleanDir(BOOTABLETEMP,'YES')
    rc=sysRmDir(BOOTABLETEMP)
    if rc<>0 then
        call errorMessage 'Failed to remove temp directory' BOOTABLETEMP '('rc')'
    return rc


/* ------------------------------------------------------------------------ */
/* Escape single quotes in a text string                                    */
/* ARG1: Text to escape                                                     */
/* RET:  Escaped TXT                                                        */
/* ------------------------------------------------------------------------ */
escapeQuotes: procedure
    TXT=ARG(1)
    IDX=1
    do until IDX=0
        IDX=pos("'",TXT,IDX)
        if IDX>0 then do
            TXT=left(TXT,IDX)||'"'||"'"||'"'||"'"||substr(TXT,IDX+1)
            IDX=IDX+5
        end
    end
    return TXT


/* ------------------------------------------------------------------------ */
/* Determine if an option is set                                            */
/* ARG1: Value to test                                                      */
/* RET:  True (1) or False (0)                                              */
/* ------------------------------------------------------------------------ */
isSet: procedure
    return (left(translate(ARG(1)),1)='Y')


/* ------------------------------------------------------------------------ */
/* Inline version of the if clause                                          */
/* ARG1: Test                                                               */
/* ARG2: Value if TRUE                                                      */
/* ARG3: Value if FALSE                                                     */
/* RET:  If ARG1 then ARG2, else ARG3                                       */
/* Note: ARG2 and ARG3 are always evaluated!                                */
/* ------------------------------------------------------------------------ */
ifClause: procedure
    if (ARG(1)) then
        return ARG(2)
    else
        return ARG(3)


/* ------------------------------------------------------------------------ */
/* Combination of ifClause and isSet                                        */
/* ARG1: Value to test                                                      */
/* ARG2: Value if set (i.e., Yes)                                           */
/* ARG3: Value if not set                                                   */
/* RET:  If ARG1 is set then ARG2, else ARG3                                */
/* Note: ARG2 and ARG3 are always evaluated!                                */
/* ------------------------------------------------------------------------ */
ifSet: procedure
    if (isSet(ARG(1))) then
        return ARG(2)
    else
        return ARG(3)


/* ------------------------------------------------------------------------ */
/* Return the value of a variable or an empty string of not set             */
/* ARG1: Variable name                                                      */
/* RET:  Value of variable                                                  */
/* ------------------------------------------------------------------------ */
varValue:
    if symbol(arg(1))='VAR' then
        return value(arg(1))
    else
        return ''


/* ------------------------------------------------------------------------ */
/* Sets a variable to a default if not already set                          */
/* ARG1: Variable name                                                      */
/* ARG2: Default value for variable if not already specified                */
/* ARG3: Ask for value if not yet specified and default value is empty      */
/*       (if set to ASKIFNOTSET, ask even if default not empty)             */
/* ARG4: Text type of question                                              */
/* RET:  The value of the variable                                          */
/* ------------------------------------------------------------------------ */
defaultIfNotSet:
    VARNAME=ARG(1)
    VARVALUE=ARG(2)
    ASK=ARG(3)<>'' & (VARVALUE='' | ARG(3)='ASKIFNOTSET')
    if symbol(arg(1))='LIT' then do
        if ASK then
            VARVALUE=askText(VARVALUE,ARG(4))
        rc=value(VARNAME,VARVALUE)
    end
    return value(VARNAME)


/* ------------------------------------------------------------------------ */
/* Ask a multiple choice question                                           */
/* ARG1: List of keys                                                       */
/* ARG2: Question Text                                                      */
/* ARG3: Default key                                                        */
/* RET:  Answer Key                                                         */
/* ------------------------------------------------------------------------ */
multipleChoice: procedure
    KEYS=translate(ARG(1))
    TXT=ARG(2)' '
    DEFKEY=''
    if ARG(3)<>'' then
        DEFKEY=translate(left(ARG(3),1))
    if DEFKEY<>'' then
        TXT=TXT||'['DEFKEY'] '
    rc=charout(,TXT)
    rc=1
    do until rc=0
        KEY=translate(sysGetKey('NOECHO'))
        if key=d2c(13) & DEFKEY<>'' then
            KEY=DEFKEY
        if pos(KEY,KEYS)>0 then
            rc=0
        else
            call beep 524,100
    end
    return KEY


/* ------------------------------------------------------------------------ */
/* Ask a Yes/No/Quit question                                               */
/* ARG1: Question text                                                      */
/* ARG2: Default key                                                        */
/* ARG3: No cleanup when quit                                               */
/* RET:  Yes/No Answer                                                      */
/* ------------------------------------------------------------------------ */
yesNoQuit: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    NOCLEANUP=ARG(3)<>''
    TXT=ARG(1) '([Y]es/[N]o/[Q]uit'ifClause(NOCLEANUP,' without cleaning up')')'
    KEY=multipleChoice('YNQ',TXT,ARG(2))
    VAL=yesNoQuitText(KEY)
    say VAL
    if KEY='Q' then do
        TXT='ABORT: Script aborted'
        if NOCLEANUP then
            CLEANUP='NO'
        if \isSet(CLEANUP) then
            TXT=TXT 'without cleaning up'
        call error TXT, 'NOPAUSE'
    end
    return VAL


/* ------------------------------------------------------------------------ */
/* Return Yes/No/Quit text based on key value                               */
/* ARG1: Key                                                                */
/* RET:  Yes/No/Quit                                                        */
/* ------------------------------------------------------------------------ */
yesNoQuitText: procedure
    KEY=ARG(1)
    select
        when KEY='Y' then TXT='Yes'
        when KEY='N' then TXT='No'
        when KEY='Q' then TXT='Quit'
        otherwise         TXT='???'
    end
    return TXT


/****************************************************************************/
/**** Directory and File Handling Functions *********************************/
/****************************************************************************/

/* ------------------------------------------------------------------------ */
/* Returns the file argument if it exists, an empty string if not           */
/* If the second argument is not specified, the SEARCHPATH is used when the */
/*   file can not be found on the specified location                        */
/* Can be used in configuration files to add optional files                 */
/* ARG1: Filename                                                           */
/* ARG2: Optional do not searchpath                                         */
/* RET:  File specification                                                 */
/* ------------------------------------------------------------------------ */
fileExists: procedure expose SEARCHPATH
    FILESPEC=stream(ARG(1), 'C', 'QUERY EXISTS')
    if FILESPEC='' & ARG(2)='' & SEARCHPATH<>'' & pos('*', ARG(1))=0 & pos('?', ARG(1))=0 then
        FILESPEC=sysSearchPath('SEARCHPATH', fileSpec('NAME', ARG(1)))
    return FILESPEC


/* ------------------------------------------------------------------------ */
/* Returns the first file argument if it exists, the second if not          */
/* ARG1: Filename                                                           */
/* ARG2: Optional second text                                               */
/* RET:  File specification                                                 */
/* ------------------------------------------------------------------------ */
ifExists: procedure expose SEARCHPATH
    FILESPEC=fileExists(ARG(1))
    if FILESPEC<>'' then
        return FILESPEC
    else
        return ARG(2)


/* ------------------------------------------------------------------------ */
/* Returns the 1 if the directory exists, 0 if not                          */
/* ARG1: Directory                                                          */
/* RET:  True (1) or False (0)                                              */
/* ------------------------------------------------------------------------ */
dirExists: procedure
    DIR=ARG(1)
    if ((length(DIR)=3 & right(DIR,2)=':\') | (length(DIR)=2 & right(DIR,1)=':')) & pos(translate(left(DIR,2)),sysDriveMap('A:'))>0 then
        return 1
    rc=sysFileTree(DIR, 'DIRS', 'D')
    return (rc=0 & DIRS.0=1)

/* ------------------------------------------------------------------------ */
/* Return file name portion of a filespec                                   */
/* ARG1: Filename                                                           */
/* ARG2: Optional truncate name to 8.3                                      */
/* RET:  File name                                                          */
/* ------------------------------------------------------------------------ */
fileName: procedure
    FSPEC=filespec('NAME', ARG(1))
    if ARG(2) <> '' then do
        ORGFS=FSPEC
        FSDOT=pos('.', ORGFS)
        if FSDOT>0 then do
            FSNAME=left(ORGFS,min(FSDOT-1,8))
            FSEXT=right(ORGFS,length(ORGFS)-FSDOT)
        end
        else do
            FSNAME=left(ORGFS,min(length(ORGFS),8))
            FSEXT=''
        end
        FSPEC=FSNAME||ifClause(FSDOT>0,'.')||FSEXT
        if (FSPEC <> ORGFS) then
           call progress 2, 'Truncated filename: ' FSPEC '<' ARG(1)
    end

    return FSPEC


/* ------------------------------------------------------------------------ */
/* Return directory portion of a filespec                                   */
/* ARG1: Filename/Directory                                                 */
/* RET:  Directory                                                          */
/* ------------------------------------------------------------------------ */
directoryName: procedure
    return filespec('PATH', ARG(1))


/* ------------------------------------------------------------------------ */
/* Return drive portion of a filespec                                       */
/* ARG1: Filename/Directory                                                 */
/* RET:  Drive                                                              */
/* ------------------------------------------------------------------------ */
driveName: procedure
    return filespec('DRIVE', ARG(1))


/* ------------------------------------------------------------------------ */
/* Return path of a filespec (drive and directory)                          */
/* ARG1: Filename/Directory                                                 */
/* RET:  Drive and directory                                                */
/* ------------------------------------------------------------------------ */
pathName: procedure
    return driveName(ARG(1))||directoryName(ARG(1))


/* ------------------------------------------------------------------------ */
/* Create a file with given content                                         */
/* ARG1: File                                                               */
/* ARG2: Content                                                            */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
createFile: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    FILE=ARG(1)
    TXT=ARG(2)
    call progress 2, 'Creating file' FILE
    rc=charout(FILE,TXT)
    if rc=0 then
        rc=charout(FILE)
    if rc<>0 then
        call error 'Failed to create' FILE
    return 0


/* ------------------------------------------------------------------------ */
/* Create a directory                                                       */
/* ARG1: Directory                                                          */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
createDir: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    DIRNAME=ARG(1)
    if right(DIRNAME,1)='\' then
        DIRNAME=left(DIRNAME,length(DIRNAME)-1)

    /* Don't create it if it already exists */
    if right(DIRNAME,1)=':' | dirExists(DIRNAME) then
        return 0

    /* Make sure all higher directory levels are created */
    IDX=lastpos('\',DIRNAME)
    if IDX>3 then rc=createDir(left(DIRNAME,IDX-1))

    /* Create directory */
    call progress 2, 'Creating directory' DIRNAME
    rc=sysMkDir(DIRNAME)
    select
        when rc=2   then ERR='File not found'
        when rc=3   then ERR='Path not found'
        when rc=5   then ERR='Access denied'
        when rc=26  then ERR='Not a DOS disk'
        when rc=87  then ERR='Invalid parameter'
        when rc=108 then ERR='Drive locked'
        when rc=206 then ERR='Filename exceeds range'
        otherwise ERR=''
    end
    if rc<>0 then
        call error 'Error creating directory' DIRNAME||':' ERR
    return 0


/* ------------------------------------------------------------------------ */
/* Clean a directory tree                                                   */
/* ARG1: Directory                                                          */
/* ARG2: Quit when error, or just show error                                */
/* RET:  rc                                                                 */
/* ------------------------------------------------------------------------ */
cleanDir: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    call progress 1, 'Cleaning up directory' ARG(1)
    /* First delete all files */
    rc=sysFileTree(ARG(1)'\*', 'FILETREE', 'FOS',, '-*---')
    do I=1 to FILETREE.0 while rc=0
        rc=sysFileDelete(FILETREE.I)
    end
    if rc<>0 then
        ERR='Failed to remove files from directory' ARG(1) '('rc')'
    /* Then delete all directories */
    if rc=0 then do
        drop FILETREE.
        rc=sysFileTree(ARG(1)'\*', 'FILETREE', 'DOS',, '-*---')
        do I=FILETREE.0 to 1 by -1 while rc=0
            rc=sysRmDir(FILETREE.I)
        end
        if rc<>0 then
            ERR='Failed to remove directory structure' ARG(1) '('rc')'
    end
    if rc<>0 then
        if isSet(ARG(2)) then
            call errorMessage ERR
        else
            call error ERR
    return rc


/* ------------------------------------------------------------------------ */
/* Provide description of xcopy return code                                 */
/* ARG1: RC of xcopy command                                                */
/* RET:  Error description (empty in case of rc=0)                          */
/* ------------------------------------------------------------------------ */
xcopyErrorDescription: procedure
    rc=ARG(1)
    select
        when rc=0 then ERR=''
        when rc=1 then ERR='No files were found to xcopy'
        when rc=2 then ERR='Some files or directories were not copied due to file errors'
        when rc=3 then ERR='Ended by user'
        when rc=4 then ERR='Ended due to error'
        otherwise ERR='Unknown error ('rc')'
    end
    return ERR


/****************************************************************************/
/**** INI File Handling Functions *******************************************/
/****************************************************************************/

/* ------------------------------------------------------------------------ */
/* Return the text portion of an INI entry (i.e., strip of the 0 delimiter) */
/*   Hex values are returned as hex strings                                 */
/* ARG1: INI File to use                                                    */
/* ARG2: Application name                                                   */
/* ARG3: Key name                                                           */
/* ARG4: TXT delimiter?                                                     */
/* RET:  Text of INI entry                                                  */
/* ------------------------------------------------------------------------ */
getFromIni: procedure
    return inival2string(sysIni(ARG(1), ARG(2), ARG(3)),ARG(4))


/* ------------------------------------------------------------------------ */
/* Return the text portion of an INI entry (i.e., strip of the 0 delimiter) */
/*   Hex values are returned as hex strings                                 */
/* If the INI setting did not exist, the default value is returned instead  */
/* ARG1: INI File to use                                                    */
/* ARG2: Application name                                                   */
/* ARG3: Key name                                                           */
/* ARG4: Default value                                                      */
/* ARG5: TXT delimiter?                                                     */
/* RET:  Text of INI entry                                                  */
/* ------------------------------------------------------------------------ */
getFromIniWithDefault: procedure
    TXT = inival2string(sysIni(ARG(1), ARG(2), ARG(3)),ARG(5))
    if TXT='ERROR:' then
        return ARG(4)
    else
        return TXT


/* ------------------------------------------------------------------------ */
/* Convert a binary INI file to a text template file                        */
/* ARG1: Source INI file                                                    */
/* ARG2: Destination TXT file                                               */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
ini2txt: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    INIFILE=ARG(1)
    TXTFILE=ARG(2)
    APPLEN=31
    KEYLEN=31

    rc=sysIni(INIFILE, 'All:', 'APPS.')
    if rc='ERROR:' then
        call error 'Failed to read ini file' INIFILE
    do I=1 to APPS.0
        rc=sysIni(INIFILE, APPS.I, 'All:', 'KEYS')
        APPNAME='"'APPS.I'"'
        if length(APPS.I)<APPLEN then APPNAME=left(APPNAME,APPLEN)
        do j=1 to keys.0
            VAL=sysIni(INIFILE, APPS.I, KEYS.J)
            KEYNAME='"'KEYS.J'"'
            if length(KEYNAME)<KEYLEN then KEYNAME=left(KEYNAME,KEYLEN)
            rc=lineout(TXTFILE,APPNAME KEYNAME inival2string(VAL,'DELIM'))
        end
    end
    rc=lineout(TXTFILE)
    return 0


/* ------------------------------------------------------------------------ */
/* Format an INI value using the template format                            */
/* ARG1: INI Value                                                          */
/* ARG2: Add "'s around text values                                         */
/* RET:  Template form of value                                             */
/* ------------------------------------------------------------------------ */
inival2string: procedure
    TXTVAL=ARG(1)
    if TXTVAL='ERROR:' then
        return TXTVAL
    if length(TXTVAL)>0 & c2d(right(TXTVAL,1))<>0 then
        return '0x'c2x(TXTVAL)
    do C=1 to length(TXTVAL)-1
        DVAL=c2d(substr(TXTVAL,C,1))
        if DVAL<32|DVAL>=127|DVAL=c2d('"') then do
            return '0x'c2x(TXTVAL)
        end
    end
    if length(TXTVAL)>0 & c2d(right(TXTVAL,1))=0 then
        TXTVAL=left(TXTVAL,length(TXTVAL)-1)
    if ARG(2)<>'' then
        return '"'TXTVAL'"'
    else
        return TXTVAL


/****************************************************************************/
/**** System Functions ******************************************************/
/****************************************************************************/

/* ------------------------------------------------------------------------ */
/* Run a command checking return code                                       */
/* ARG1: Command to run                                                     */
/* ARG2: Optional error text if command fails                               */
/* ARG3: Ignore error                                                       */
/* RET:  0 for success                                                      */
/* ------------------------------------------------------------------------ */
runCMD: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    CMD=ARG(1)
    if ARG(2)<>'' then
        ERROR=ARG(2)

    else
        ERROR='Failed to run' ARG(1)
    IGNOREERROR=ARG(3)<>''
    CMD IGNOREOUTPUT
    if rc<>0 then
        if IGNOREERROR then call warningMessage ERROR '(rc='rc')'
        else call error ERROR '(rc='rc')'

    return rc

/* ------------------------------------------------------------------------ */
/* Run a piece of REXX code, checking return code                           */
/* ARG1: REXX code to run                                                   */
/* ARG2: Optional error text if command fails                               */
/* ARG3: REXXEXIT_CLEANUP? (special error handling)                         */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */

runREXX:
    REXXCMD=parseText(ARG(1))
    if ARG(2)<>'' then
        ERROR=ARG(2)
    else
        ERROR='Failed to run' left(strip(ARG(1), 50))
    IN_REXXEXIT_CLEANUP=ARG(3)<>''
    rc=0
    interpret REXXCMD
    if rc<>0 & rc<>'' then do
        if \IN_REXXEXIT_CLEANUP then call error ERROR '(rc='rc')'
        else call errormessage ERROR '(rc='rc')'
    end
    return 0


/****************************************************************************/
/**** bootAble File Handling Functions **************************************/
/****************************************************************************/

/* ------------------------------------------------------------------------ */
/* Copy a list of files                                                     */
/* ARG1: List of files                                                      */
/* ARG2: Destination                                                        */
/* ARG3: Truncate filenames to 8.3?                                         */
/* ARG4: Ignore not found?                                                  */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
copyFiles: procedure expose SEARCHPATH CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT MEMDISK WRITERDRIVE VIRTUALDRIVE REMOVEROFLAG LXLITE LXLITECOMMAND LXLITEOPTIONS CREATEBOOTCD CDRECORD CDRECORDMKISOFSLIST CRLF
    DESTINATION=ARG(2)
    if right(DESTINATION,1)<>'\' then DESTINATION=DESTINATION'\'
    TRUNCATE=ARG(3)
    IGNORENOTFOUND=ARG(4)
    call progress 2, 'Copying files to' DESTINATION
    rc=parseFileList(ARG(1))
    do I=1 to FILELIST.0
        drop FILETREE.
        rc=sysFileTree(FILELIST.I, 'FILETREE', 'F')
        if rc=0 then do
            if FILETREE.0=0 & IGNORENOTFOUND='' then do
                /* If not found at specified location, search additional path */
                SOURCEFILE=fileExists(FILELIST.I)
                if SOURCEFILE<>'' then do
                    drop FILETREE.
                    rc=sysFileTree(SOURCEFILE, 'FILETREE', 'F')
                end
            end
            if rc=0 then do
                if FILETREE.0=0 & IGNORENOTFOUND='' then
                    call error 'File not found' FILELIST.I, , 'ASK'
                do II=1 to FILETREE.0
                    rc=copyFile(FILETREE.II, DESTINATION, TRUNCATE)
                end II
            end
        end
        if rc<>0 then
            call error 'Not enough memory copying' FILELIST.I
    end I
    return 0


/* ------------------------------------------------------------------------ */
/* Copy a file (removing extended attributes when copying to boot image)    */
/* ARG1: File to copy in sysFileTree format                                 */
/* ARG2: Destination                                                        */
/* ARG3: Truncate filename to 8.3?                                          */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
copyFile: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT MEMDISK WRITERDRIVE VIRTUALDRIVE REMOVEROFLAG LXLITE LXLITECOMMAND LXLITEOPTIONS CREATEBOOTCD CDRECORD CDRECORDMKISOFSLIST CRLF
    parse value ARG(1) with dummy dummy SIZE ATTRIBUTES SOURCEFILESPEC
    ATTRIBUTES=space(translate(ATTRIBUTES, ' ', '-'),0)
    SOURCEFILESPEC=strip(SOURCEFILESPEC)
    SOURCEDIR=fileSpec('PATH', SOURCEFILESPEC)
    SOURCEFILE=fileName(SOURCEFILESPEC)
    DESTDIR=ARG(2)
    DESTFILE=fileName(SOURCEFILE, ARG(3))
    COMPLEXCOPY=isSet(LXLITE) | SOURCEFILE<>DESTFILE |,
        (\isSet(MEMDISK) & abbrev(translate(DESTDIR||DESTFILE),translate(VIRTUALDRIVE))) |,
        (isSet(MEMDISK)  & abbrev(translate(DESTDIR||DESTFILE),translate(VIRTUALDRIVE'\TOTAR.###')))

    if fileExists(DESTDIR||DESTFILE, 'T')<>'' then do
        call progress 2, 'Ignored' left(ATTRIBUTES,4) SOURCEFILESPEC '(file already copied)'
        return 0
    end

    if isSet(CREATEBOOTCD) & isSet(CDRECORD) & \abbrev(translate(SOURCEFILESPEC),translate(BOOTABLETEMP)) & abbrev(translate(DESTDIR||DESTFILE),translate(WRITERDRIVE)) then do
        DESTFILESPEC=substr(DESTDIR||DESTFILE,length(WRITERDRIVE)+1)
        IDX=1
        do UNTIL IDX=0
            IDX=pos('=',DESTFILESPEC,IDX)
            if IDX>0 then do
                DESTFILESPEC=left(DESTFILESPEC,IDX-1)||'\='||substr(DESTFILESPEC,IDX+1)
                IDX=IDX+2
            end
        end
        ISOFSLISTSPEC=DESTFILESPEC'='SOURCEFILESPEC
        rc=addISOFSListSpec(ISOFSLISTSPEC)
    end
    else if SIZE=0 then do
        rc=createFile(DESTDIR||DESTFILE,'')
    end
    else do
        if \isSet(REMOVEROFLAG) | pos('H',ATTRIBUTES)>0 | pos('S',ATTRIBUTES)>0 then
            XCOPYROFLAG='/r'
        else do
            ATTRIBUTES=space(translate(ATTRIBUTES, ' ', 'R'))
            XCOPYROFLAG=''
        end
        call progress 2, 'Copying' left(ATTRIBUTES,4) SOURCEFILESPEC '=>' DESTDIR||DESTFILE
        if COMPLEXCOPY then do
            'xcopy /h/t'XCOPYROFLAG '"'SOURCEFILESPEC'"' '"'strip(BOOTABLETEMP'\XCOPY.###','T','\')'"\' IGNOREOUTPUT
        end
        else do
            'xcopy /h/t'XCOPYROFLAG '"'SOURCEFILESPEC'"' '"'strip(DESTDIR,'T','\')'"\' IGNOREOUTPUT
        end
        ERR=xcopyErrorDescription(rc)
        if rc=0 & COMPLEXCOPY then do
            'attrib -a -h -r -s' '"'BOOTABLETEMP'\XCOPY.###\'SOURCEFILE'"' IGNOREOUTPUT
            if rc<>0 then ERR='Resetting attributes failed ('rc')'
        end
        if rc=0 & COMPLEXCOPY then do
            'eautil /s /r' '"'BOOTABLETEMP'\XCOPY.###\'SOURCEFILE'"' BOOTABLETEMP'\eautil.###' IGNOREOUTPUT
            if rc<>0 then ERR='Removing extended attributes failed'
        end
        if rc=0 & COMPLEXCOPY & isSet(LXLITE) then do
            LXLITECOMMAND LXLITEOPTIONS BOOTABLETEMP'\XCOPY.###\'SOURCEFILE IGNOREOUTPUT
            if rc<>0 then ERR='Lxlite failed ('rc')'
        end
        if rc=0 & COMPLEXCOPY then do
            'copy' '"'BOOTABLETEMP'\XCOPY.###\'SOURCEFILE'"' '"'DESTDIR||DESTFILE'"' IGNOREOUTPUT
            if rc<>0 then ERR='Copy to' DESTDIR||DESTFILE 'failed ('rc')'
        end
        if rc=0 & COMPLEXCOPY then do
            'del' '"'BOOTABLETEMP'\XCOPY.###\'SOURCEFILE'"' IGNOREOUTPUT
            if rc<>0 then ERR='Deletion of' BOOTABLETEMP'\XCOPY.###\'SOURCEFILE 'failed ('rc')'
        end
        if rc=0 & COMPLEXCOPY & ATTRIBUTES<>'' then do
            do I=1 to length(ATTRIBUTES)
                SETATTRIBUTES=' +'||substr(ATTRIBUTES,I,1)
            end
            'attrib' SETATTRIBUTES '"'DESTDIR||DESTFILE'"' IGNOREOUTPUT
            if rc<>0 then ERR='Setting' ATTRIBUTES 'attributes failed ('rc')'
        end
    end
    if rc<>0 then
            call error 'Error copying' SOURCEFILESPEC||':' ERR, , 'QueryAbort'
    return 0


/* ------------------------------------------------------------------------ */
/* Simple copy and rename file                                              */
/* ARG1: Source file                                                        */
/* ARG2: Destination file                                                   */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
sCopyFile: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    SOURCEFILESPEC=strip(ARG(1))
    DESTFILESPEC=strip(ARG(2))
    SOURCEFILESPEC=ifExists(SOURCEFILESPEC,SOURCEFILESPEC)

    call progress 2, 'sCopying' SOURCEFILESPEC '=>' DESTFILESPEC
    rc=runCMD('copy' '"'SOURCEFILESPEC'"' '"'DESTFILESPEC'"', 'Error copying' SOURCEFILESPEC 'to' DESTFILESPEC)
    return 0


/****************************************************************************/
/**** bootAble Admin Functions **********************************************/
/****************************************************************************/

/* ------------------------------------------------------------------------ */
/* Return the name of the boot image                                        */
/* ARG: None                                                                */
/* RET: Name of boot image                                                  */
/* ------------------------------------------------------------------------ */
bootImageName: procedure expose CREATEBOOTCD TWODISKBOOT IMAGENR WRITERDRIVE
    return strip(ifSet(CREATEBOOTCD,'Boot Image'||ifSet(TWODISKBOOT,' 'IMAGENR),'Maintenance Partition' WRITERDRIVE))


/* ------------------------------------------------------------------------ */
/* Return the name of the destination                                       */
/* ARG: None                                                                */
/* RET: Name of the destination                                             */
/* ------------------------------------------------------------------------ */
destinationName: procedure expose CDRECORD CDRECORDMEDIA CREATEBOOTCD TWODISKBOOT IMAGENR WRITERDRIVE DEST
    return strip(ifClause(isSet(CREATEBOOTCD) & DEST='CD',ifClause(CDRECORDMEDIA=0,'CD','DVD')||ifClause(isSet(CDRECORD)&isSet(CDRECORDCREATEISOONLY),' ISO Image'),bootImageName()))


/* ------------------------------------------------------------------------ */
/* Return drive letter based on destination                                 */
/* ARG1: Destination ('CD' or 'IMG')                                        */
/* RET:  Drive                                                              */
/* ------------------------------------------------------------------------ */
destinationDrive: procedure expose CREATEBOOTCD MEMDISK MEMDISKDRIVE WRITERDRIVE CDROMDRIVE
    DESTINATION=ARG(1)
    if isSet(CREATEBOOTCD) then
        if DESTINATION='CD' then
            return CDROMDRIVE
        else
            return ifSet(MEMDISK,MEMDISKDRIVE,'A:')
    return WRITERDRIVE


/* ------------------------------------------------------------------------ */
/* Check the Search Path and remove double/illegal entries                  */
/* ARG: None                                                                */
/* RET: 0                                                                   */
/* ------------------------------------------------------------------------ */
checkSearchPath: procedure expose SEARCHPATH CRLF
    NEWSEARCHPATH=''
    SEARCHPATH=translate(SEARCHPATH)
    if right(SEARCHPATH,1)<>';' then
        SEARCHPATH=SEARCHPATH';'
    IDX=1
    do until IDX=0
        IDX=pos(CRLF,SEARCHPATH)
        if IDX>0 then
            SEARCHPATH=strip(left(SEARCHPATH,IDX-1))';'strip(substr(SEARCHPATH,IDX+2))
    end
    do while length(SEARCHPATH)>0
        IDX=pos(';',SEARCHPATH)
        PATHENTRY=strip(left(SEARCHPATH,IDX-1))
        SEARCHPATH=substr(SEARCHPATH,IDX+1)
        if PATHENTRY<>'' then do
           if pos(PATHENTRY';',NEWSEARCHPATH)=0 & dirExists(PATHENTRY) then
               NEWSEARCHPATH=NEWSEARCHPATH||PATHENTRY||';'
        end
    end
    SEARCHPATH=NEWSEARCHPATH
    return 0


/* ------------------------------------------------------------------------ */
/* Parse text argument (replacing special $$ and %% constructs)             */
/* ARG1: Text                                                               */
/* RET:  Parsed text                                                        */
/* ------------------------------------------------------------------------ */
parseText:
    PARSETXT=ARG(1)

    PARSECONSTRUCT.0=4
    PARSECONSTRUCT.1='$$'
    PARSECONSTRUCT.2='%%'
    PARSECONSTRUCT.3='$$'
    PARSECONSTRUCT.4='%%'

    do PI=1 to PARSECONSTRUCT.0
        STARTPOS=1
        do until STARTPOS=0
            STARTPOS=pos(PARSECONSTRUCT.PI, PARSETXT, STARTPOS)
            if (STARTPOS<>0) then do
                ENDPOS=pos(PARSECONSTRUCT.PI, PARSETXT, STARTPOS+2)
                if ENDPOS<>0 then do
                    LEFTPARSETXT=left(PARSETXT,STARTPOS-1)
                    RIGHTPARSETXT=substr(PARSETXT,ENDPOS+2)
                    MIDPARSETXT=substr(PARSETXT,STARTPOS+2,ENDPOS-STARTPOS-2)
                    if (PI=2|PI=4) then do
                        if left(MIDPARSETXT,1)<>'!' & left(MIDPARSETXT,1)<>'~' then
                            VALUEMIDPARSETXT=ifSet(value(MIDPARSETXT),'Y','N')
                        else
                            VALUEMIDPARSETXT=ifSet(value(substr(MIDPARSETXT,2)),'N','Y')
                        MIDPARSETXT=''
                    end
                    else do
                        if left(MIDPARSETXT,1)<>'@' then
                            MIDPARSETXT=value(MIDPARSETXT)
                        else do
                            interpret 'MIDPARSETXT='substr(MIDPARSETXT,2)
                        end
                    end
                    if (PI=2|PI=4) & \isSet(VALUEMIDPARSETXT) then do
                        CRLFPOSBEFORE=lastpos(CRLF,left(PARSETXT,STARTPOS))
                        CRLFPOSAFTER=pos(CRLF,PARSETXT,ENDPOS)
                        if CRLFPOSAFTER=0 then CRLFPOSAFTER=length(PARSETXT)
                        if CRLFPOSBEFORE=0 then do
                            PARSETXT=substr(PARSETXT,CRLFPOSAFTER+2)
                            ENDPOS=1
                        end
                        else do
                            PARSETXT=left(PARSETXT,CRLFPOSBEFORE+1)||substr(PARSETXT,CRLFPOSAFTER+2)
                            ENDPOS=CRLFPOSBEFORE+1
                        end
                    end
                    else do
                        PARSETXT=LEFTPARSETXT||MIDPARSETXT||RIGHTPARSETXT
                        ENDPOS=STARTPOS+length(MIDPARSETXT)
                    end
                end
                STARTPOS=ENDPOS
            end
        end
    end
    return PARSETXT


/* ------------------------------------------------------------------------ */
/* Parse a list of files into FILELIST variable                             */
/* ARG1: List of files                                                      */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
parseFileList: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT FILELIST.
    FILE=strip(ARG(1))
    I=0
    do while length(FILE)>0
        if left(FILE,2)='/@' then do
            LISTFILE='X'
            FILE=strip(substr(FILE,3))
        end
        else
            LISTFILE=''
        LFT=left(FILE,1)
        if LFT='"' then
            FILE=substr(FILE,2)
        else
            LFT=' '
        IDX=pos(LFT,FILE)
        I=I+1
        if IDX=0 then do
            FILELIST.I=FILE
            FILE=''
        end
        else do
            FILELIST.I=left(FILE,IDX-1)
            FILE=strip(right(FILE,length(FILE)-IDX))
        end
        IDX=1
        do until IDX=0
            IDX=pos('\\',FILELIST.I)
            if (IDX>0) then do
                FILELIST.I=left(FILELIST.I,IDX-1)||substr(FILELIST.I,IDX+1)
            end
        end
        if LISTFILE<>'' then do
            LISTFILE=fileExists(FILELIST.I)
            if LISTFILE='' then
                call error 'List file' FILELIST.I 'not found',, 'ASK'
            else do
                LISTFILES=''
                do while lines(LISTFILE)
                    LISTFILELINE=strip(linein(LISTFILE))
                    if left(LISTFILELINE,1)<>';' then
                        LISTFILES=LISTFILES LISTFILELINE
                end
                rc=stream(LISTFILE, 'C', 'CLOSE')
                if rc<>'READY:' then call error 'Failed to close' LISTFILE '('rc')'
                I=I-1
                FILE=strip(LISTFILES FILE)
            end
        end
    end
    FILELIST.0=I
    return 0


/* ------------------------------------------------------------------------ */
/* Add an ISOFS List file specification to the existing list, checking for  */
/*   double (destination) entries.                                          */
/* ARG1: ISOFS List file specification (dest-file=source-file)              */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
addISOFSListSpec: procedure expose CDRECORDMKISOFSLIST CRLF CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    LISTSPEC=ARG(1)
    /* First make sure all '\'s are replaced by '/'s, except for \= */
    IDXa=pos('\=',LISTSPEC)
    IDXb=pos('\',LISTSPEC)
    do while IDXb<>0
        if IDXa<>IDXb then
            LISTSPEC=left(LISTSPEC,IDXb-1)||'/'||substr(LISTSPEC,IDXb+1)
        IDXa=pos('\=',LISTSPEC,IDXb+1)
        IDXb=pos('\',LISTSPEC,IDXb+1)
    end
    /* Make sure destination begins with a '/' */
    if left(LISTSPEC,1)<>'/' then
        LISTSPEC='/'LISTSPEC
    /* Determine the destination file specification */
    IDXa=pos('\=',LISTSPEC)
    IDXb=pos('=',LISTSPEC)
    do while IDXa<>0
        IDXb=pos('=',LISTSPEC,IDXa+2)
        IDXa=pos('\=',LISTSPEC, IDXa+2)
    end
    if IDXb=0 then call error 'Illegal ISOFS List file specification' LISTSPEC
    DESTFILE=left(LISTSPEC,IDXb)
    /* Has the destination already been added? */
    if pos(translate(CRLF||DESTFILE),translate(CRLF||CDRECORDMKISOFSLIST))<>0 then
        call progress 2, 'Ignored' LISTSPEC '(destination file already added to the mkisofs list)'
    else do
        call progress 2, 'Adding to mkisofs list' LISTSPEC
        CDRECORDMKISOFSLIST=CDRECORDMKISOFSLIST||LISTSPEC||CRLF
    end

    return 0


/* ------------------------------------------------------------------------ */
/* Extends a path line with the given path                                  */
/*   Makes sure no duplicates get added.                                    */
/* ARG1: Current path line                                                  */
/* ARG2: Path to add                                                        */
/* RET: Extended path                                                       */
/* ------------------------------------------------------------------------ */
addToPathLine: procedure
    PATHLINE=ARG(1)
    ADDTOPATHTXT=ARG(2)
    if \(right(ADDTOPATHTXT,1)=';') then ADDTOPATHTXT=ADDTOPATHTXT';'
    if pos(translate(ADDTOPATHTXT),translate(PATHLINE))=0 then
        return PATHLINE||ADDTOPATHTXT
    else
        return PATHLINE


/* ------------------------------------------------------------------------ */
/* Write a line to the config.sys file                                      */
/* ARG1: Text                                                               */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
configLine: procedure expose VIRTUALDRIVE
    rc=lineout(VIRTUALDRIVE'\config.sys',strip(ARG(1)))
    return 0


/****************************************************************************/
/**** bootAble Config Retrieval Functions ***********************************/
/****************************************************************************/

/* ------------------------------------------------------------------------ */
/* Return the remainder of the first line of the specified file starting    */
/*   with argument                                                          */
/* Optionally an index number and a delimiter can be supplied, then only    */
/*   the indexed part of the string as delimited is returned                */
/* ARG1: Filename                                                           */
/* ARG2: Search 'start of line'                                             */
/* ARG3: Optional (word) index of return value                              */
/* ARG4: Optional delimiter for index                                       */
/* ARG5: Ignore file not found                                              */
/* RET:  Remainder of matching line                                         */
/* ------------------------------------------------------------------------ */
getFromFile: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    ORGFILE=ARG(1)
    FILE=fileExists(ORGFILE)
    KEYTXT=translate(ARG(2))
    INDEXTXT=ARG(3)
    DELIMTXT=ARG(4)
    TXT=''
    if FILE='' then
        if ARG(5)='' then
            call error 'File' ORGFILE 'not found'
        else
            return ''
    do while lines(FILE)
        LINE=space(linein(FILE),1)
        if abbrev(translate(LINE), KEYTXT) then do
            TXT=substr(LINE,length(KEYTXT)+1)
            leave
        end
    end
    rc=stream(FILE, 'C', 'CLOSE')
    if INDEXTXT<>'' then do
        if DELIMTXT<>'' then do
            LENDELIMTXT=length(DELIMTXT)
            do I=1 to INDEXTXT-1
                POSTXT=pos(DELIMTXT, TXT)
                if POSTXT>0 then
                    TXT=substr(TXT,POSTXT+LENDELIMTXT)
                else do
                    TXT=''
                    leave
                end
            end
            POSTXT=pos(DELIMTXT, TXT)
            if POSTXT>0 then
                TXT=left(TXT,POSTXT-1)
        end
        else
            TXT=word(TXT,INDEXTXT)
    end
    return TXT


/* ------------------------------------------------------------------------ */
/* Return the remainders of the lines of the specified file starting with   */
/*   argument                                                               */
/* Optionally an index number and a delimiter can be supplied, then only    */
/*   the indexed part of the string as delimited is returned                */
/* ARG1: Filename                                                           */
/* ARG2: Search 'start of line'                                             */
/* ARG3: Optional (word) index of return value                              */
/* ARG4: Optional delimiter for index                                       */
/* ARG5: Ignore double entries                                              */
/* ARG6: Ignore file not found                                              */
/* RET:  NR of matched lines, the matched lines themselves are placed in    */
/*         the variable stem MATCHEDLINES.                                  */
/* ------------------------------------------------------------------------ */
getFromFileMulti: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT MATCHEDLINES.
    ORGFILE=ARG(1)
    FILE=fileExists(ORGFILE)
    KEYTXT=translate(ARG(2))
    INDEXTXT=ARG(3)
    DELIMTXT=ARG(4)
    IGNOREDOUBLE=ARG(5)
    drop MATCHEDLINES.
    NRMATCHED=0
    if FILE='' then
        if ARG(6)='' then
            call error 'File' ORGFILE 'not found'
        else
            return 0
    do while lines(FILE)
        TXT=''
        LINE=space(linein(FILE),1)
        if abbrev(translate(LINE), KEYTXT) then do
            TXT=substr(LINE,length(KEYTXT)+1)
            if INDEXTXT<>'' then do
                if DELIMTXT<>'' then do
                    LENDELIMTXT=length(DELIMTXT)
                    do I=1 to INDEXTXT-1
                        POSTXT=pos(DELIMTXT, TXT)
                        if POSTXT>0 then
                            TXT=substr(TXT,POSTXT+LENDELIMTXT)
                        else do
                            TXT=''
                            leave
                        end
                    end
                    POSTXT=pos(DELIMTXT, TXT)
                    if POSTXT>0 then
                        TXT=left(TXT,POSTXT-1)
                end
                else
                    TXT=word(TXT,INDEXTXT)
            end
            NEWMATCH=1
            if IGNOREDOUBLE<>'' then do
                do I=1 to NRMATCHED
                    if MATCHEDLINES.NRMATCHED=TXT then
                        NEWMATCH=0
                end
            end
            if NEWMATCH then do
                NRMATCHED=NRMATCHED+1
                MATCHEDLINES.NRMATCHED=TXT
            end
        end
    end
    rc=stream(FILE, 'C', 'CLOSE')
    return NRMATCHED


/* ------------------------------------------------------------------------ */
/* Return 'YES' if the specifid file contains a line made up of the first   */
/*   argument folowed by the second argument possibly prefixed by a path    */
/* ARG1: Filename                                                           */
/* ARG2: Search 'start of line'                                             */
/* ARG3: Search 'file text'                                                 */
/* ARG4: Optional empty 'file text' => 0 matches                            */
/* ARG5: Optional file does not exist => 0 matches (error otherwise)        */
/* RET:  Yes/No                                                             */
/* ------------------------------------------------------------------------ */
fileContains: procedure
    return ifClause(fileCountMatches(ARG(1), ARG(2), ARG(3), 'NOCOUNT', ARG(4), ARG(5))>0,'YES','NO')


/* ------------------------------------------------------------------------ */
/* Return the number of times the specified file contains a line made up of */
/*   the first argument folowed by the second argument possibly prefixed by */
/*   a path                                                                 */
/* If the optional third argument is given, counting stops after the first  */
/*   occurence (in that case the line number is returned).                  */
/* The optional fourth argument specifies if an empty second argument is to */
/*   cause 0 matches.                                                       */
/* The fifth optional argument determines if a file not found is to cause 0 */
/*   matches or fail with an error                                          */
/* ARG1: Filename                                                           */
/* ARG2: Search 'start of line'                                             */
/* ARG3: Search 'file text'                                                 */
/* ARG4: Optional do not count and return line number of first match        */
/* ARG5: Optional empty 'file text' => 0 matches                            */
/* ARG6: Optional file does not exist => 0 matches (error otherwise)        */
/* RET:  Number of matches                                                  */
/* ------------------------------------------------------------------------ */
fileCountMatches: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    ORGFILE=ARG(1)
    FILE=fileExists(ORGFILE)
    KEYTXT=translate(ARG(2))
    FILETXT=translate(ARG(3))
    COUNT=ARG(4)=''
    if FILETXT='' & ARG(5)<>'' then
        return 0
    if FILE='' then
        if ARG(6)='' then
            call error 'File' ORGFILE 'not found'
        else
            return 0
    FOUND=0
    LINENR=0
    do while lines(FILE)
        LINE=translate(space(linein(FILE),1))
        LINENR=LINENR+1
        if abbrev(LINE, KEYTXT) then do
            TXT=''
            if FILETXT<>'' then do
                LINE=substr(LINE,length(KEYTXT)+1)
                IDX=pos(' ',LINE)
                if IDX>0 then
                    TXT=left(LINE,IDX-1)
                else
                    TXT=LINE
                TXT=substr(TXT,lastpos('\',TXT)+1)
            end
            if TXT=FILETXT then do
                FOUND=FOUND+1
                if \COUNT then do
                    FOUND=LINENR
                    leave
                end
            end
        end
    end
    rc=stream(FILE, 'C', 'CLOSE')
    return FOUND


/* ------------------------------------------------------------------------ */
/* Return the remainder of the first line in the specified file starting    */
/*   with the first argument folowed by the second argument possibly        */
/*   prefixed by a path                                                     */
/*   The third optional argument determines if a file not found is to cause */
/*   an empty string to be returned or fail with an error                   */
/* ARG1: Filename                                                           */
/* ARG2: Search 'start of line'                                             */
/* ARG3: Search 'file text'                                                 */
/* ARG4: Optional file does not exist => empty string (error otherwise)     */
/* RET:  Remainder of matching line                                         */
/* ------------------------------------------------------------------------ */
fileMatchRemainder: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    ORGFILE=ARG(1)
    FILE=fileExists(ORGFILE)
    KEYTXT=translate(ARG(2))
    FILETXT=translate(ARG(3))
    if FILETXT='' then return ''
    if FILE='' then
        if ARG(4)='' then
            call error 'File' ORGFILE 'not found'
        else
            return 0
    OPTIONTXT=''
    do while lines(FILE)
        LINE=translate(space(linein(FILE),1))
        if abbrev(LINE, KEYTXT) then do
            TXT=''
            LINE=substr(LINE,length(KEYTXT)+1)
            IDX=pos(' ',LINE)
            if IDX>0 then
                TXT=left(LINE,IDX-1)
            else
                TXT=LINE
            IDX=lastpos('\',TXT)+1
            TXT=substr(TXT,IDX)
            if TXT=FILETXT then do
                OPTIONTXT=strip(substr(LINE,IDX+length(FILETXT)))
                leave
            end
        end
    end
    rc=stream(FILE, 'C', 'CLOSE')
    return OPTIONTXT


/* ------------------------------------------------------------------------ */
/* Return the full file path of the file referenced on the first line in    */
/*   the specified file starting with the first argument folowed by the     */
/*   second argument possibly prefixed by a path                            */
/*   The third optional argument determines if a file not found is to cause */
/*   an empty string to be returned or fail with an error                   */
/* ARG1: Filename                                                           */
/* ARG2: Search 'start of line'                                             */
/* ARG3: Search 'file text'                                                 */
/* ARG4: Optional file does not exist => empty string (error otherwise)     */
/* RET:  Full file path                                                     */
/* ------------------------------------------------------------------------ */
fileMatchFullFile: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    ORGFILE=ARG(1)
    FILE=fileExists(ORGFILE)
    KEYTXT=translate(ARG(2))
    FILETXT=translate(ARG(3))
    if FILETXT='' then return ''
    if FILE='' then
        if ARG(4)='' then
            call error 'File' ORGFILE 'not found'
        else
            return 0
    FULLFILETXT=''
    do while lines(FILE)
        LINE=translate(space(linein(FILE),1))
        if abbrev(LINE, KEYTXT) then do
            TXT=''
            LINE=substr(LINE,length(KEYTXT)+1)
            IDX=pos(' ',LINE)
            if IDX>0 then
                LINE=left(LINE,IDX-1)
            TXT=LINE
            IDX=lastpos('\',TXT)+1
            TXT=substr(TXT,IDX)
            if TXT=FILETXT then do
                FULLFILETXT=LINE
                leave
            end
        end
    end
    rc=stream(FILE, 'C', 'CLOSE')
    return FULLFILETXT


/* ------------------------------------------------------------------------ */
/* Return the remainder of the first config.sys line starting with argument */
/* Optionally an index number and a delimiter can be supplied, then only    */
/*   the indexed part of the string as delimited is returned                */
/* ARG1: Search 'start of line'                                             */
/* ARG2: Optional (word) index of return value                              */
/* ARG3: Optional delimiter for index                                       */
/* RET:  Remainder of matching line                                         */
/* ------------------------------------------------------------------------ */
getFromConfigSys: procedure expose CONFIGSYSFILE
    return getFromFile(CONFIGSYSFILE, ARG(1), ARG(2), ARG(3))


/* ------------------------------------------------------------------------ */
/* Return the remainders of config.sys lines starting with argument         */
/* Optionally an index number and a delimiter can be supplied, then only    */
/*   the indexed part of the string as delimited is reurned                 */
/* ARG1: Search 'start of line'                                             */
/* ARG2: Optional (word) index of return value                              */
/* ARG3: Optional delimiter for index                                       */
/* ARG4: Ignore double entries                                              */
/* ARG5: Ignore file not found                                              */
/* RET:  NR of matched lines, the matched lines themselves are placed in    */
/*         the variable stem MATCHEDLINES.                                  */
/* ------------------------------------------------------------------------ */
getFromConfigSysMulti: procedure expose CONFIGSYSFILE MATCHEDLINES.
    return getFromFileMulti(CONFIGSYSFILE, ARG(1), ARG(2), ARG(3), ARG(4), ARG(5))


/* ------------------------------------------------------------------------ */
/* Return 'YES' if the config.sys file contains a line made up of the first */
/*   argument folowed by the second argument possibly prefixed by a path    */
/* ARG1: Search 'start of line'                                             */
/* ARG2: Search 'file text'                                                 */
/* RET:  Yes/No                                                             */
/* ------------------------------------------------------------------------ */
configSysContains: procedure expose CONFIGSYSFILE
    return ifClause(configSysCount(ARG(1), ARG(2), 'NOCOUNT')>0,'YES','NO')


/* ------------------------------------------------------------------------ */
/* Return the number of times the config.sys file contains a line made up   */
/*   of the first argument folowed by the second argument possibly prefixed */
/*   by a path                                                              */
/* If the optional third argument is given, counting stops after the first  */
/*   occurence                                                              */
/* ARG1: Search 'start of line'                                             */
/* ARG2: Search 'file text'                                                 */
/* ARG3: Optional do not count and return line number of first match        */
/* RET:  Number of matches                                                  */
/* ------------------------------------------------------------------------ */
configSysCount: procedure expose CONFIGSYSFILE
    return fileCountMatches(CONFIGSYSFILE, ARG(1), ARG(2), ARG(3), 'EMPTY=0')


/* ------------------------------------------------------------------------ */
/* Return the remainder of the first config.sys line starting with the      */
/*   first argument folowed by the second argument possibly prefixed by a   */
/*   path                                                                   */
/* ARG1: Search 'start of line'                                             */
/* ARG2: Search 'file text'                                                 */
/* RET:  Remainder of matching line                                         */
/* ------------------------------------------------------------------------ */
configSysOptions: procedure expose CONFIGSYSFILE
    return fileMatchRemainder(CONFIGSYSFILE, ARG(1), ARG(2))


/* ------------------------------------------------------------------------ */
/* Return the full file path of the file referenced on the first config.sys */
/*   line starting with the first argument folowed by the                   */
/*   second argument possibly prefixed by a path                            */
/* ARG1: Search 'start of line'                                             */
/* ARG2: Search 'file text'                                                 */
/* RET:  Full file path                                                     */
/* ------------------------------------------------------------------------ */
configSysFullFile: procedure expose CONFIGSYSFILE
    return fileMatchFullFile(CONFIGSYSFILE, ARG(1), ARG(2))


/* ------------------------------------------------------------------------ */
/* Strip the LOGFILE option from a SIO config line                          */
/* ARG1: Option text                                                        */
/* RET:  Option text without LOGFILE option                                 */
/* ------------------------------------------------------------------------ */
ignoreSIOLogfileOption: procedure
    OPTIONTXT=ARG(1)
    IDX=pos('LOGFILE=',OPTIONTXT)
    if IDX<>0 then
        OPTIONTXT=strip(delstr(OPTIONTXT' ',IDX,pos(' ',OPTIONTXT' ',IDX)-IDX+1),'B')
    return OPTIONTXT


/* ------------------------------------------------------------------------ */
/* Returns the files required for the QSINIT system loader, includes the    */
/*   kernels as specified in the OS2LDR.INI file                            */
/* RET:  List of all kernel files (empty when QSINIT is not enabled)        */
/* ------------------------------------------------------------------------ */
qsInitFiles: procedure expose QSINIT OSBOOTDRIVE
    FILES=''
    if (isSet(QSINIT) & fileExists(OSBOOTDRIVE'\OS2LDR.INI')<>'') then do
        FILES=,
            OSBOOTDRIVE'\os2ldr.ini',
            OSBOOTDRIVE'\qsinit.ldi',
            fileExists(OSBOOTDRIVE'\qssetup.cmd')
        FILE=OSBOOTDRIVE'\os2ldr.ini'
        MARKERFOUND=0
        do while lines(FILE)
            LINE=space(linein(FILE),1)
            if \MARKERFOUND then
                MARKERFOUND=translate(LINE)='[KERNEL]'
            else do
                POS=pos('=', LINE)
                if (POS>0) then do
                    LINE=LEFT(LINE,POS-1)
                    FILES=FILES OSBOOTDRIVE'\'LINE
                end
            end
        end
        rc=stream(FILE, 'C', 'CLOSE')
    end
    return FILES


/* ------------------------------------------------------------------------ */
/* Return the UNICODE keyboard layout based on the KEYB layout              */
/* ARG1: Keyboard code                                                      */
/* RET:  UNICODE Keyboard code                                              */
/* ------------------------------------------------------------------------ */
unicodeKeyboard: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT OSBOOTDRIVE
    KEYB=translate(ARG(1))
    UNIKEYB=''
    select
        when KEYB='AR470' then UNIKEYB='AA470'   /* Arabic 101 */
        when KEYB='GK'    then UNIKEYB='EL'      /* Greek */
        when KEYB='GK220' then UNIKEYB='EL220'   /* Greek 101 */
        when KEYB='GK459' then UNIKEYB='EL459'   /* Greek 459 */
        when KEYB='GR'    then UNIKEYB='DE'      /* Germany */
        when KEYB='GR129' then UNIKEYB='DE'      /* Germany */
        when KEYB='SG'    then UNIKEYB='SD'      /* Swiss German */
        when KEYB='SP'    then UNIKEYB='ES'      /* Spain */
        when KEYB='SU'    then UNIKEYB='FI'      /* Finland */
        when KEYB='UX'    then UNIKEYB='USINTER' /* US International */
        otherwise              UNIKEYB=KEYB
    end
    if length(UNIKEYB)>2 & fileExists(OSBOOTDRIVE'\language\keyboard\'UNIKEYB'.kbl')='' then
        UNIKEYB=left(UNIKEYB,2)
    if fileExists(OSBOOTDRIVE'\language\keyboard\'UNIKEYB'.kbl')='' then
        call error 'Unknown keyboard layout' KEYB', specify UNIKEYB manually'
    return UNIKEYB'.kbl'


/* ------------------------------------------------------------------------ */
/* Return the INI setting for a specified display resolution                */
/* ARG: None                                                                */
/* RET: Video resolution INI setting                                        */
/* ------------------------------------------------------------------------ */
videoResolutionINISetting: procedure expose VIDEODRIVER VIDEORESOLUTION VIDEOCOLORDEPTH
    INISETTING=''
    if VIDEODRIVER<>1 and VIDEORESOLUTION<>0 then do
        /* Video resolution and color depth are encoded in 32bit */
        /* Little-Endian notation                                */
        select
            when VIDEORESOLUTION=2 then INISETTING='0x20030000 58020000' /*  800x 600 */
            when VIDEORESOLUTION=3 then INISETTING='0x00040000 00030000' /* 1024x 768 */
            when VIDEORESOLUTION=4 then INISETTING='0x00050000 00040000' /* 1280x1024 */
            when VIDEORESOLUTION=5 then INISETTING='0x40060000 b0040000' /* 1600x1200 */
            otherwise                   INISETTING='0x80020000 e0010000' /*  640x 480 */
        end
        select
            when VIDEOCOLORDEPTH=2 then INISETTING=INISETTING '00800000' /* 32K colors */
            when VIDEOCOLORDEPTH=3 then INISETTING=INISETTING '00000100' /* 64K colors */
            when VIDEOCOLORDEPTH=4 then INISETTING=INISETTING '00000001' /* 16M colors */
            otherwise                   INISETTING=INISETTING '00010000' /* 256 colors */
        end
        /* Two extra 32bit integers don't know what they are for (refresh rate?) */
        INISETTING=INISETTING '01000000 40000000'
    end
    return INISETTING


/* ------------------------------------------------------------------------ */
/* Return the INI setting for a specified font size                         */
/* ARG: None                                                                */
/* RET: Font size INI setting                                               */
/* ------------------------------------------------------------------------ */
wpsShellFontSizeINISetting: procedure expose WPSSHELLFONTSIZE
    INISETTING=''
    select
        when WPSSHELLFONTSIZE=1  then INISETTING='0x0808'
        when WPSSHELLFONTSIZE=2  then INISETTING='0x060A'
        when WPSSHELLFONTSIZE=3  then INISETTING='0x080A'
        when WPSSHELLFONTSIZE=4  then INISETTING='0x050C'
        when WPSSHELLFONTSIZE=5  then INISETTING='0x080C'
        when WPSSHELLFONTSIZE=6  then INISETTING='0x060E'
        when WPSSHELLFONTSIZE=7  then INISETTING='0x080E'
        when WPSSHELLFONTSIZE=8  then INISETTING='0x070F'
        when WPSSHELLFONTSIZE=9  then INISETTING='0x0510'
        when WPSSHELLFONTSIZE=10 then INISETTING='0x0810'
        when WPSSHELLFONTSIZE=11 then INISETTING='0x0C10'
        when WPSSHELLFONTSIZE=12 then INISETTING='0x0812'
        when WPSSHELLFONTSIZE=13 then INISETTING='0x0A12'
        when WPSSHELLFONTSIZE=14 then INISETTING='0x0C14'
        when WPSSHELLFONTSIZE=15 then INISETTING='0x0C16'
        when WPSSHELLFONTSIZE=16 then INISETTING='0x0719'
        when WPSSHELLFONTSIZE=17 then INISETTING='0x0C1E'
    end
    return INISETTING


/* ------------------------------------------------------------------------ */
/* Return the font size of the current shell font setting                   */
/* ARG: None                                                                */
/* RET: Current font size number                                            */
/* ------------------------------------------------------------------------ */
currentWpsShellFontSize: procedure
    INISETTING=getFromIni(,'Shield','~Font Size...')
    select
        when INISETTING='0x0808' then FONTSIZE=1
        when INISETTING='0x060A' then FONTSIZE=2
        when INISETTING='0x080A' then FONTSIZE=3
        when INISETTING='0x050C' then FONTSIZE=4
        when INISETTING='0x080C' then FONTSIZE=5
        when INISETTING='0x060E' then FONTSIZE=6
        when INISETTING='0x080E' then FONTSIZE=7
        when INISETTING='0x070F' then FONTSIZE=8
        when INISETTING='0x0510' then FONTSIZE=9
        when INISETTING='0x0810' then FONTSIZE=10
        when INISETTING='0x0C10' then FONTSIZE=11
        when INISETTING='0x0812' then FONTSIZE=12
        when INISETTING='0x0A12' then FONTSIZE=13
        when INISETTING='0x0C14' then FONTSIZE=14
        when INISETTING='0x0C16' then FONTSIZE=15
        when INISETTING='0x0719' then FONTSIZE=16
        when INISETTING='0x0C1E' then FONTSIZE=17
        otherwise FONTSIZE=5
    end
    return FONTSIZE


/* ------------------------------------------------------------------------ */
/* Return the drive letter of the CDROM                                     */
/* ARG: None                                                                */
/* RET: CDROM drive letter                                                  */
/* ------------------------------------------------------------------------ */
cdromDriveLetter: procedure expose RESERVEDRIVELETTER CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    if RESERVEDRIVELETTER<>'' then
        return d2c(c2d(RESERVEDRIVELETTER)+1)':'
    DRIVEMAP=sysDriveMap(,'LOCAL')
    do I=1 to words(DRIVEMAP)
        DRIVE=word(DRIVEMAP,I)
        DRIVEFS=sysFileSystemType(DRIVE)
        if DRIVEFS='' | DRIVEFS='CDFS' then
            return DRIVE
    end
    call error 'No CDROM drive found, specify CDROMDRIVE manually'
    return ''


/* ------------------------------------------------------------------------ */
/* Return the writer drive path                                             */
/* ARG: None                                                                */
/* RET: Path for the writer drive                                           */
/* ------------------------------------------------------------------------ */
writerDrivePath: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CREATEBOOTCD CDRECORD RSJATTACHCD TEMP CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    VARNAME='WRITERDRIVE'
    if isSet(CREATEBOOTCD) then do
        if isSet(CDRECORD) then do
            COMMENT='; Directory to store image files for use with mkisofs/cdrecord'
            DRIVE=askText(TEMP'bootAble.IMG','DIRECTORY')
            return DRIVE
        end
        if isSet(RSJATTACHCD) then do
            DRIVEMAP=sysDriveMap(,'FREE')
            DRIVE=word(DRIVEMAP,1)
            if DRIVE='' then
                call error 'No available drives to attach CD-R(W)'
            return DRIVE
        end
        call progress 1, 'Looking for attached CD-R(W), this may take a while...'
        DRIVEMAP=sysDriveMap(,'REMOTE')
        do I=1 to words(DRIVEMAP)
            DRIVE=word(DRIVEMAP,I)
            if sysFileSystemType(DRIVE)='CDWFS' then
                return DRIVE
        end
        call error 'No attached CD-R(W) found'
    end
    COMMENT='; Drive for the maintenance partition'
    DRIVE=askText(,'DRIVE')
    return DRIVE


/* ------------------------------------------------------------------------ */
/* Return the cdrecorder device parameter for cdrecord                      */
/* ARG: None                                                                */
/* RET: cdrecord device parameter                                           */
/* ------------------------------------------------------------------------ */
cdRecorderDevice: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CREATEBOOTCD CDRECORDDEVICENR CDRECORDDEVICE CDRECORD CDRECORDMEDIA CDRECORDEXE CDRECORDCREATEISOONLY CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    DEVICE='0,1,0'
    NRDEVICES=0
    DEVICENR=0
    if isSet(CREATEBOOTCD) & isSet(CDRECORD) & CDRECORDMEDIA=0 & \isSet(CDRECORDCREATEISOONLY) then do
        if  fileExists(CDRECORDEXE)<>'' then do
            QUEUE=rxqueue('CREATE')
            rc=rxqueue('SET',QUEUE)
            rc=runCMD(CDRECORDEXE '-scanbus | rxqueue' QUEUE,,'IGNOREERROR')
            do while queued()>0
                parse pull DEV')' "'"BRAND"'" "'"MODEL"'" "'"VERSION"'" DESCRIPTION
                DESCRIPTION=strip(DESCRIPTION)
                if pos('CD-R ',MODEL)=1 | pos('DVD',MODEL)=1 | pos('RW/DVD',MODEL)=1 then do
                    DEV=strip(DEV)
                    DEV=strip(DEV,,'	')
                    DEV=left(DEV,pos('	',DEV))
                    NRDEVICES=NRDEVICES+1
                    DEVICES.NRDEVICES=DEV
                    DEVICESDESCRIPTION.NRDEVICES="'"BRAND"'" "'"MODEL"'" "'"VERSION"'" DESCRIPTION
                end
            end
            rc=rxqueue('DELETE', QUEUE)
            if rc<>0 then
                call error 'Failed to delete REXX queue' QUEUE '('rc')'
            if NRDEVICES=0 | NRDEVICES=1
               then DEVICENR=NRDEVICES
            else do
                if CDRECORDDEVICENR='' & isSet(CDRECORDCREATEISOONLY) then
                    CDRECORDDEVICENR=1
                if CDRECORDDEVICENR='' then do
                    COMMENT='; CD Writer device number to use'||CRLF||,
                        '; The following CD writer devices were found:'
                    do I=1 to NRDEVICES
                        COMMENT=COMMENT||CRLF'; 'I':	'DEVICES.I'	'DEVICESDESCRIPTION.I
                    end
                    VARNAME='CDRECORDDEVICENR'
                    CDRECORDDEVICENR=askNumeric(1,1,NRDEVICES)
                end
                if dataType(CDRECORDDEVICENR)<>'NUM' | CDRECORDDEVICENR<1 | CDRECORDDEVICENR>NRDEVICES then
                    call error 'Illegal CDRECORDDEVICENR='CDRECORDDEVICENR 'specification, bootAble found' NRDEVICES 'writer devices'
                DEVICENR=CDRECORDDEVICENR
            end
        end
        if DEVICENR>0 then
            DEVICE=DEVICES.DEVICENR
        else do
            call warningMessage 'Unable to automatically determine CD writer device'||CRLF||,
                '  please specify CDRECORDDEVICE manually'
            COMMENT='; CD Writer device to use (bus,id,lun)'
            VARNAME='CDRECORDDEVICE'
            CDRECORDDEVICE=askText(DEVICE)
        end
    end
    return DEVICE


/* ------------------------------------------------------------------------ */
/* Return the dvd recorder device parameter for dvddao                      */
/* ARG: None                                                                */
/* RET: dvddao device parameter                                             */
/* ------------------------------------------------------------------------ */
dvdRecorderDevice: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CREATEBOOTCD CDRECORDDVDDAODEVICENR CDRECORDDVDDAODEVICE CDRECORD CDRECORDMEDIA CDRECORDEXE CDRECORDCREATEISOONLY CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    DEVICE='0,1,0'
    NRDEVICES=0
    DEVICENR=0
    if isSet(CREATEBOOTCD) & isSet(CDRECORD) & CDRECORDMEDIA>0 & \isSet(CDRECORDCREATEISOONLY) then do
        if  fileExists(CDRECORDEXE)<>'' then do
            QUEUE=rxqueue('CREATE')
            rc=rxqueue('SET',QUEUE)
            rc=runCMD(CDRECORDEXE '-scanbus | rxqueue' QUEUE)
            do while queued()>0
                parse pull DEV NR')' "'"BRAND"'" "'"MODEL"'" "'"VERSION"'" DESCRIPTION
                DESCRIPTION=strip(DESCRIPTION)
                if pos('DVD',MODEL)=1 | pos('RW/DVD',MODEL)=1 then do
                    DEV=strip(DEV)
                    DEV=strip(DEV,,'	')
                    NRDEVICES=NRDEVICES+1
                    DEVICES.NRDEVICES=DEV
                    DEVICESDESCRIPTION.NRDEVICES="'"BRAND"'" "'"MODEL"'" "'"VERSION"'" DESCRIPTION
                end
            end
            rc=rxqueue('DELETE', QUEUE)
            if rc<>0 then
                call error 'Failed to delete REXX queue' QUEUE '('rc')'
            if NRDEVICES=0 | NRDEVICES=1
               then DEVICENR=NRDEVICES
            else do
                if CDRECORDVDDAODDEVICENR='' & isSet(CDRECORDCREATEISOONLY) then
                    CDRECORDDVDDAODEVICENR=1
                if CDRECORDDVDDAODEVICENR='' then do
                    COMMENT='; DVD Writer device number to use'||CRLF||,
                        '; The following DVD writer devices were found:'
                    do I=1 to NRDEVICES
                        COMMENT=COMMENT||CRLF'; 'I':	'DEVICES.I'	'DEVICESDESCRIPTION.I
                    end
                    VARNAME='CDRECORDDVDDAODEVICENR'
                    CDRECORDDVDDAODEVICENR=askNumeric(1,1,NRDEVICES)
                end
                if dataType(CDRECORDDVDDAODEVICENR)<>'NUM' | CDRECORDDVDDAODEVICENR<1 | CDRECORDDVDDAODEVICENR>NRDEVICES then
                    call error 'Illegal CDRECORDDVDDAODEVICENR='CDRECORDDVDDAODEVICENR 'specification, bootAble found' NRDEVICES 'writer devices'
                DEVICENR=CDRECORDDVDDAODEVICENR
            end
        end
        if DEVICENR>0 then
            DEVICE=DEVICES.DEVICENR
        else do
            call warningMessage 'Unable to automatically determine DVD writer device'||CRLF||,
                '  please specify CDRECORDDVDDAODEVICE manually'
            COMMENT='; DVD Writer device to use (bus,id,lun)'
            VARNAME='CDRECORDDVDDAODEVICE'
            CDRECORDDVDDAODEVICE=askText(DEVICE)
        end
    end
    return DEVICE


/* ------------------------------------------------------------------------ */
/* Determine the SCSI setup                                                 */
/* ARG: None                                                                */
/* RET: 0                                                                   */
/* ------------------------------------------------------------------------ */
determineSCSI: procedure expose IDE IDEADDFILE SCSI SCSIBEFOREIDE SCSIADDFILE. SCSIOPTIONS. SCSIADAPTERLIST CONFIGSYSFILE CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    SCSI=0
    SCSIBEFOREIDE='NO'
    rc=parseFileList(SCSIADAPTERLIST)
    MINLINE=0
    MAXLINE=0
    do I=1 to FILELIST.0
        FILENAME=fileName(FILELIST.I)
        FILELIST_LINENR.I=configSysCount('BASEDEV=', FILENAME, 'NOCOUNT')
        FILELIST_OPTIONS.I=configSysOptions('BASEDEV=', FILENAME)
        if FILELIST_LINENR.I>0 & (MINLINE=0 | FILELIST_LINENR.I<MINLINE) then
            MINLINE=FILELIST_LINENR.I
        MAXLINE=max(MAXLINE,FILELIST_LINENR.I)
        if FILELIST_LINENR.I>0 then
            SCSI=SCSI+1
    end

    if isSet(IDE) & SCSI>0 & MINLINE<configSysCount('BASEDEV=', IDEADDFILE, 'NOCOUNT') then
        SCSIBEFOREIDE='YES'

    NEXTLINE=0
    do I=1 to SCSI
        do II=1 to FILELIST.0
            if FILELIST_LINENR.II=MINLINE then do
                SCSIADDFILE.I=FILELIST.II
                SCSIOPTIONS.I=FILELIST_OPTIONS.II
            end
            if FILELIST_LINENR.II>MINLINE then
                if NEXTLINE=0 then
                    NEXTLINE=FILELIST_LINENR.II
                else
                    NEXTLINE=min(NEXTLINE,FILELIST_LINENR.II)
        end
        MINLINE=NEXTLINE
        NEXTLINE=0
    end

    return 0


/* ------------------------------------------------------------------------ */
/* Determine the network card used                                          */
/* ARG: None                                                                */
/* RET: 0                                                                   */
/* ------------------------------------------------------------------------ */
determineNetworkCard: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. NETWORKCARDFILE OSBOOTDRIVE CONFIGSYSFILE CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    NRMATCHED=getFromConfigSysMulti('DEVICE='OSBOOTDRIVE'\ibmcom\macs\',1,,'IGNOREDOUBLE')
    if NRMATCHED=0 then call error 'No network card drivers found in' CONFIGSYSFILE||CRLF||'Specify one manually using the NETWORKCARDFILE setting in your bootAble.mycfg'
    if NRMATCHED=1 then
        NETWORKCARDFILE=OSBOOTDRIVE'\ibmcom\macs\'||MATCHEDLINES.1
    else do
        COMMENT='; Network card driver to use'||CRLF||,
            '; The following network card drivers where found:'
        do I=1 to NRMATCHED
            COMMENT=COMMENT||CRLF||'; 'I':	'MATCHEDLINES.I
        end
        VARNAME='NETWORKCARDFILE'
        NR=askNumeric(1,1,NRMATCHED)
        NETWORKCARDFILE=OSBOOTDRIVE'\ibmcom\macs\'||MATCHEDLINES.NR
        rc=addAskedValue("'"NETWORKCARDFILE"'",1)
    end

    return 0


/* ------------------------------------------------------------------------ */
/* Determine the GENMAC driver to use (e.g., WRND32$ WRND322$, etc.)        */
/* ARG: None                                                                */
/* RET: Exact name of the GENMAC driver on the system                       */
/* ------------------------------------------------------------------------ */
determineGENMACDriver: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. GENMACDRIVER OSBOOTDRIVE CONFIGSYSFILE CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    ORGPROTOCOLINI=fileExists(pathName(fileExists('protman.os2'))||'\protocol\protocol.ini')
    if ORGPROTOCOLINI='' then call error 'Unable to locate original protocol.ini'
    NRMATCHED=getFromFileMulti(ORGPROTOCOLINI,'DriverName = WRND32',1)
    if NRMATCHED=0 then call error 'No GENMAC network card drivers found in' ORGPROTOCOLINI||CRLF||'Specify one manually using the GENMACDRIVER setting in your bootAble.mycfg'
    if NRMATCHED=1 then
        GENMACDRIVER='WRND32'MATCHEDLINES.1
    else do
        COMMENT='; GENMAC driver to use'||CRLF||,
            '; The following GENMAC drivers where found:'
        do I=1 to NRMATCHED
            COMMENT=COMMENT||CRLF||'; 'I':	WRND32'MATCHEDLINES.I
            MATCHED=0
            DONE=0
            TXT_VENDOR=''
            TXT_DEVICE=''
            TXT_DRIVER=''
            do while \DONE & lines(ORGPROTOCOLINI)>0
                ORGLINE=space(lineIn(ORGPROTOCOLINI))
                if MATCHED>0 | pos('DriverName = WRND32'MATCHEDLINES.I,ORGLINE)>0 then
                    MATCHED=MATCHED+1
                if MATCHED>1 then
                    if strip(ORGLINE)='' then
                        DONE=1
                    else if pos('VENDOR = ', ORGLINE)>0 then
                        TXT_VENDOR=strip(substr(ORGLINE,10,6))
                    else if pos('DEVICE = ', ORGLINE)>0 then
                        TXT_DEVICE=strip(substr(ORGLINE,10,6))
                    else if pos('NDIS_INF = ',ORGLINE)>0 then do
                        TXT_DRIVER=strip(substr(ORGLINE,12),,'"')
                        TXT_DRIVER=left(TXT_DRIVER,length(TXT_DRIVER)-4)
                    end
            end
            COMMENT=COMMENT||' 	['TXT_VENDOR':'TXT_DEVICE']	'TXT_DRIVER
            rc=stream(ORGPROTOCOLINI, 'C', 'CLOSE')
        end
        VARNAME='GENMACDRIVER'
        NR=askNumeric(1,1,NRMATCHED)
        GENMACDRIVER='WRND32'MATCHEDLINES.NR
        rc=addAskedValue("'"GENMACDRIVER"'",1)
    end

    return 0


/* ------------------------------------------------------------------------ */
/* Determine XWLAN ini entries                                              */
/* ARG: None                                                                */
/* RET: 0                                                                   */
/* ------------------------------------------------------------------------ */
determineXWLANIni: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT CRLF EXTRAUSERINI
    call progress 1, 'Determining XWLAN ini entries'

    APPS.1='Wireless LAN Monitor:Networks'
    APPS.2='Wireless LAN Monitor:Profiles'
    APPS.3='Wireless LAN Monitor:Properties'
    APPS.4='Wireless LAN Monitor:StreamProfiles'
    APPS.0=4

    do I=1 to APPS.0
        rc=sysIni('USER', APPS.I, 'All:', 'KEYS')
        if rc='ERROR:' then
            call error 'Failed to read user ini file'
        do J=1 to KEYS.0
            if KEYS.J<>'ScriptPath' & KEYS.J<>'LastBitmapDir' & KEYS.J<>'ProfilePath' then
                EXTRAUSERINI=EXTRAUSERINI||CRLF||'"'APPS.I'" "'KEYS.J'" 'getFromIni('USER',APPS.I,KEYS.J,'DELIM')
        end
    end

    return 0


/* ------------------------------------------------------------------------ */
/* Determine the drives with given file system                              */
/* ARG: File system                                                         */
/* RET: String consisting of all drive letters with given file system       */
/*        (autocheck format)                                                */
/* ------------------------------------------------------------------------ */
drivesWithFS: procedure
    FS=ARG(1)
    DRIVES=''
    DRIVEMAP=sysDriveMap(,'LOCAL')
    do I=1 to words(DRIVEMAP)
        DRIVE=word(DRIVEMAP,I)
        if sysFileSystemType(DRIVE)=FS then
            DRIVES=DRIVES||left(DRIVE,1)
    end
    return DRIVES


/* ------------------------------------------------------------------------ */
/* Determine if the user is using TCPIP 431                                 */
/* ARG: None                                                                */
/* RET: Yes or No                                                           */
/* ------------------------------------------------------------------------ */
usingTCPIP431: procedure expose SEARCHPATH
    SYSLEVELDHC=fileExists('syslevel.dhc')
    TCPIP431='NO'
    if SYSLEVELDHC<>'' then do
        COMPONENT=charin(SYSLEVELDHC,141,9)
        VERSION=c2x(charin(SYSLEVELDHC,41,2))
        CSDLEVEL=charin(SYSLEVELDHC,47,5)
        TCPIP431=ifClause(VERSION>='4301','YES','NO')
        rc=stream(SYSLEVELDHC,'C','CLOSE')
    end
    return TCPIP431


/* ------------------------------------------------------------------------ */
/* Determine list of snoop files                                            */
/* ARG: None                                                                */
/* RET: All snoop files including snoop.lst itself                          */
/* ------------------------------------------------------------------------ */
determineSnoopFiles: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT SEARCHPATH SNOOPLISTFILE SNOOPFILES
    FILE=fileExists(SNOOPLISTFILE)
    SNOOPFILES=FILE SNOOPFILES
    if FILE='' then
        call error 'Snoop file' SNOOPLISTFILE 'not found'
    do while lines(FILE)
        LINE=space(linein(FILE),1)
        if left(LINE,1) <> ';' then
        SNOOPFILES=SNOOPFILES LINE
    end
    rc=stream(FILE, 'C', 'CLOSE')
    return SNOOPFILES


/****************************************************************************/
/**** bootAble Questions Functions ******************************************/
/****************************************************************************/

/* ------------------------------------------------------------------------ */
/* Show introduction text for questions asked                               */
/* ARG: None                                                                */
/* RET: 0                                                                   */
/* ------------------------------------------------------------------------ */
askIntro: procedure expose NRASKED
    if NRASKED<>0 then return 0
    say
    say 'You will be asked to fill in the values for certain bootAble settings'
    say 'that can not be determined automatically.  The values you specify can'
    say 'be saved to a configuration file (e.g., bootAble.myCfg).  You can'
    say 'then use and customize this file for future use.'
    return 0


/* ------------------------------------------------------------------------ */
/* Ask YES/NO question                                                      */
/* ARG1: Default value                                                      */
/* ARG2: Ask?                                                               */
/* ARG3: Default value if not asked                                         */
/* ARG4: Do not save                                                        */
/* RET:  Answer value                                                       */
/* ------------------------------------------------------------------------ */
askYesNo: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. VARNAME COMMENT CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    DEFVAL=ARG(1)
    if ARG(2)<>'' then
        ASK=ARG(2)
    else
        ASK=1
    SAVEVALUE=ARG(4)=''
    if \ASK then
        if ARG(3)<>'' then
            VAL=ARG(3)
        else
            VAL=DEFVAL
    else do
        rc=askIntro()
        say CRLF||COMMENT
        VAL=yesNoQuit('Do you want to set' VARNAME,DEFVAL)
        rc=addAskedValue("'"VAL"'",SAVEVALUE)
    end
    return VAL


/* ------------------------------------------------------------------------ */
/* Ask text question                                                        */
/* ARG1: Default value                                                      */
/* ARG2: Type                                                               */
/* ARG3: Ask?                                                               */
/* ARG4: Default value if not asked                                         */
/* ARG5: Do not save                                                        */
/* RET:  Answer value                                                       */
/* ------------------------------------------------------------------------ */
askText: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. VARNAME COMMENT CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    DEFVAL=ARG(1)
    TYPE=ARG(2)
    if ARG(3)<>'' then
        ASK=ARG(3)
    else
        ASK=1
    SAVEVALUE=ARG(5)=''
    NONEMPTY=1
    if \ASK then
        if ARG(4)<>'' then
            VAL=ARG(4)
        else
            VAL=DEFVAL
    else do
        rc=askIntro()
        say CRLF||COMMENT
        select
            when TYPE='DRIVE'     then DESCRIPTION='drive letter'
            when TYPE='DIRECTORY' then DESCRIPTION='directory'
            when TYPE='FILE'      then DESCRIPTION='filename'
            otherwise do
                if TYPE<>'NONEMPTYTXT' then
                    NONEMPTY=0
                DESCRIPTION='text value'
            end
        end
        DEFSTR=ifClause(DEFVAL<>'',' ('DEFVAL')')
        rc=1
        do until rc=0
            rc=charout(,'Enter' DESCRIPTION 'for' VARNAME||DEFSTR': ')
            VAL=linein()
            if VAL='' then
                VAL=DEFVAL
            rc=NONEMPTY & VAL=''
            if rc=0 & TYPE='DRIVE' then do
                VAL=translate(VAL)
                DRVCODE=c2d(left(VAL,1))
                if length(VAL)=1 then VAL=VAL||':'
                if length(VAL)<>2 | right(VAL,1)<>':' | DRVCODE<c2d('A') | DRVCODE>c2d('Z') then
                    rc=1
            end
        end
        rc=addAskedValue("'"VAL"'",SAVEVALUE)
    end
    return VAL


/* ------------------------------------------------------------------------ */
/* Ask multiline text question                                              */
/* ARG1: Default value                                                      */
/* ARG2: Ask?                                                               */
/* ARG3: Add CRLF between lines?                                            */
/* ARG4: Do not save                                                        */
/* RET:  Answer value                                                       */
/* ------------------------------------------------------------------------ */
askMultiText: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. VARNAME COMMENT CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    DEFVAL=ARG(1)
    if ARG(2)<>'' then
        ASK=ARG(2)
    else
        ASK=1
    if ARG(3)<>'' then
        ADDCRLF=0
    else
        ADDCRLF=1
    SAVEVALUE=ARG(4)=''
    NONEMPTY=1
    if \ASK then
        VAL=DEFVAL
    else do
        rc=askIntro()
        say CRLF||COMMENT
        DEFSTR=ifClause(DEFVAL<>'',' ('DEFVAL')')
        rc=1
        do until rc=0
            say 'Enter multi-line text value for' VARNAME||DEFSTR
            say 'Use a blank line to terminate the input' ifClause(DEFSTR<>'', '(a blank first line accepts the default value)')
            rc=1
            LINENR=0
            TXTVAL=''
            VAL=''
            do while rc
                LINEVAL=linein()
                if LINEVAL='' then do
                    if LINENR=0 then do
                        VAL=DEFVAL
                        TXTVAL="'"escapeQuotes(DEFVAL)"'"
                    end
                    rc=0
                end
                else do
                    LINEVAL=escapeQuotes(LINEVAL)
                    if LINENR>0 then do
                        VAL=VAL||ifClause(ADDCRLF,CRLF,' ')
                        TXTVAL=TXTVAL||ifClause(ADDCRLF,'||CRLF||')||','||CRLF||'    '
                    end
                    VAL=VAL||LINEVAL
                    TXTVAL=TXTVAL"'"LINEVAL"'"
                    LINENR=LINENR+1
                    rc=1
                end
            end
        end
        rc=addAskedValue(TXTVAL,SAVEVALUE)
    end
    return VAL


/* ------------------------------------------------------------------------ */
/* Ask numeric question                                                     */
/* ARG1: Default value                                                      */
/* ARG2: Minimum value                                                      */
/* ARG3: Maximum value                                                      */
/* ARG4: Ask?                                                               */
/* ARG5: Default value if not asked                                         */
/* ARG6: Do not save                                                        */
/* RET:  Answer value                                                       */
/* ------------------------------------------------------------------------ */
askNumeric: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. VARNAME COMMENT CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    DEFVAL=ARG(1)
    MINVAL=ARG(2)
    MAXVAL=ARG(3)
    SAVEVALUE=ARG(6)=''
    BETWEENSTR=''
    if MINVAL<>'' then BETWEENSTR='>='MINVAL
    if MAXVAL<>'' then do
        if BETWEENSTR<>'' then
            BETWEENSTR=BETWEENSTR' & '
        BETWEENSTR=BETWEENSTR'<='MAXVAL
    end
    DEFSTR=ifClause(DEFVAL<>'',' ('DEFVAL')')
    if ARG(4)<>'' then
        ASK=ARG(4)
    else
        ASK=1
    if \ASK then
        if ARG(5)<>'' then
            VAL=ARG(5)
        else
            VAL=DEFVAL
    else do
        rc=askIntro()
        say CRLF||COMMENT
        rc=1
        do until rc=0
            rc=charout(,'Enter numeric value' BETWEENSTR 'for' VARNAME||DEFSTR': ')
            VAL=linein()
            if VAL='' then
                VAL=DEFVAL
            rc=datatype(VAL)<>'NUM'
            if rc=0 & MINVAL<>'' then
                rc=VAL<MINVAL
            if rc=0 & MAXVAL<>'' then
                rc=VAL>MAXVAL
            if rc then
                call beep 524,100
        end
        rc=addAskedValue("'"VAL"'",SAVEVALUE)
    end
    return VAL


/* ------------------------------------------------------------------------ */
/* Add an asked value to the repository                                     */
/* ARG1: Value (properly quoted)                                            */
/* ARG2: Save value?                                                        */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
addAskedValue: procedure expose NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. VARNAME COMMENT
    NRASKED=NRASKED+1
    if \ARG(2) then
        return 0
    INDEXASKED=0
    if translate(VARNAME)<>'ADDTOSEARCHPATH' & \abbrev(VARNAME,'_ADDBEFORE_') & \abbrev(VARNAME,'_ADDAFTER_') then do I=1 to NRASKEDSAVED
        if abbrev(ASKEDLINES.I, translate(VARNAME)'=') then do
            INDEXASKED=I
            leave
        end
    end
    if INDEXASKED=0 then do
        NRASKEDSAVED=NRASKEDSAVED+1
        ASKEDLINES.0=NRASKEDSAVED
        INDEXASKED=NRASKEDSAVED
    end
    ASKEDLINES.INDEXASKED=''||translate(VARNAME)'='ARG(1)
    ASKEDCOMMENT.INDEXASKED=COMMENT
    return 0


/* ------------------------------------------------------------------------ */
/* Write asked lines to file                                                */
/* ARG1: File to write to                                                   */
/* ARG2: Optional additional text to go at the end of the file              */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
writeAskedLinesToFile: procedure expose CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    FILE=ARG(1)
    TXT=fileName(FILE) 'created by bootAble.cmd on' date() time()
    LEN=length(TXT)
    if LEN<78-6 then do
        LEN=78-6
        TXT=left(TXT,LEN)
    end
    TXTLINE='/* 'TXT' */'
    SPACELINE='/* 'left(' ',LEN)' */'
    STARLINE=translate(SPACELINE,'*',' ')
    TXT=STARLINE||CRLF||SPACELINE||CRLF||TXTLINE||CRLF||SPACELINE||CRLF||STARLINE||CRLF||CRLF
    rc=createFile(FILE, TXT)
    do I=1 to NRASKEDSAVED
        rc=lineOut(FILE,'')
        rc=lineOut(FILE,ASKEDCOMMENT.I)
        rc=lineOut(FILE,ASKEDLINES.I)
    end
    if ARG(2)<>'' then do
        rc=lineOut(FILE,'')
        rc=lineOut(FILE,ARG(2))
    end
    rc=lineOut(FILE,'')
    rc=lineOut(FILE)
    return 0



/****************************************************************************/
/**** bootAble Admin Functions **********************************************/
/****************************************************************************/

/* ------------------------------------------------------------------------ */
/* Initialization                                                           */
/* ARG1: Command line arguments to bootAble                                 */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
initialize:
    /* Load the rexxUtil functions */
    call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
    call SysLoadFuncs

    /* Constants */
    _NO_='NO'
    _YES_='YES'
    CRLF=d2c(13)||d2c(10)

    /* Global initialization */
    OSBOOTDRIVE=sysBootDrive()
    CONFIGSYSFILE=OSBOOTDRIVE'\config.sys'
    EXTRADEVICE=0
    EXTRABASEDEV=0
    EXTRADIRECTORIES=0
    EXTRAZIPWPIS=0
    EXTRACREATEFILES=0
    EXTRAINIFILES=0
    EXTRAUSERINIRCFILES=1 /* Note: first one is WPSUSERINIRC */
    EXTRASYSINIRCFILES=1  /* Note: first one is WPSSYSINIRC  */
    REXXEXIT_CLEANUP=''

    APPDEST_ROOTDEFAULT='\PROGRAMS'

    /* Searchpath */
    SEARCHPATH=OSBOOTDRIVE'\os2\boot;'OSBOOTDRIVE'\ecs\boot;'getFromConfigSys('LIBPATH=')';'value('PATH',,'OS2ENVIRONMENT')';'value('DPATH',,'OS2ENVIRONMENT')';'value('HELP',,'OS2ENVIRONMENT')';'value('BOOKSHELF',,'OS2ENVIRONMENT')';'OSBOOTDRIVE'\psfonts;'OSBOOTDRIVE'\psfonts\pfm;'
    rc=checkSearchPath()
    rc=value('SEARCHPATH',SEARCHPATH,'OS2ENVIRONMENT')
    PREVSEARCHPATH=SEARCHPATH
    ORGSEARCHPATH=SEARCHPATH

    /* Internal settings */
    NRASKED=0
    NRASKEDSAVED=0
    CREATECLONECDCMDLINEARGS=''
    CDRECORDMKISOFSLIST=''

    /* Verbose settings */
    CLEANUP='NO'
    IGNOREOUTPUT=''
    ARGUMENTS=ARG(1)
    ARG1=translate(word(ARGUMENTS,1))
    ARG2=word(ARGUMENTS,2)
    VERBOSE=65
    if (ARG1='/V' | ARG1='-V') then do
        if datatype(ARG2)='NUM' then do
            VERBOSE=ARG2
            ARGUMENTS=delword(ARGUMENTS,1,2)
        end
        else
            call error 'Usage: bootAble [/v <verbose>] [<config file>|/@<config-listfile>...]'
    end

    /* Determine if command output should be shown */
    select
        when bitMask(VERBOSE,128) then IGNOREOUTPUT=''
        when bitMask(VERBOSE,64)  then IGNOREOUTPUT='> /dev/nul'
        otherwise                      IGNOREOUTPUT='> /dev/nul 2>&1'
    end

    /* Create temp directory */
    TEMP=value('TEMP',,'OS2ENVIRONMENT')
    if TEMP='' then TEMP=value('TMP',,'OS2ENVIRONMENT')
    if right(TEMP,1)<>'\' then TEMP=TEMP'\'
    BOOTABLETEMP=TEMP'bootAble.###'
    rc=createDir(BOOTABLETEMP)
    rc=cleanDir(BOOTABLETEMP)
    rc=createDir(BOOTABLETEMP'\XCOPY.###')
    rc=cleanDir(BOOTABLETEMP'\XCOPY.###')
    CLEANUP='YES'

    /* Read the configuration file(s) */
    rc=readConfig(ARGUMENTS)

    /* Check the configuration */
    rc=checkConfig()

    /* Write the configuration questions */
    rc=writeConfig()

    return 0


/* ------------------------------------------------------------------------ */
/* Finalize CD / DVD / ISO image                                            */
/* ARG: None                                                                */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
finalizeImage:
    if isSet(CREATEBOOTCD) & isSet(PAUSEBEFOREFINALIZING) then do
        say 'Pausing before finalizing'
        if \isSet(yesNoQuit('Continue with script?','Yes','NOCLEAN')) then
            call error 'ABORT: Script aborted', 'NOPAUSE'
    end

    /* Finalize CD/DVD when using RSJ */
    if isSet(CREATEBOOTCD) & \isSet(CDRECORD) & isSet(AUTOBURNCD) then do
        call progress 1, 'Finalizing CD/DVD using RSJ'
        rc=runCMD(RSJCDATTACHEXE WRITERDRIVE '-s', 'Failed to finalize CD/DVD in drive' WRITERDRIVE)
    end

    /* Finalize CD/DVD / Create ISO image file when using CDRECORD */
    if isSet(CREATEBOOTCD) & isSet(CDRECORD) then do
        if isSet(AUTOBURNCD) then
            CDRECORDMKISOFSLISTFILE=BOOTABLETEMP'\mkisofs.fsl'
        else
            CDRECORDMKISOFSLISTFILE='burnCD.fsl'
        if fileExists(CDRECORDMKISOFSLISTFILE,'Do not Search Path')<>'' then do
            rc=sysFileDelete(CDRECORDMKISOFSLISTFILE)
            if rc<>0 then
                call error 'Failed to remove existing' CDRECORDMKISOFSLISTFILE '('rc')'
        end
        do while CDRECORDMKISOFSADDITIONALISOFSFILES<>''
            IDX=pos(CRLF,CDRECORDMKISOFSADDITIONALISOFSFILES)
            if IDX=0 then
                IDX=length(CDRECORDMKISOFSADDITIONALISOFSFILES)+1
            LISTSPEC=strip(left(CDRECORDMKISOFSADDITIONALISOFSFILES,IDX-1),'B')
            CDRECORDMKISOFSADDITIONALISOFSFILES=substr(CDRECORDMKISOFSADDITIONALISOFSFILES,IDX+2)
            if left(LISTSPEC,1)<>'@' then
                rc=addISOFSListSpec(LISTSPEC)
            else do
                LISTFILE=fileExists(substr(LISTSPEC,2))
                if LISTFILE='' then call error 'List file' substr(LISTSPEC,2) 'not found'
                call progress 2, 'Adding ISOFS List file specifications from' LISTFILE
                do while lines(LISTFILE)
                    LISTSPEC=linein(LISTFILE)
                    rc=addISOFSListSpec(LISTSPEC)
                end
            rc=stream(LISTFILE, 'C', 'CLOSE')
            if rc<>'READY:' then call error 'Failed to close' LISTFILE '('rc')'
            end
        end
        if translate(fileName(CDRECORDMKISOFSEXE))='MKISOFS.EXE' then
            CDRECORDMKISOFSLIST=transLate(CDRECORDMKISOFSLIST)
        rc=createFile(CDRECORDMKISOFSLISTFILE, CDRECORDMKISOFSLIST)


        if isSet(CDRECORDBURNCMD) | isSet(AUTOBURNCD) then do
            if \isSet(AUTOBURNCD) & fileExists('burnCD.cmd','Do not Search Path')<>'' then do
                rc=sysFileDelete('burnCD.cmd')
                if rc<>0 then
                    call error 'Failed to remove existing burnCD.cmd ('rc')'
            end

            /* mkisofs command-line */
            CMD=CDRECORDMKISOFSEXE CDRECORDMKISOFSOPTIONS '-V' VOLUMELABEL '-path-list' CDRECORDMKISOFSLISTFILE ifSet(MEMDISK,'-no-emul-boot -b BOOTIMGS/cdloader.bin','-b' BOOTIMAGENAME.1)

            /* determine size of cd/dvd image */
            QUEUE=rxqueue('CREATE')
            rc=rxqueue('SET',QUEUE)
            rc=runCMD(CMD '-print-size' WRITERDRIVE '| rxqueue' QUEUE)
            if rc<>0 then call error 'Failed to determine ISOSIZE, check mkisofs/cdrecord parameters'
            do while queued()>0
                parse pull ISOSIZE
            end
            rc=rxqueue('DELETE', QUEUE)
            if rc<>0 then call error 'Failed to delete REXX queue' QUEUE '('rc')'

            ISOSIZEHUMAN=ISOSIZE*2 /* Iso Size is in blocks of 2048 bytes */
            if ISOSIZEHUMAN<1024 then
                ISOSIZEHUMAN=ISOSIZEHUMAN'kiB'
            else if ISOSIZEHUMAN<10240 then
                ISOSIZEHUMAN=format(ISOSIZEHUMAN/1024,,2)'MiB'
            else if ISOSIZEHUMAN<102400 then
                ISOSIZEHUMAN=format(ISOSIZEHUMAN/1024,,1)'MiB'
            else if ISOSIZEHUMAN<1024*1024 then
                ISOSIZEHUMAN=format(ISOSIZEHUMAN/1024,,0)'MiB'
            else if ISOSIZEHUMAN<1024*10240 then
                ISOSIZEHUMAN=format(ISOSIZEHUMAN/1024/1024,,2)'GiB'
            else if ISOSIZEHUMAN<1024*102400 then
                ISOSIZEHUMAN=format(ISOSIZEHUMAN/1024/1024,,1)'GiB'
            else
                ISOSIZEHUMAN=format(ISOSIZEHUMAN/1024/1024,,0)'GiB'

            if isSet(CDRECORDCREATEISOONLY) then
                COMMENTTXT='Creating bootAble ISO file using mkisofs.'
            else if CDRECORDMEDIA=0 then
                COMMENTTXT='Burning bootAble CD using mkisofs and cdrecord.'
            else
                COMMENTTXT='Burning bootAble DVD using mkisofs and dvddao.'
            COMMENTTXT=COMMENTTXT||' Image size is' ISOSIZEHUMAN
            TXT='@echo off'||CRLF||,
                'echo' COMMENTTXT||CRLF

            if isSet(AUTOBURNCD) then call progress 1, COMMENTTXT

            if CDRECORDMEDIA=0 then do
                if 2*ISOSIZE>700*1024 & \isSet(CDRECORDALLOWCD700PLUS) then
                    if CDRECORDALLOWCD700PLUS<>'' then
                        call error 'Resulting CD image size ('ISOSIZEHUMAN') is larger than 700MiB, creation aborted'
                    else do
                        QTXT='The resulting CD image size ('ISOSIZEHUMAN') is larger than 700MiB'||CRLF||,
                            'Note: Specify CDRECORDALLOWCD700PLUS if you do not want to be asked this'||CRLF||,
                            '  question again in the future'||CRLF||,
                            'Do you want to create the CD'ifSet(CDRECORDCREATEISOONLY,' image')' anyway?'
                        if \isSet(yesNoQuit(QTXT,'NO','NOCLEAN')) then
                            call error 'ABORT: Creation aborted.', 'NOPAUSE'
                    end
                else if 2*ISOSIZE>650*1024 & \isSet(CDRECORDALLOWCD700PLUS) & \isSet(CDRECORDALLOWCD650PLUS) then
                    if CDRECORDALLOWCD650PLUS<>'' then
                        call error 'Resulting CD image size ('ISOSIZEHUMAN') is larger than 650MiB, creation aborted'
                    else do
                        QTXT='The resulting CD image size ('ISOSIZEHUMAN') is larger than 650MiB'||CRLF||,
                            'Note: Specify CDRECORDALLOWCD650PLUS if you do not want to be asked this'||CRLF||,
                            '  question again in the future'||CRLF||,
                            'Do you want to create the CD'ifSet(CDRECORDCREATEISOONLY,' image')' anyway?'
                        if \isSet(yesNoQuit(QTXT,'NO','NOCLEAN')) then
                            call error 'ABORT: Creation aborted.', 'NOPAUSE'
                    end
            end
            else do
                if 2*ISOSIZE>8.5*1024*1024 & \isSet(CDRECORDALLOWDVD85PLUS) then
                    if CDRECORDALLOWDVD85PLUS<>'' then
                        call error 'Resulting DVD image size ('ISOSIZEHUMAN') is larger than 8.5GiB, creation aborted'
                    else do
                        QTXT='The resulting DVD image size ('ISOSIZEHUMAN') is larger than 8.5GiB'||CRLF||,
                            'Note: Specify CDRECORDALLOWDVD85PLUS if you do not want to be asked this'||CRLF||,
                            '  question again in the future'||CRLF||,
                            'Do you want to create the DVD'ifSet(CDRECORDCREATEISOONLY,' image')' anyway?'
                        if \isSet(yesNoQuit(QTXT,'NO','NOCLEAN')) then
                            call error 'ABORT: Creation aborted.', 'NOPAUSE'
                    end
                else if 2*ISOSIZE>4.7*1024*1024 & \isSet(CDRECORDALLOWDVD85PLUS) & \isSet(CDRECORDALLOWDVD47PLUS) then
                    if CDRECORDALLOWDVD47PLUS<>'' then
                        call error 'Resulting DVD image size ('ISOSIZEHUMAN') is larger than 4.7GiB, creation aborted'
                    else do
                        QTXT='The resulting DVD image size ('ISOSIZEHUMAN') is larger than 4.7GiB'||CRLF||,
                            'Note: Specify CDRECORDALLOWDVD47PLUS if you do not want to be asked this'||CRLF||,
                            '  question again in the future'||CRLF||,
                            'Do you want to create the DVD'ifSet(CDRECORDCREATEISOONLY,' image')' anyway?'
                        if \isSet(yesNoQuit(QTXT,'NO','NOCLEAN')) then
                            call error 'ABORT: Creation aborted.', 'NOPAUSE'
                    end
            end

            if isSet(CDRECORDCREATEISOONLY) then do
                if CDRECORDMEDIA>0 & ISOSIZE>524288 then
                    CMD=CMD '-split-output'
                CMD=CMD '-o' CDRECORDISOIMAGENAME WRITERDRIVE
            end
            else if CDRECORDMEDIA=0 then
                CMD=CMD WRITERDRIVE '|' CDRECORDEXE ifSet(ERASECD,CDRECORDERASEOPTION) CDRECORDEXEOPTIONS '-'
            else
                CMD=CMD WRITERDRIVE '|' CDRECORDDVDDAOEXE ifSet(ERASECD,CDRECORDDVDDAOERASEOPTION) CDRECORDDVDDAOEXEOPTIONS '--size' ISOSIZE's' '-'

            if isSet(AUTOBURNCD) then
                rc=runCMD(CMD, 'Failed' COMMENTTXT)
            else do
                if CDRECORDMEDIA>0 then
                    TXT=TXT||CRLF||,
                        'REM If you change the content of the DVD before you burn it, you may need to'||CRLF||,
                        'REM change the --size option of below command-line!'||CRLF||,
                        'REM You can determine the correct size with the -print-size option of mkisofs'||CRLF
                TXT=TXT||CRLF||CMD||CRLF
                If isSet(CDRECORDCREATEISOONLY) then
                    if CDRECORDMEDIA=0 then
                        TXT=TXT||CRLF||,
                            'REM To burn the CD use e.g. the following command (specify in the correct device)'||CRLF||,
                            'REM 'CDRECORDEXE ifSet(ERASECD,CDRECORDERASEOPTION) CDRECORDEXEOPTIONS CDRECORDISOIMAGENAME
                    else
                        TXT=TXT||CRLF||,
                            'REM To burn the DVD use e.g. the following command (specify the correct device)'||CRLF||,
                            'REM ' CDRECORDDVDDAOEXE ifSet(ERASECD,CDRECORDDVDDAOERASEOPTION) CDRECORDDVDDAOEXEOPTIONS '--size' ISOSIZE's' CDRECORDISOIMAGENAME
            end

            if isSet(CDRECORD) & isSet(AUTOBURNCD) then do
                if isSet(CDRECORDCLEANDIRAFTER) then do
                    rc=cleandir(WRITERDRIVE)
                    rc=sysRmDir(WRITERDRIVE)
                end
                else
                    call warningMessage 'Directory' WRITERDRIVE 'still contains the (temporary) image files!'
            end
            else do
                TXT=TXT||CRLF||CRLF||,
                    'REM Remember that' WRITERDRIVE 'still contains the (temporary) image files after running this command!'
                rc=createFile('burnCD.cmd',TXT)
            end
        end
        else do
            TXT='You chose not to create an ISO file or burn it.'||CRLF||,
                '  If you want to create a CD/DVD yourself, the CD/DVD image files can be found in'||CRLF||,
                '    'WRITERDRIVE||CRLF||,
                '  Make sure you also include the parameter:'||CRLF||,
                '    -path-list' CDRECORDMKISOFSLISTFILE||CRLF||,
                '  on the mkisofs command-line to include the additional files specified there'||CRLF||,
                '  as well.'||CRLF||,
                '  Of course, you could also run the program again, selecting to burn a CD/DVD,'||CRLF||,
                '  or create an ISO file.'
            call warningMessage TXT
        end
    end
    return 0


/* ------------------------------------------------------------------------ */
/* Read and parse the configuration file(s)                                 */
/* ARG1: The configuration files to read                                    */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
readConfig:
    rc=parseFileList(ARGUMENTS)
    /* Process all specified configuration files */
    do I=1 to FILELIST.0
        CREATECLONECDCMDLINEARGS=CREATECLONECDCMDLINEARGS||' "'FILELIST.I'"'
        rc=readConfigFile(FILELIST.I)
    end

    /* Process the defaults */
    rc=readConfigFile('bootAble.cfg', 'YES')

    return 0


/* Check the configuration ------------------------------------------------ */
/* ARG: None                                                                */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
checkConfig:
    UNZIPREQUIRED=0
    WICREQUIRED=0

    /* Make sure the RESERVEDRIVELETTER has no trailing ':' */
    RESERVEDRIVELETTER=translate(left(RESERVEDRIVELETTER,1))

    /* Enforce dependant settings */
    if isSet(JJSCDROM) then
        CDROM='YES'
    if isSet(IBMIDE) then
        IDE='YES'
    if isSet(DANIIDE) then
        IDE='YES'
    if isSet(AHCI) then
        IDE='YES'
    if isSet(DANIATAPI) then
        ATAPI='YES'
    if isSet(JFS) then
        UNICODE='YES'

    /* Force destination settings */
    OS2APPSDESTINATION=ifClause(translate(OS2APPSDESTINATION)='IMG','IMG','CD')
    OS2EXTRADESTINATION=ifClause(translate(OS2EXTRADESTINATION)='IMG','IMG','CD')
    REXXDESTINATION=ifClause(translate(REXXDESTINATION)='IMG','IMG','CD')
    WPSDESTINATION=ifClause(translate(WPSDESTINATION)='IMG','IMG','CD')
    WPSHELPDESTINATION=ifClause(translate(WPSHELPDESTINATION)='IMG','IMG','CD')
    WPSFONTSDESTINATION=ifClause(translate(WPSFONTSDESTINATION)='IMG','IMG','CD')
    WPSAPPSDESTINATION=ifClause(translate(WPSAPPSDESTINATION)='IMG','IMG','CD')
    WPSEPMDESTINATION=ifClause(translate(WPSEPMDESTINATION)='IMG','IMG','CD')
    WPSREGEDIT2DESTINATION=ifClause(translate(WPSREGEDIT2DESTINATION)='IMG','IMG','CD')
    NETWORKINGDESTINATION=ifClause(translate(NETWORKINGDESTINATION)='IMG','IMG','CD')
    GENMACDESTINATION=ifClause(translate(GENMACDESTINATION)='IMG','IMG','CD')
    XWLANDESTINATION=ifClause(translate(XWLANDESTINATION)='IMG','IMG','CD')
    EXTRAFILESDESTINATION=ifClause(translate(EXTRAFILESDESTINATION)='IMG','IMG','CD')
    EXTRADLLFILESDESTINATION=ifClause(translate(EXTRADLLFILESDESTINATION)='IMG','IMG','CD')
    EXTRADEVICEDESTINATION=ifClause(translate(EXTRADEVICEDESTINATION)='IMG','IMG','CD')

    /* Force ACPI settings */
    if \isSet(ACPI) then do
        ACPIAPMADD='NO'
        ACPIDAEMON='NO'
    end

    /* Force serial port settings */
    if isSet(SIO2KSERIALPORT) then do
        SIOSERIALPORT='NO'
        SERIALPORT='NO'
    end
    else do
        SIO2KPATCH='NO'
        if isSet(SIOSERIALPORT) then
            SERIALPORT='NO'
    end

    /* Force MEMDISK settings */
    if \isSet(CREATEBOOTCD) then
        MEMDISK='NO'
    if isSet(MEMDISK) then do
            if fileExists(MEMDISKTAREXE)='' then call error 'Required executable MEMDISKTAREXE ('MEMDISKTAREXE') does not exist'
            QUEUE=rxqueue('CREATE')
            rc=rxqueue('SET',QUEUE)
            rc=runCMD(MEMDISKTAREXE '--version 2>&1 | rxqueue' QUEUE,,'IGNOREERROR')
            do while queued()>0
                parse pull VERSIONLINE
                if pos('1.18', VERSIONLINE)<>0 then call error 'Incorrect version 1.18 of tar executable MEMDISKTAREXE ('MEMDISKTAREXE'), please see bootAble documentation.'
            end
            rc=rxqueue('DELETE', QUEUE)
            if rc<>0 then
                call error 'Failed to delete REXX queue' QUEUE '('rc')'

        if isSet(RAMDISK) & translate(RAMDISKDRIVE)=translate(MEMDISKDRIVE) then
            call error 'Both the RAMDISKDRIVE and MEMDISKDRIVE point to the same drive' translate(MEMDISKDRIVE)
        if \isSet(RAMDISK) then
            RAMDISKDRIVE=MEMDISKDRIVE
        VIRTUALDRIVE=BOOTABLETEMP'\MEMDISK'
        rc=createDir(VIRTUALDRIVE)
        TWODISKBOOT='NO'
    end

    /* Force SCSI settings */
    select
        when SCSI=0 then do
            SCSIDMD='NO'
            REMOVABLESCSI='NO'
        end
        when SCSI=-1 then do
            rc=determineSCSI()
        end
        otherwise do I=1 to SCSI
            rc=defaultIfNotSet('SCSIOPTIONS.'I,'')
        end
    end

    /* Force USB settings */
    if (USBOHCI>0 | USBUHCI>0 | USBEHCI>0) then
        USB='YES'
    else do
        USB='NO'
        USBKEYB='NO'
        USBCOM='NO'
        USBMOUSE='NO'
        USBPRINTER='NO'
        USBCDROM='NO'
        USBSTORAGE='NO'
    end

    /* VirtualBox */
    if isSet(VBOXADDITIONS) then do
        GCCRT='YES'
        EXTRADIRECTORIES=EXTRADIRECTORIES+1
        EXTRADIRECTORY.EXTRADIRECTORIES=VBOXADDITIONSDIRECTORY
        EXTRADIRECTORYDESTINATION.EXTRADIRECTORIES='CD'
        EXTRADIRECTORYFILES.EXTRADIRECTORIES=VBOXADDITIONSFILES
        EXTRADIRECTORYADDTOPATHS.EXTRADIRECTORIES=''
    end

    /* Air-Boot */
    if isSet(AIRBOOT) then do
        /* Add the setboot/setaboot.exe file to \OS2 */
        EXTRAFILES=EXTRAFILES AIRBOOTSETBOOTFILE
        /* The airboot2.exe and airboot.bin go into their own directory on the 'CD' */
        EXTRADIRECTORIES=EXTRADIRECTORIES+1
        EXTRADIRECTORY.EXTRADIRECTORIES=AIRBOOTDIRECTORY
        EXTRADIRECTORYDESTINATION.EXTRADIRECTORIES='CD'
        EXTRADIRECTORYFILES.EXTRADIRECTORIES=AIRBOOTFILES
        EXTRADIRECTORYADDTOPATHS.EXTRADIRECTORIES=''
        /* Create airboot2 command file in \OS2 so you can call airboot2 wherever you are */
        EXTRACREATEFILES=EXTRACREATEFILES+1
        EXTRACREATEFILEDIRECTORY.EXTRACREATEFILES='\OS2'
        EXTRACREATEFILEDESTINATION.EXTRACREATEFILES='CD'
        EXTRACREATEFILENAME.EXTRACREATEFILES='airboot2.cmd'
        EXTRACREATEFILECONTENT.EXTRACREATEFILES=''||,
            '@'destinationDrive('CD')||CRLF||,
            '@cd 'AIRBOOTDIRECTORY||CRLF||,
            '@airboot2.exe'||CRLF
    end


    /* EMX Runtime */
    if isSet(EMXRT) then do
        /* EMX doc directory and content */
        EXTRADIRECTORIES=EXTRADIRECTORIES+1
        EXTRADIRECTORY.EXTRADIRECTORIES=EMXRTDIRECTORY'\DOC'
        EXTRADIRECTORYDESTINATION.EXTRADIRECTORIES=EMXRTDESTINATION
        EXTRADIRECTORYFILES.EXTRADIRECTORIES=EMXRTDOCFILES
        EXTRADIRECTORYADDTOPATHS.EXTRADIRECTORIES=''
        EXTRADIRECTORYIGNORENOTFOUND.EXTRADIRECTORIES='Yes'

        /* EMX book directory and content */
        EXTRADIRECTORIES=EXTRADIRECTORIES+1
        EXTRADIRECTORY.EXTRADIRECTORIES=EMXRTDIRECTORY'\BOOK'
        EXTRADIRECTORYDESTINATION.EXTRADIRECTORIES=EMXRTDESTINATION
        EXTRADIRECTORYFILES.EXTRADIRECTORIES=EMXRTBOOKFILES
        EXTRADIRECTORYADDTOPATHS.EXTRADIRECTORIES='HELP BOOKSHELF'
        EXTRADIRECTORYIGNORENOTFOUND.EXTRADIRECTORIES='Yes'

        /* EMX bin directory and content */
        EXTRADIRECTORIES=EXTRADIRECTORIES+1
        EXTRADIRECTORY.EXTRADIRECTORIES=EMXRTDIRECTORY'\BIN'
        EXTRADIRECTORYDESTINATION.EXTRADIRECTORIES=EMXRTDESTINATION
        EXTRADIRECTORYFILES.EXTRADIRECTORIES=EMXRTBINFILES
        EXTRADIRECTORYADDTOPATHS.EXTRADIRECTORIES='PATH'

        /* EMX dll directory and content */
        EXTRADIRECTORIES=EXTRADIRECTORIES+1
        EXTRADIRECTORY.EXTRADIRECTORIES=EMXRTDIRECTORY'\DLL'
        EXTRADIRECTORYDESTINATION.EXTRADIRECTORIES=EMXRTDESTINATION
        EXTRADIRECTORYFILES.EXTRADIRECTORIES=EMXRTDLLFILES
        EXTRADIRECTORYADDTOPATHS.EXTRADIRECTORIES='LIBPATH'

        /* EMX etc directory and content */
        EXTRADIRECTORIES=EXTRADIRECTORIES+1
        EXTRADIRECTORY.EXTRADIRECTORIES=EMXRTDIRECTORY'\ETC'
        EXTRADIRECTORYDESTINATION.EXTRADIRECTORIES=EMXRTDESTINATION
        EXTRADIRECTORYFILES.EXTRADIRECTORIES=EMXRTETCFILES
        EXTRADIRECTORYADDTOPATHS.EXTRADIRECTORIES='DPATH'
        EXTRADIRECTORYIGNORENOTFOUND.EXTRADIRECTORIES='Yes'
    end

    /* GCC Runtime */
    if isSet(GCCRT) then do
        /* GCC dll directory and content */
        EXTRADIRECTORIES=EXTRADIRECTORIES+1
        EXTRADIRECTORY.EXTRADIRECTORIES=GCCRTDIRECTORY
        EXTRADIRECTORYDESTINATION.EXTRADIRECTORIES=GCCRTDESTINATION
        EXTRADIRECTORYFILES.EXTRADIRECTORIES=GCCRTDLLFILES
        EXTRADIRECTORYADDTOPATHS.EXTRADIRECTORIES='LIBPATH'
    end

    /* Force WPS settings */
    WPSDRIVE=WRITERDRIVE
    if isSet(WPS) then do
        if isSet(WPSXWP) then do
            /* XWorkplace files */
            EXTRADIRECTORIES=EXTRADIRECTORIES+1
            EXTRADIRECTORY.EXTRADIRECTORIES=WPSXWPDIRECTORY
            EXTRADIRECTORYDESTINATION.EXTRADIRECTORIES=WPSXWPDESTINATION
            EXTRADIRECTORYFILES.EXTRADIRECTORIES=WPSXWPSOURCE'\*.*'
            EXTRADIRECTORYADDTOPATHS.EXTRADIRECTORIES=''
            EXTRADIRECTORYRECURSE.EXTRADIRECTORIES='YES'

            /* lswhook.dll is required for LSwitcher */
            if fileExists(WPSXWPSOURCE'\plugins\xcenter\lswidget.dll')<>'' then
                EXTRADLLFILES=EXTRADLLFILES 'lswhook.dll'

            /* Do not autostart warpcenter */
            if isSet(WPSWCAUTOSTART) then call warningMessage('You have specified WPSWCAUTOSTART while also running XWorkplace.'||CRLF||'  This may result in both the WarpCenter and XCenter to start up.')
            WPSWCSTARTUP='NO'
        end
        if VIDEODRIVER=3 then do
            UNZIPREQUIRED=1
            OS2APPS='YES'
            if isSet(MSVMADDITIONS) then
                call warningMessage 'Microsoft Virtual Machine mouse integration does not work with SNAP SE'
            if isSet(CREATEBOOTCD) & \isset(MEMDISK) then
                call errorMessage 'SNAP SE only supported with the MEMDISK boot method'

            /* Create temp directorie for SNAP files */
            _TMPSNAP=BOOTABLETEMP'\SNAP.###'
            rc=createDir(_TMPSNAP)
            /* gradbase */
            rc=runCMD(UNZIPCOMMAND '-C -q -o -j -d '_TMPSNAP' 'SNAPSEDISTRIBUTIONFILE' gradbase', 'Failed to unzip SNAP SE gradbase')
            rc=runCMD('unpack2 '_TMPSNAP'\gradbase '_TMPSNAP, 'Failed to unpack SNAP SE gradbase')
            /* gradd */
            rc=runCMD(UNZIPCOMMAND '-C -q -o -j -d '_TMPSNAP' 'SNAPSEDISTRIBUTIONFILE' gradd', 'Failed to unzip SNAP SE gradd')
            rc=runCMD('unpack2 '_TMPSNAP'\gradd '_TMPSNAP, 'Failed to unpack SNAP SE gradd')
            /* gre2vman.dll */
            rc=runCMD(UNZIPCOMMAND '-C -q -o -j -d '_TMPSNAP' 'SNAPSEDISTRIBUTIONFILE' gre2vman.dll', 'Failed to unzip SNAP SE dll')
            /* inscfg32.dl_ */
            rc=runCMD(UNZIPCOMMAND '-C -q -o -j -d '_TMPSNAP' 'SNAPSEDISTRIBUTIONFILE' inscfg32.dl_', 'Failed to SNAP SE unzip ')
            rc=runCMD('unpack2 '_TMPSNAP'\inscfg32.dl_ '_TMPSNAP' /N:inscfg32.dll', 'Failed to unpack SNAP SE inscfg32')
            /* sddgradd */
            rc=runCMD(UNZIPCOMMAND '-C -q -o -j -d '_TMPSNAP' 'SNAPSEDISTRIBUTIONFILE' sddgradd', 'Failed to unzip SNAP SE sddgradd')
            rc=runCMD('unpack2 '_TMPSNAP'\sddgradd '_TMPSNAP, 'Failed to unpack SNAP SE sddgradd')
            /* sddgraph */
            rc=runCMD(UNZIPCOMMAND '-C -q -o -j -d '_TMPSNAP' 'SNAPSEDISTRIBUTIONFILE' sddgraph', 'Failed to unzip SNAP SE sddgraph')
            rc=runCMD('unpack2 '_TMPSNAP'\sddgraph '_TMPSNAP, 'Failed to unpack SNAP SE sddgraph')
            /* vga */
            rc=runCMD(UNZIPCOMMAND '-C -q -o -j -d '_TMPSNAP' 'SNAPSEDISTRIBUTIONFILE' vga', 'Failed to unzip SNAP SE vga')
            rc=runCMD('unpack2 '_TMPSNAP'\vga '_TMPSNAP, 'Failed to unpack SNAP SE vga')
            /* en\snap*mri */
            rc=runCMD(UNZIPCOMMAND '-C -q -o -j -d '_TMPSNAP' 'SNAPSEDISTRIBUTIONFILE' en/snap*mri', 'Failed to unzip SNAP SE snap*mri')
            rc=runCMD('unpack2 '_TMPSNAP'\snap*mri '_TMPSNAP, 'Failed to unpack SNAP SE snap*mri')
            /* sbcs\vgamri */
            rc=runCMD(UNZIPCOMMAND '-C -q -o -j -d '_TMPSNAP' 'SNAPSEDISTRIBUTIONFILE' en/vgamri', 'Failed to unzip SNAP SE vgamri')
            rc=runCMD('unpack2 '_TMPSNAP'\vgamri '_TMPSNAP, 'Failed to unpack SNAP SE vgamri')
            /* sbcs\vgacs */
            rc=runCMD(UNZIPCOMMAND '-C -q -o -j -d '_TMPSNAP' 'SNAPSEDISTRIBUTIONFILE' sbcs/vgacs', 'Failed to unzip SNAP SE vgacs')
            rc=runCMD('unpack2 '_TMPSNAP'\vgacs '_TMPSNAP, 'Failed to unpack SNAP SE vgacs')

            /* Add SNAP SE Files to searchpath */
            SEARCHPATH=_TMPSNAP';'SEARCHPATH
            rc=value('SEARCHPATH',SEARCHPATH,'OS2ENVIRONMENT')

            /* Add vacpprt.dll to VIDEODRIVERDLLFILES (only for snapse491.exe) */
            if fileExists('vacpprt.dll')<>'' then
                VIDEODRIVERDLLFILES.3=VIDEODRIVERDLLFILES.3 'vacpprt.dll'

            /* Add additional SNAP SE files to CD */
            EXTRADIRECTORIES=EXTRADIRECTORIES+1
            EXTRADIRECTORY.EXTRADIRECTORIES='\SNAP'
            EXTRADIRECTORYDESTINATION.EXTRADIRECTORIES=WPSDESTINATION
            EXTRADIRECTORYFILES.EXTRADIRECTORIES=SNAPSEFILES
            EXTRADIRECTORYADDTOPATHS.EXTRADIRECTORIES=''
            EXTRADIRECTORYRECURSE.EXTRADIRECTORIES='NO'

            EXTRADIRECTORIES=EXTRADIRECTORIES+1
            EXTRADIRECTORY.EXTRADIRECTORIES='\OS2\DRIVERS\SNAP'
            EXTRADIRECTORYDESTINATION.EXTRADIRECTORIES=WPSDESTINATION
            EXTRADIRECTORYFILES.EXTRADIRECTORIES=SNAPSEDRIVERSSNAPFILES
            EXTRADIRECTORYADDTOPATHS.EXTRADIRECTORIES=''
            EXTRADIRECTORYRECURSE.EXTRADIRECTORIES='NO'

            EXTRADIRECTORIES=EXTRADIRECTORIES+1
            EXTRADIRECTORY.EXTRADIRECTORIES='\OS2\DRIVERS\SNAP\CONFIG'
            EXTRADIRECTORYDESTINATION.EXTRADIRECTORIES=WPSDESTINATION
            EXTRADIRECTORYFILES.EXTRADIRECTORIES=SNAPSEDRIVERSSNAPCONFIGFILES
            EXTRADIRECTORYADDTOPATHS.EXTRADIRECTORIES=''
            EXTRADIRECTORYRECURSE.EXTRADIRECTORIES='NO'
        end
        if isSet(CREATEBOOTCD) then do
            if \isSet(MEMDISK) then
                RAMDISK='YES'
            WPSDRIVE=RAMDISKDRIVE
        end
        MOUSE='YES'
        SYSDLLS_LOADONETIME=translate(SYSDLLS_LOADONETIME)
        if isSet(REXX) & wordpos('REXXINIT', SYSDLLS_LOADONETIME)=0 then
            SYSDLLS_LOADONETIME=SYSDLLS_LOADONETIME 'REXXINIT'
        SYSDLLS_LOADPERPROCESS=translate(SYSDLLS_LOADPERPROCESS)
        if wordpos('PMCTLS', SYSDLLS_LOADPERPROCESS)=0 then
            SYSDLLS_LOADPERPROCESS=SYSDLLS_LOADPERPROCESS 'PMCTLS'
    end
    else do
        WPSBOOT='NO'
        WPSAPPS='NO'
        WPSEPM='NO'
        WPSREGEDIT2='NO'
        WPSXWP='NO'
        XWLAN='NO'
    end

    /* Force networking settings */
    if \isSet(NETWORKING) then do
        TCPIP='NO'
        NETBIOS='NO'
        NETBIOSTCPIP='NO'
        PEER='NO'
        PEERSTARTUP='NO'
        XWLAN='NO'
    end
    if isSet(PEER) & \(isSet(NETBIOS)|isSet(NETBIOSTCPIP)) then
        call error 'PEER support requires either NETBIOS or NETBIOSTCPIP to be enabled'
    if isSet(PEER) & isSet(CREATEBOOTCD) & \isSet(MEMDISK) then
        call error 'PEER support on a bootAble CD/DVD requires the MEMDISK boot method'
    if isSet(PEER) & (\isSet(WPS) | \isSet(WPSBOOT)) then
        call warningMessage 'PEER support *may* not work without the WPS'
    if isSet(NETBIOSTCPIP) then do
        TCPIP='YES'
        TCPIPLAN='YES'
    end
    if isSet(NETWORKING) & NETWORKCARDFILE='' then
        rc=determineNetworkCard()
    GENMAC=ifClause(isSet(NETWORKING) & fileName(NETWORKCARDFILE)='GENM32W.OS2','YES','NO')
    if isSet(GENMAC) & GENMACDRIVER='' then
        rc=determineGENMACDriver()

    /* GENMAC settings */
    if isSet(GENMAC) then do
        if GENMACSOURCE='' then call error 'Unable to determine the source of the GENMAC specific files, specify GENMACSOURCE manually'
        EXTRADIRECTORIES=EXTRADIRECTORIES+1
        EXTRADIRECTORY.EXTRADIRECTORIES='\GENMAC'
        EXTRADIRECTORYDESTINATION.EXTRADIRECTORIES=GENMACDESTINATION
        EXTRADIRECTORYFILES.EXTRADIRECTORIES=GENMACSOURCE'\*'
        EXTRADIRECTORYADDTOPATHS.EXTRADIRECTORIES=''
        EXTRADIRECTORYRECURSE.EXTRADIRECTORIES='YES'
    end

    /* XWLAN settings */
    if isSet(XWLAN) then do
        REXX='YES'
        if \isSet(TCPIP)     then call error 'XWLAN support requires TCPIP'
        if \isSet(TCPIPDHCP) then call error 'XWLAN support requires TCPIPDHCP'
        if \isSet(TCPIPLAN)  then call error 'XWLAN support requires TCPIPLAN'
        if \isSet(WPSXWP)    then XWLANXWP='NO'

        /* INI file settings */
        rc=determineXWLANIni()

        /* XWLAN Files */
        if isSet(XWLANXWP) then do
            EXTRAZIPWPIS=EXTRAZIPWPIS+1
            EXTRAZIPWPIDIRECTORY.EXTRAZIPWPIS=WPSXWPDIRECTORY'\Plugins'
            EXTRAZIPWPIFILE.EXTRAZIPWPIS=XWLANSRCZIP
            EXTRAZIPWPIOPTIONS.EXTRAZIPWPIS='xcenter\* -x xcenter\wlanstat.exe'
            EXTRAZIPWPIDESTINATION.EXTRAZIPWPIS=WPSXWPDESTINATION
            EXTRAZIPWPIADDTOPATHS.EXTRAZIPWPIS=''
        end
        else do
            EXTRAZIPWPIS=EXTRAZIPWPIS+1
            EXTRAZIPWPIDIRECTORY.EXTRAZIPWPIS=APPDEST_ROOTDEFAULT'\XWLAN'
            EXTRAZIPWPIFILE.EXTRAZIPWPIS=XWLANSRCZIP
            EXTRAZIPWPIDESTINATION.EXTRAZIPWPIS=XWLANDESTINATION
            EXTRAZIPWPIADDTOPATHS.EXTRAZIPWPIS=''
        end

        /* XWLAN Installation script */
        if \isSet(XWLANXWP) then do
            EXTRACREATEFILES=EXTRACREATEFILES+1
            EXTRACREATEFILEDIRECTORY.EXTRACREATEFILES=APPDEST_ROOTDEFAULT'\XWLAN'
            EXTRACREATEFILEDESTINATION.EXTRACREATEFILES=XWLANDESTINATION
            EXTRACREATEFILENAME.EXTRACREATEFILES='xwlaninstall.cmd'
            EXTRACREATEFILECONTENT.EXTRACREATEFILES=''||,
              "/* XWLAN Installation */"||CRLF||,
              "call RxFuncAdd    'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'"||CRLF||,
              "call SysLoadFuncs"||CRLF||,
              "/* Remove script from startup */"||CRLF||,
              "rc=sysDestroyObject('<XWLAN_INSTALL>')"||CRLF||,
              "/* Run XWLAN installation script */"||CRLF||,
              "'call" destinationDrive(XWLANDESTINATION)||APPDEST_ROOTDEFAULT"\XWLAN\INSTEXE.CMD'"||CRLF||,
              "/* Complex way to move WLAN folder to Programs and kill the open folder window */"||CRLF||,
              "rc=sysSetObjectData('<WLAN_FOLDER>','OBJECTID=<_WLAN_FOLDER>')"||CRLF||,
              "rc=sysCreateObject('WPFolder','Wireless LAN Monitor','<WP_PROGRAMSFOLDER>','OBJECTID=<WLAN_FOLDER>;')"||CRLF||,
              "rc=sysMoveObject('<WLAN_EXEC>','<WLAN_FOLDER>')"||CRLF||,
              "rc=sysMoveObject('<WLAN_HELP>','<WLAN_FOLDER>')"||CRLF||,
              "rc=sysMoveObject('<WLAN_BITMAP_FOLDER>','<WLAN_FOLDER>')"||CRLF||,
              "rc=sysDestroyObject('<_WLAN_FOLDER>')"||CRLF||,
              "rc=sysCreateObject('WPShadow','Wireless LAN Monitor','<WP_START>','SHADOWID=<WLAN_EXEC>;OBJECTID=<WLAN_EXECSHADOW>;')"||CRLF||,
              "/* Start the program */"||CRLF||,
              "rc=sysSetObjectData('<WLAN_EXEC>','OPEN=DEFAULT;')"||CRLF
        end
    end

    /* Check TCPIP settings */
    if \isSet(TCPIP) then do
        TCPIPDHCP='NO'
        TCPIP431='NO'
    end

    /* Add MS VM addition to front of startup.cmd */
    if isSet(MSVMADDITIONS) then
        STARTUPCMD='@start vmserv.exe daemon'||CRLF||STARTUPCMD

    /* Add VirtualBox addition to front of startup.cmd */
    if (isSet(VBOXADDITIONS) & isSet(VBOXSTARTDMN)) then
        STARTUPCMD='@'destinationDrive('CD')''VBOXADDITIONSDIRECTORY'\VBoxService.exe'||CRLF||STARTUPCMD

    /* Force OS2APPS setting */
    if isset(TCPIP) & isSet(CREATEBOOTCD) & (isSet(RAMDISK)|isSet(MEMDISK)) & NETWORKINGDESTINATION='CD' then
        OS2APPS='YES'

    /* Force extra directory settings */
    do I=1 to EXTRADIRECTORIES
        IDX=pos(":", EXTRADIRECTORY.I)
    if IDX<>0 then EXTRADIRECTORY.I=right(EXTRADIRECTORY.I, length(EXTRADIRECTORY.I)-IDX)
        if left(EXTRADIRECTORY.I,1)<>'\' then EXTRADIRECTORY.I='\'EXTRADIRECTORY.I
        EXTRADIRECTORYADDTOPATHS.I=translate(EXTRADIRECTORYADDTOPATHS.I)
        EXTRADIRECTORYADDTOPATHS.I=translate(EXTRADIRECTORYADDTOPATHS.I, '  ', ',;')
        EXTRADIRECTORYADDTOPATHS.I=strip(EXTRADIRECTORYADDTOPATHS.I)
        EXTRADIRECTORYDESTINATION.I=ifClause(translate(EXTRADIRECTORYDESTINATION.I)='IMG','IMG','CD')
    end

    /* Force extra zip/wpi settings */
    do I=1 to EXTRAZIPWPIS
        IDX=pos(":", EXTRAZIPWPIDIRECTORY.I)
        if IDX<>0 then EXTRAZIPWPIDIRECTORY.I=right(EXTRAZIPWPIDIRECTORY.I, length(EXTRAZIPWPIDIRECTORY.I)-IDX)
        if left(EXTRAZIPWPIDIRECTORY.I,1)<>'\' then EXTRAZIPWPIDIRECTORY.I='\'EXTRAZIPWPIDIRECTORY.I
        EXTRAZIPWPIADDTOPATHS.I=translate(EXTRAZIPWPIADDTOPATHS.I)
        EXTRAZIPWPIADDTOPATHS.I=translate(EXTRAZIPWPIADDTOPATHS.I, '  ', ',;')
        EXTRAZIPWPIADDTOPATHS.I=strip(EXTRAZIPWPIADDTOPATHS.I)
        EXTRAZIPWPIDESTINATION.I=ifClause(translate(EXTRAZIPWPIDESTINATION.I)='IMG','IMG','CD')
        if symbol('EXTRAZIPWPIOPTIONS.I')='LIT' then
            EXTRAZIPWPIOPTIONS.I=''
        if translate(right(EXTRAZIPWPIFILE.I,4))<>'.WPI' & \isSet(EXTRAZIPWPIISWPI.I) then do
            UNZIPREQUIRED=1
            EXTRAZIPWPIISWPI.I='No'
        end
        else do
            WICREQUIRED=1
            EXTRAZIPWPIISWPI.I='Yes'
        end
    end

    /* Force extra create file settings */
    do I=1 to EXTRACREATEFILES
        IDX=pos(":", EXTRACREATEFILEDIRECTORY.I)
        if IDX<>0 then EXTRACREATEFILEDIRECTORY.I=right(EXTRACREATEFILEDIRECTORY.I, length(EXTRACREATEFILEDIRECTORY.I)-IDX)
        if left(EXTRACREATEFILEDIRECTORY.I,1)<>'\' then EXTRACREATEFILEDIRECTORY.I='\'EXTRACREATEFILEDIRECTORY.I
        EXTRACREATEFILEDESTINATION.I=ifClause(translate(EXTRACREATEFILEDESTINATION.I)='IMG','IMG','CD')
    end

    /* Force extra INI file settings */
    EXTRAUSERINIRCFILE.1=WPSUSERINIRC
    EXTRASYSINIRCFILE.1=WPSSYSINIRC
    do I=1 to EXTRAUSERINIRCFILES
        rc=defaultIfNotSet('EXTRAUSERINIRCFILEUPDATE.'I,'NO')
    end
    do I=1 to EXTRASYSINIRCFILES
        rc=defaultIfNotSet('EXTRASYSINIRCFILEUPDATE.'I,'NO')
    end
    do I=1 to EXTRAINIFILES
        IDX=pos(":", EXTRAINIFILEDIRECTORY.I)
        if IDX<>0 then EXTRAINIFILEDIRECTORY.I=right(EXTRAINIFILEDIRECTORY.I, length(EXTRAINIFILEDIRECTORY.I)-IDX)
        if left(EXTRAINIFILEDIRECTORY.I,1)<>'\' then EXTRAINIFILEDIRECTORY.I='\'EXTRAINIFILEDIRECTORY.I
        EXTRAINIFILEDESTINATION.I=ifClause(translate(EXTRAINIFILEDESTINATION.I)='IMG','IMG','CD')
    end

    /* Force settings when creating a maintenance partition */
    if \isSet(CREATEBOOTCD) then do
        VIRTUALDRIVE=WRITERDRIVE
        CDROMDRIVE=WRITERDRIVE
        MAKEROOM='NO'
        TWODISKBOOT='NO'
    end

    /* Check the virtual drive when creating a non-memdisk bootAble CD/DVD */
    if isSet(CREATEBOOTCD) & \isSet(MEMDISK) then do
        DRVINFO=sysDriveInfo(VIRTUALDRIVE)
        DRVFSTYPE=sysFilesystemType(VIRTUALDRIVE)
        if DRVINFO='' | word(DRVINFO,3)<>2931712 | DRVFSTYPE<>'FAT' then
            call error 'VIRTUALDRIVE ('VIRTUALDRIVE') does not point to a mounted 2.88MB floppy'
    end

    /* Only one of HPFS386 and HPFS can be set */
    if isSet(HPFS386) then HPFS='NO'

    /* Force AUTOCHECK options */
    if \isSet(HPFS)    then HPFSAUTOCHECK='NO'
    if \isSet(HPFS386) then HPFS386AUTOCHECK='NO'
    if \isSet(JFS)     then JFSAUTOCHECK='NO'
    if isSet(HPFSAUTOCHECK) | isSet(JFSAUTOCHECK) | isSet(HPFS386AUTOCHECK) then
        AUTOCHECK='YES'
    else
        AUTOCHECK='NO'

    if WICREQUIRED   & fileExists(WICCOMMAND)=''      then call error 'Required executable WICCOMMAND ('WICCOMMAND') does not exist'
    if WICREQUIRED   & fileExists(WPIRTLDLLFILE)=''   then call error 'Required executable WPIRTLDLLFILE ('WPIRTLDLLFILE') does not exist'
    if UNZIPREQUIRED & fileExists(UNZIPCOMMAND)=''    then call error 'Required executable UNZIPCOMMAND ('UNZIPCOMMAND') does not exist'

    /* Clone CD/DVD setup */
    if \isSet(CREATEBOOTCD) then
        CREATECLONECD='NO'
    if isSet(CREATECLONECD) then do
        OS2APPS='YES'
        REXX='YES'
        EXTRAFILES=EXTRAFILES SYSINSTX 'makeini.exe' 'eautil.exe',
          ifClause(UNZIPREQUIRED,UNZIPCOMMAND,''),
          ifClause(WICREQUIRED,WICCOMMAND WPIRTLDLLFILE,''),
          ifClause(isSet(WPS) & VIDEODRIVER=3,'unpack2.exe' SNAPSEDISTRIBUTIONFILE,'')
        EXTRADIRECTORIES=EXTRADIRECTORIES+1
        EXTRADIRECTORY.EXTRADIRECTORIES=CREATECLONECDDIRECTORY
        EXTRADIRECTORYDESTINATION.EXTRADIRECTORIES='CD'
        EXTRADIRECTORYFILES.EXTRADIRECTORIES='bootAble.cmd bootAble.ico bootAble.cfg' ifSet(WPS,'bootAble.lvmrefresh.ico bootAble.lvmgui.ico') CREATECLONECDCMDLINEARGS
        do I=1 to EXTRAZIPWPIS
            EXTRADIRECTORYFILES.EXTRADIRECTORIES=EXTRADIRECTORYFILES.EXTRADIRECTORIES EXTRAZIPWPIFILE.I
        end
        EXTRADIRECTORYADDTOPATHS.EXTRADIRECTORIES='PATH'
        EXTRADIRECTORYRECURSE.EXTRADIRECTORIES='NO'
        if isSet(WPS) then do
            do I=1 to EXTRAUSERINIRCFILES
                EXTRADIRECTORYFILES.EXTRADIRECTORIES=EXTRADIRECTORYFILES.EXTRADIRECTORIES '"'EXTRAUSERINIRCFILE.I'"'
            end
            do I=1 to EXTRASYSINIRCFILES
                EXTRADIRECTORYFILES.EXTRADIRECTORIES=EXTRADIRECTORYFILES.EXTRADIRECTORIES '"'EXTRASYSINIRCFILE.I'"'
            end
        end
    end

    /* Determine tardirs */
    NRTARDIRS=0
    drop TARDIRS.
    if isSet(UNICODE) then do
        NRTARDIRS=NRTARDIRS+1
        TARDIRS.NRTARDIRS='language'
    end
    if isSet(HPFS386) then do
        NRTARDIRS=NRTARDIRS+1
        TARDIRS.NRTARDIRS='ibm386fs'
    end
    if isSet(NETWORKING) & IBMCOMDESTINATION='IMG' then do
        NRTARDIRS=NRTARDIRS+1
        TARDIRS.NRTARDIRS='ibmcom'
    end
    if isSet(NETWORKING) & NETWORKINGDESTINATION='IMG' & isSet(TCPIP) then do
        NRTARDIRS=NRTARDIRS+1
        TARDIRS.NRTARDIRS='mptn'
    end
    if isSet(PEER) & NETWORKINGDESTINATION='IMG' then do
        NRTARDIRS=NRTARDIRS+1
        TARDIRS.NRTARDIRS='muglib'
    end
    if isSet(PEER) then do
        NRTARDIRS=NRTARDIRS+1
        TARDIRS.NRTARDIRS='ibmlan'
    end
    if \isSet(MAKEROOM) | isSet(AUTOCHECK) |,
       (isSet(OS2APPS)              & OS2APPSDESTINATION='IMG') |,
       (isSet(OS2EXTRA)             & OS2EXTRADESTINATION='IMG') |,
       (isSet(PEER)                 & NETWORKINGDESTINATION='IMG') |,
       (isSet(REXX)                 & REXXDESTINATION='IMG') |,
       (isSet(WPS)                  & WPSDESTINATION='IMG') |,
       (isSet(WPSAPPS)              & WPSAPPSDESTINATION='IMG') |,
       (isSet(WPSEPM)               & WPSEPMDESTINATION='IMG') |,
       (isSet(WPSREGEDIT2)          & WPSREGEDIT2DESTINATION='IMG') |,
       (strip(EXTRAFILES)<>''       & EXTRAFILESDESTINATION='IMG') |,
       (strip(EXTRADLLFILES)<>''    & EXTRADLLFILESDESTINATION='IMG') |,
       (strip(EXTRADEVICEFILES)<>'' & EXTRADEVICEDESTINATION='IMG')
    then do
        NRTARDIRS=NRTARDIRS+1
        TARDIRS.NRTARDIRS='os2'
    end
    if isSet(WPS) & WPSFONTSDESTINATION='IMG' then do
        NRTARDIRS=NRTARDIRS+1
        TARDIRS.NRTARDIRS='psfonts'
    end

    do I=1 to EXTRADIRECTORIES
        if EXTRADIRECTORYDESTINATION.I='IMG' then do
            LENDIR=pos('\',EXTRADIRECTORY.I,2)-2
            if LENDIR=-2 then LENDIR=length(EXTRADIRECTORY.I)-1
            EXTRADIR=translate(substr(EXTRADIRECTORY.I,2,LENDIR))
            do II=1 to NRTARDIRS
                if translate(TARDIRS.II)=EXTRADIR then do
                    EXTRADIR=''
                    leave
                end
            end
            if EXTRADIR<>'' then do
                NRTARDIRS=NRTARDIRS+1
                TARDIRS.NRTARDIRS=EXTRADIR
            end
        end
    end
    do I=1 to EXTRAZIPWPIS
        if EXTRAZIPWPIDESTINATION.I='IMG' then do
            LENDIR=pos('\',EXTRAZIPWPIDIRECTORY.I,2)-2
            if LENDIR=-2 then LENDIR=length(EXTRAZIPWPIDIRECTORY.I)-1
            EXTRADIR=translate(substr(EXTRAZIPWPIDIRECTORY.I,2,LENDIR))
            do II=1 to NRTARDIRS
                if translate(TARDIRS.II)=EXTRADIR then do
                    EXTRADIR=''
                    leave
                end
            end
            if EXTRADIR<>'' then do
                NRTARDIRS=NRTARDIRS+1
                TARDIRS.NRTARDIRS=EXTRADIR
            end
        end
    end
    do I=1 to EXTRACREATEFILES
        if EXTRACREATEFILEDESTINATION.I='IMG' then do
            LENDIR=pos('\',EXTRACREATEFILEDIRECTORY.I,2)-2
            if LENDIR=-2 then LENDIR=length(EXTRACREATEFILEDIRECTORY.I)-1
            EXTRADIR=translate(substr(EXTRACREATEFILEDIRECTORY.I,2,LENDIR))
            do II=1 to NRTARDIRS
                if translate(TARDIRS.II)=EXTRADIR then do
                    EXTRADIR=''
                    leave
                end
            end
            if EXTRADIR<>'' then do
                NRTARDIRS=NRTARDIRS+1
                TARDIRS.NRTARDIRS=EXTRADIR
            end
        end
    end
    TARDIRS.0=NRTARDIRS

    /* Check required executables */
    if fileExists(SYSINSTX)='' then
        call error 'Required executable SYSINSTX ('SYSINSTX') does not exist'
    if isSet(CREATEBOOTCD) & isSet(CDRECORD) & (isSet(AUTOBURNCD) | isSet(CDRECORDBURNCMD)) then do
        if fileExists(CDRECORDMKISOFSEXE)='' then
            call error 'Required executable CDRECORDMKISOFSEXE ('CDRECORDMKISOFSEXE') does not exist'
        if CDRECORDMEDIA=0 & \isSet(CDRECORDCREATEISOONLY) & fileExists(CDRECORDEXE)='' then
            call error 'Required executable CDRECORDEXE ('CDRECORDEXE') does not exist'
        if CDRECORDMEDIA>0 & \isSet(CDRECORDCREATEISOONLY) & fileExists(CDRECORDDVDDAOEXE)='' then
            call error 'Required executable CDRECORDDVDDAOEXE ('CDRECORDDVDDAOEXE') does not exist'
    end
    if isSet(CREATEBOOTCD) & isSet(CDBOOTMENU) & \isSet(MEMDISK) & fileExists(CDBOOTMENUFILE)='' then
        call error 'Required executable CDBOOTMENUFILE ('CDBOOTMENUFILE') does not exist'
    if isSet(CREATEBOOTCD) & \isSet(CDRECORD) & fileExists(RSJCDBOOTEXE)='' then
        call error 'Required executable RSJCDBOOTEXE ('RSJCDBOOTEXE') does not exist'
    if isSet(CREATEBOOTCD) & \isSet(CDRECORD) & \isSet(MEMDISK) & isSet(CDBOOTMENU) & RSJCDBOOTEXE=CDBOOTMENUFILE then
        call error 'RSJCDBOOTEXE and CDBOOTMENUFILE point to the same file ('RSJCDBOOTEXE')'
    if isSet(CREATEBOOTCD) & \isSet(CDRECORD) & isSet(RSJATTACHCD) & fileExists(RSJCDATTACHEXE)='' then
        call error 'Required executable RSJCDATTACHEXE ('RSJCDATTACHEXE') does not exist'
    if isSet(CREATEBOOTCD) & \isSet(CDRECORD) & isSet(AUTOBURNCD) & isSet(ERASECD) & fileExists(RSJCDWFMTEXE)='' then
        call error 'Required executable RSJCDWFMTEXE ('RSJCDWFMTEXE') does not exist'

    /* Add optional EXIT to startup.cmd file */
    if STARTUPCMD<>'' & isSet(STARTUPCMDEXIT) then
        STARTUPCMD=STARTUPCMD||ifClause(right(STARTUPCMD,2)<>CRLF,'',CRLF)||'@exit'

    return 0


/* ------------------------------------------------------------------------ */
/* Write asked questions to file                                            */
/* ARG: None                                                                */
/* RET: 0                                                                   */
/* ------------------------------------------------------------------------ */
writeConfig:
    /* Write the asked values to file if requested */
    if NRASKEDSAVED>0 then do
        if varValue('WRITEASKEDTOFILE')='' then do
            COMMENT='; Would you like to save the set options in a file for later reuse'||CRLF||'; and customization?'
            VARNAME='WRITEASKEDTOFILE'
            WRITEASKEDTOFILE=askYesNo('YES',,,'DONOTSAVE')
        end
        if isSet(WRITEASKEDTOFILE) then do
            FILEOK=0
            do until FILEOK
                if varValue('WRITEASKEDTOFILENAME')='' then do
                    COMMENT='; Filename of file to write the asked questions to'
                    VARNAME='WRITEASKEDTOFILENAME'
                    WRITEASKEDTOFILENAME=askText('bootAble.myCfg','FILE',,,'DONOTSAVE')
                end
                if fileExists(WRITEASKEDTOFILENAME,'Do not Search Path')<>'' then do
                    say
                    KEY=strip(left(WRITEASKEDTOFILEALREADYEXISTS,1))
                    if KEY='' then do
                        KEY=multipleChoice('ANOQS',WRITEASKEDTOFILENAME 'already exists,'||CRLF||'[S]ave to new file, [O]verwrite, [A]ppend, do [N]ot save, or [Q]uit?')
                        say KEY
                    end
                    FILEOK=1
                    select
                        when KEY='A' then rc=lineOut(WRITEASKEDTOFILENAME,'')
                        when KEY='N' then WRITEASKEDTOFILE='NO'
                        when KEY='O' then do
                            rc=sysFileDelete(WRITEASKEDTOFILENAME)
                            if rc<>0 then
                                call error 'Failed to remove existing' WRITEASKEDTOFILENAME '('rc')',,, 'NOWRITEASKED'
                        end
                        when KEY='Q' then do
                            TXT='ABORT: Script aborted'
                            if \isSet(CLEANUP) then
                                TXT=TXT 'without cleaning up'
                            call error TXT, 'NOPAUSE',, 'NOWRITEASKED'
                        end
                        otherwise do
                            WRITEASKEDTOFILENAME=''
                            FILEOK=0
                        end
                    end
                end
                else
                    FILEOK=1
            end
            if isSet(WRITEASKEDTOFILE) then do
                rc=writeAskedLinesToFile(WRITEASKEDTOFILENAME)
                rc=stream(WRITEASKEDTOFILENAME, 'C', 'CLOSE')
                NRASKEDSAVED=0
            end
        end
    end

    if NRASKED>0 & isSet(PAUSEAFTERQUESTIONS) then do
        say
        DEST='CD'
        TXT='Setup is now complete.  Would you like to create your' destinationName() 'now?'||CRLF||,
            'If you answer "No", you can customize' WRITEASKEDTOFILENAME', and then run'||CRLF||,
            'bootAble again whenever you'"'"'re ready to continue.'||CRLF||CRLF||,
            'Continue the script?'
        if \isSet(yesNoQuit(TXT,'YES','NOCLEAN')) then
            call error 'ABORT: Script aborted.' ifSet(WRITEASKEDTOFILE,'Run "bootAble' WRITEASKEDTOFILENAME'" when you'"'"'re ready to continue.'), 'NOPAUSE',,'NOWRITEASKED'
    end

    return 0


/* ------------------------------------------------------------------------ */
/* Read and parse an individual configuration file                          */
/* ARG1: Configuration file to load                                         */
/* ARG2: Process configuration file as "defaults" file                      */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
readConfigFile:
    CONFIGFILE=ARG(1)
    SETDEFAULTS=ARG(2)
    CONFIGFILE=fileExists(CONFIGFILE)
    if CONFIGFILE='' then do
        call error 'Configuration file' ARG(1) 'not found',, 'ASK'
        return 0
    end
    call progress 2, 'Reading configuration file' CONFIGFILE ifSet(SETDEFAULTS, '(defaults)')
    STATEMENT=''
    COMMENT=''
    LINENR=0
    do while lines(CONFIGFILE)>0
        LINE=strip(linein(CONFIGFILE))
        LINENR=LINENR+1
        call progress 4, CONFIGFILE'('LINENR'):' LINE
        CONT=0
        if right(LINE,1)=',' then do
            CONT=1
            LINE=left(LINE,length(LINE)-1)
        end
        if left(LINE,2)<>'/*' & left(LINE,1)<>';' then
            STATEMENT=strip(STATEMENT LINE)
        else do
            if COMMENT<>'' then
                COMMENT=COMMENT||CRLF
            COMMENT=COMMENT||LINE
        end
        if CONT=0 then do
            DOPARSE='YES'
            VARNAME='NOVARIABLESETTING'
            EQUALSIGN=pos('=', STATEMENT)
            if EQUALSIGN>0 & words(left(STATEMENT,EQUALSIGN))=1 then
                VARNAME=left(STATEMENT,EQUALSIGN-1)
            if isSet(SETDEFAULTS) & translate(VARNAME)<>'RC' & translate(VARNAME)<>value(VARNAME) then
                DOPARSE='NO'
            if isSet(DOPARSE) & STATEMENT<>'' then do
                interpret STATEMENT
                if VARNAME='ADDTOSEARCHPATH' then
                    SEARCHPATH=ADDTOSEARCHPATH';'SEARCHPATH
                if abbrev(VARNAME,'_ADDAFTER_') then do
                    _VARNAME=substr(VARNAME,11)
                    if _VARNAME='' then
                        call error CONFIGFILE'('LINENR'): _ADDAFTER_ variable name missing'
                    rc=value(_VARNAME,varValue(_VARNAME)||value(VARNAME))
                end
                if abbrev(VARNAME,'_ADDBEFORE_') then do
                    _VARNAME=substr(VARNAME,12)
                    if _VARNAME='' then
                        call error CONFIGFILE'('LINENR'): _ADDBEFORE_ variable name missing'
                    rc=value(_VARNAME,value(VARNAME)||varValue(_VARNAME))
                end
                if SEARCHPATH<>PREVSEARCHPATH then do
                    rc=checkSearchPath()
                    PREVSEARCHPATH=SEARCHPATH
                    rc=value('SEARCHPATH',SEARCHPATH,'OS2ENVIRONMENT')
                end
            end
            STATEMENT=''
            if left(LINE,2)<>'/*' & left(LINE,1)<>';' then
                COMMENT=''
        end
    end
    rc=stream(CONFIGFILE, 'C', 'CLOSE')
    if rc<>'READY:' then
        call error 'Failed to close' CONFIGFILE 'rc='rc

    return 0


/* ------------------------------------------------------------------------ */
/* List of MAKEROOM files for the os2 directory                             */
/* ARG: None                                                                */
/* RET: List of files                                                       */
/* ------------------------------------------------------------------------ */
makeRoomFiles:
    FILES_=VITALFILES_ SHELLFILE PROTSHELLFILE
    if isSet(WPS)               then FILES_=FILES_ WPSRUNWORKPLACEFILE
    if isSet(CDFS)              then FILES_=FILES_ CDFSFILES
    if isSet(FAT32)             then FILES_=FILES_ FAT32FILES FAT32CACHEFILE
    if isSet(NTFS)              then FILES_=FILES_ NTFSFILES
    if isSet(UDF)               then FILES_=FILES_ UDFFILES
    if isSet(APM)               then FILES_=FILES_ APMFILE APMDAEMONFILE
    if isSet(ACPIDAEMON)        then FILES_=FILES_ ACPIDAEMONFILE
    if isSet(SERIALPORT)        then FILES_=FILES_ SERIALPORTFILE
    if isSet(SIOSERIALPORT)     then FILES_=FILES_ SIOSERIALPORTFILE
    if isSet(SIO2KSERIALPORT)   then FILES_=FILES_ SIO2KUARTFILE SIO2KSERIALPORTFILE
    if isSet(SIO2KPATCH)        then FILES_=FILES_ SIO2KPATCHFILE
    if isSet(MOUSE)             then FILES_=FILES_ MOUSEPOINTDDFILE MOUSEFILE MOUSEDEVFILE
    if isSet(USBKEYB)           then FILES_=FILES_ USBKEYBFILE
    if isSet(USBCOM)            then FILES_=FILES_ USBCOMFILE
    if isSet(USBMOUSE)          then FILES_=FILES_ USBMOUSEFILE
    if isSet(USBPRINTER)        then FILES_=FILES_ USBPRINTERFILE
    if isSet(LVM)               then FILES_=FILES_ LVMFILES
    if isSet(MINILVM)           then FILES_=FILES_ MINILVMFILES
    if isSet(RAMDISK)           then FILES_=FILES_ RAMDISKFILE
    if isSet(MSVMADDITIONS)     then FILES_=FILES_ MSVMADDITIONSFILES
    if isSet(HPFS)    & \isSet(HPFSAUTOCHECK)    then FILES_=FILES_ HPFSFILES
    if isSet(HPFS386) & \isSet(HPFS386AUTOCHECK) then FILES_=FILES_ HPFS386OS2FILES
    if isSet(JFS)     & \isSet(JFSAUTOCHECK)     then FILES_=FILES_ JFSFILES
    if                  \isSet(AUTOCHECK)        then FILES_=FILES_ VITALFILES__
    return FILES_


/* ------------------------------------------------------------------------ */
/* List of MAKEROOM files for the os2\dll directory                         */
/* ARG: None                                                                */
/* RET: List of files                                                       */
/* ------------------------------------------------------------------------ */
makeRoomDLLFiles:
    DLLFILES_=VITALDLLFILES_
    if isSet(CDFS)  then DLLFILES_=DLLFILES_ CDFSDLLFILES
    if isSet(FAT32) then DLLFILES_=DLLFILES_ FAT32DLLFILES
    if isSet(NTFS)  then DLLFILES_=DLLFILES_ NTFSDLLFILES
    if isSet(UDF)   then DLLFILES_=DLLFILES_ UDFDLLFILES
    if isSet(HPFS)    & \isSet(HPFSAUTOCHECK)    then DLLFILES_=DLLFILES_ HPFSDLLFILES
    if isSet(HPFS386) & \isSet(HPFS386AUTOCHECK) then DLLFILES_=DLLFILES_ HPFS386OS2DLLFILES
    if isSet(JFS)     & \isset(JFSAUTOCHECK)     then DLLFILES_=DLLFILES_ JFSDLLFILES
    if isSet(LVM)     & \isSet(AUTOCHECK)        then DLLFILES_=DLLFILES_ LVMDLLFILES
    if isSet(MINILVM)                            then DLLFILES_=DLLFILES_ MINILVMDLLFILES
    if isSet(UNICODE) & \isSet(AUTOCHECK)        then DLLFILES_=DLLFILES_ UNICODEDLLFILES
    if                  \isSet(AUTOCHECK)        then DLLFILES_=DLLFILES_ VITALDLLFILES__
    return DLLFILES_


/* ------------------------------------------------------------------------ */
/* List of AUTOCHECK files for the os2 directory                            */
/* ARG: None                                                                */
/* RET: List of files                                                       */
/* ------------------------------------------------------------------------ */
autoCheckFiles:
    FILES__=''
    if isSet(HPFS)    & isSet(HPFSAUTOCHECK)    then FILES__=FILES__ HPFSFILES
    if isSet(HPFS386) & isSet(HPFS386AUTOCHECK) then FILES__=FILES__ HPFS386OS2FILES
    if isSet(JFS)     & isSet(JFSAUTOCHECK)     then FILES__=FILES__ JFSFILES
    if                  isSet(AUTOCHECK)        then FILES__=FILES__ VITALFILES__
    return FILES__


/* ------------------------------------------------------------------------ */
/* List of AUTOCHECK files for the os2\dll directory                        */
/* ARG: None                                                                */
/* RET: List of files                                                       */
/* ------------------------------------------------------------------------ */
autoCheckDLLFiles:
    DLLFILES__=''
    if isSet(HPFS)    & isSet(HPFSAUTOCHECK)    then DLLFILES__=DLLFILES__ HPFSDLLFILES
    if isSet(HPFS386) & isSet(HPFS386AUTOCHECK) then DLLFILES__=DLLFILES__ HPFS386OS2DLLFILES
    if isSet(JFS)     & isset(JFSAUTOCHECK)     then DLLFILES__=DLLFILES__ JFSDLLFILES
    if isSet(LVM)     & isSet(AUTOCHECK)        then DLLFILES__=DLLFILES__ LVMDLLFILES
    if isSet(UNICODE) & isSet(AUTOCHECK)        then DLLFILES__=DLLFILES__ UNICODEDLLFILES
    if                  isSet(AUTOCHECK)        then DLLFILES__=DLLFILES__ VITALDLLFILES__
    return DLLFILES__


/****************************************************************************/
/**** Core creation functions ***********************************************/
/****************************************************************************/

/* ------------------------------------------------------------------------ */
/* Create a bootAble CD/DVD / Maintenance partition                         */
/* ARG: None                                                                */
/* RET: 0                                                                   */
/* ------------------------------------------------------------------------ */
createBootAble:
    /* Create directory to build CD/DVD Image when using CDRECORD */
    if isSet(CREATEBOOTCD) & isSet(CDRECORD) then do
        rc=createDir(WRITERDRIVE)
        if isSet(CDRECORDCLEANDIRBEFORE) then
            rc=cleanDir(WRITERDRIVE)
    end

    /* Attach and or Format CD-R(W) when using RSJ */
    if isSet(CREATEBOOTCD) & \isSet(CDRECORD) then do
        if isSet(RSJATTACHCD) then do
            call progress 1, 'Attaching CD/DVD to drive' WRITERDRIVE
            rc=runCMD(RSJCDATTACHEXE WRITERDRIVE, 'Failed to attach CD/DVD to drive' WRITERDRIVE)
        end
        if isSet(ERASECD) then do
            call progress 1, 'Erasing CD/DVD in drive' WRITERDRIVE
            rc=runCMD(RSJCDWFMTEXE WRITERDRIVE '/erase', 'Failed to erase CD/DVD in drive' WRITERDRIVE)
            rc=runCMD('label' WRITERDRIVE||VOLUMELABEL, 'Failed to set the label of the CD/DVD in drive' WRITERDRIVE)
        end
    end


    /* Format and prepare boot image/disk */
    rc=prepareImage(1)

    /* Copy required files to boot image/disk */
    rc=copyImageFiles(1)

    /* Create config.sys on boot image/disk */
    rc=createConfigSys()

    /* Bootmessages? */
    if isSet(BOOTMESSAGES) then
        rc=createFile(VIRTUALDRIVE'\OS2\BOOT\ALTF2ON.$$$', '')

    /* REXX Exit just before commiting first image (part) */
    if strip(REXXEXIT_IMG1)<>'' then
        rc=runREXX(REXXEXIT_IMG1, 'running REXXEXIT_IMG1')

    /* Create second disk image if creating a two disk boot */
    if isSet(TWODISKBOOT) then do
        /* Write first boot disk */
        rc=writeImage(1)
        /* Format and prepare second boot disk */
        rc=prepareImage(2)
    end

    /* Copy required files to (second) boot image/disk */
    rc=copyImageFiles(2)

    /* Add startup.cmd to (second) boot image/disk */
    if strip(STARTUPCMD)<>'' then
        rc=createFile(VIRTUALDRIVE'\STARTUP.CMD',parseText(STARTUPCMD))

    /* Add additional files to (second) boot image/disk */
    /* Note: TAR-files in case of a MEMDISK image       */
    if isSet(CREATEBOOTCD) & isSet(MEMDISK) then
        DESTDRIVE=VIRTUALDRIVE'\TOTAR.###'
    else
        DESTDRIVE=VIRTUALDRIVE
    rc=copyOtherFiles('IMG')
    rc=createFiles('IMG')

    /* REXX Exit just before commiting second image (part) */
    if strip(REXXEXIT_IMG2)<>'' then
        rc=runREXX(REXXEXIT_IMG2, 'running REXXEXIT_IMG2')

    /* Write the image/disk */
    if isSet(TWODISKBOOT) then
        rc=writeImage(2)
    else
        rc=writeImage(1)

    /* Add additional files to the CD/DVD */
    DESTDRIVE=WRITERDRIVE
    rc=copyOtherFiles('CD')
    rc=createFiles('CD')

    /* REXX Exit just before commiting complete build */
    if strip(REXXEXIT_COMPLETE)<>'' then
        rc=runREXX(REXXEXIT_COMPLETE, 'running REXXEXIT_COMPLETE')

    /* Finalize CD/DVD / ISO image */
    rc=finalizeImage()

    /* Done */
    call progress 1, 'Done creating' ifSet(CREATEBOOTCD,'bootable CD', 'Maintenance Partition' WRITERDRIVE)

    return 0


/* ------------------------------------------------------------------------ */
/* Prepare a boot image                                                     */
/* ARG1: Image number                                                       */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
prepareImage:
    IMAGENR=ARG(1)

    if isSet(MEMDISK) then
        return 0

    if (\isSet(CREATEBOOTCD) & \isSet(FORMATMAINTENANCE)) then do
        say 'You are creating a Maintenance Partition on drive' WRITERDRIVE
        say 'In order to guarantee its working, bootAble needs to format it.'
        if \isSet(yesNoQuit('Do you want to format the partition?')) then
            return 0
    end

    /* Format the (virtual) disk */
    call progress 1, 'Formatting' bootImageName()
    LABEL=ifSet(CREATEBOOTCD,'CDBOOT_'IMAGENR,VOLUMELABEL)
    rc=runCMD('label' VIRTUALDRIVE||'FAILED', 'Failed to reset the label of' VIRTUALDRIVE '(was the drive formatted before?)')
    rc=lineOut(BOOTABLETEMP'\response', 'FAILED'||CRLF||FORMATMAINTENANCERESPONSE)
    if rc=0 then
        rc=lineout(BOOTABLETEMP'\response')
    if rc<> 0 then
        call error 'Failed to create format response file'
    'format' VIRTUALDRIVE '/fs:'ifSet(CREATEBOOTCD,'fat /once',ifset(FORMATMAINTENANCEJFS,'jfs','hpfs')) '/v:'LABEL '<' BOOTABLETEMP'\response' IGNOREOUTPUT
    /* Somehow format does not always return non-zero rc in case of error... */
    if rc<>0 | word(sysDriveInfo(VIRTUALDRIVE),4)<>LABEL then
        call error 'Error formatting' bootImageName() '('VIRTUALDRIVE')'

    /* Install system files */
    call progress 1, 'Transferring system files to' bootImageName()
    rc=runCMD(SYSINSTX VIRTUALDRIVE, 'Error transferring system files to' bootImageName())

    return 0


/* ------------------------------------------------------------------------ */
/* Write a boot image to CD/DVD                                             */
/* ARG1: Image number                                                       */
/* RET: 0                                                                   */
/* ------------------------------------------------------------------------ */
writeImage:
    if \isSet(CREATEBOOTCD) then return 0

    IMAGENR=ARG(1)

    if IMAGENR=1 & isSet(CDBOOTMENU) & \isSet(MEMDISK) then do
        rc=runCMD(CDBOOTMENUFILE '/DRIVE' left(VIRTUALDRIVE,1) '/TIMEOUT' CDBOOTMENUTIMEOUT CDBOOTMENUOTHEROPTIONS, 'Failed adding boot menu')
    end

    if isSet(PAUSEBEFOREBOOTIMAGE) then do
        say 'Pausing before writing' bootImageName()
        if \isSet(yesNoQuit('Continue with script?','Yes','NOCLEAN')) then
            call error 'ABORT: Script aborted', 'NOPAUSE'
    end

    if isSet(MEMDISK) then do
        rc=writeMemDisk()
    end
    else do
        call progress 1, 'Writing' bootImageName()
        rc=runCMD(SAVEDSKFEXE '/a/d' VIRTUALDRIVE WRITERDRIVE'\'BOOTIMAGENAME.IMAGENR, 'Error writing' bootImageName())

        /* The first image will be marked bootable */
        if IMAGENR=1 & \isSet(CDRECORD) then do
            rc=runCMD(RSJCDBOOTEXE WRITERDRIVE||BOOTIMAGENAME.IMAGENR, 'Error marking Boot Image as bootAble')
        end
    end

    /* Copy config.sys to root of CD? */
    if isSet(COPYCONFIGSYS) & IMAGENR=1 then do
        CFGSYSFILE=VIRTUALDRIVE'\config.sys'
        if isSet(TWODISKBOOT) then do
             TMPCFGSYSFILE=BOOTABLETEMP'\config.sys'
             rc=sCopyFile(CFGSYSFILE, TMPCFGSYSFILE)
             CFGSYSFILE=TMPCFGSYSFILE
        end
        rc=copyFiles(CFGSYSFILE, WRITERDRIVE)
    end

    return 0


/* ------------------------------------------------------------------------ */
/* Write memdisk images to CD/DVD                                           */
/* ARG: None                                                                */
/* RET: 0                                                                   */
/* ------------------------------------------------------------------------ */
writeMemDisk:
    /* CDLOADER.BIN */
    rc=createDir(WRITERDRIVE'\BOOTIMGS')
    rc=sCopyFile(MEMDISKCDLOADERBIN, WRITERDRIVE'\BOOTIMGS\cdloader.bin')
    if \isSet(CDRECORD) then do
        rc=runCMD(RSJCDBOOTEXE WRITERDRIVE'\BOOTIMGS\cdloader.bin', 'Error marking' WRITERDRIVE'\BOOTIMGS\cdloader.bin as bootAble')
    end

    /* MEMDISK.PF */
    PFFILE=BOOTABLETEMP'\memdisk.pf'
    NEXTPFFILE=BOOTABLETEMP'\root.pf'
    PFSFILE=PFFILE's'
    PFSTXT=,
        'newline'||CRLF||,
        'output "Welcome to";'||CRLF||,
        'newline'||CRLF||,
        'output "                                                                ";'||CRLF||,
        'output "                                                            ";'||CRLF||,
        'output "                                                          ";'||CRLF||,
        'output "                                                       ";'||CRLF||,
        'output "                                                      ";'||CRLF||,
        'output "                            ";'||CRLF||,
        'output "                     ";'||CRLF||,
        'output "                                ";'||CRLF||,
        'output "                               ";'||CRLF||,
        'output "                                ";'||CRLF||,
        'output "                                     ";'||CRLF||,
        'output "                                  ";'||CRLF||,
        'output "                        ";'||CRLF||,
        'output "                                     ";'||CRLF||,
        'newline'||CRLF||,
        'newline'||CRLF
    rc=createPFFile(PFSFILE, PFSTXT, PFFILE, NEXTPFFILE)

    /* ROOT.PF */
    rc=sCopyFile(MEMDISKERASEFILSYS, VIRTUALDRIVE'\erasefil.sys')
    if (TARDIRS.0>0) then do
        rc=sCopyFile(MEMDISKUNTARDLL,    VIRTUALDRIVE'\untar.dll')
        rc=sCopyFile(MEMDISKUNTARMSG,    VIRTUALDRIVE'\untar.msg')
        rc=sCopyFile(MEMDISKLOADDLLSYS,  VIRTUALDRIVE'\loaddll.sys')
        rc=sCopyFile(MEMDISKLOADDLLMSG,  VIRTUALDRIVE'\loaddll.msg')
    end
    PFFILE=NEXTPFFILE
    NEXTPFFILE=BOOTABLETEMP'\os2.pf'
    PFSFILE=PFFILE's'
    PFSTXT=,
        'output "Loading root     files "'||CRLF||,
        'add file 'VIRTUALDRIVE'\*'||CRLF||,
        'bootsector 'MEMDISKBOOTSECTOR||CRLF
    rc=createPFFile(PFSFILE, PFSTXT, PFFILE, NEXTPFFILE)


    /* OS2.PF */
    PFFILE=NEXTPFFILE
    NEXTPFFILE=BOOTABLETEMP'\os2boot.pf'
    PFSFILE=PFFILE's'
    PFSTXT=,
        'output "Loading os2      files "'||CRLF||,
        'add file 'VIRTUALDRIVE'\os2\*'||CRLF
    rc=createPFFile(PFSFILE, PFSTXT, PFFILE, NEXTPFFILE)


    /* OS2BOOT.PF */
    PFFILE=NEXTPFFILE
    NEXTPFFILE=BOOTABLETEMP'\os2dll.pf'
    PFSFILE=PFFILE's'
    PFSTXT=,
        'output "Loading os2\boot files "'||CRLF||,
        'add file 'VIRTUALDRIVE'\os2\boot\*'||CRLF
    rc=createPFFile(PFSFILE, PFSTXT, PFFILE, NEXTPFFILE)

    /* OS2DLL.PF */
    PFFILE=NEXTPFFILE
    NEXTPFFILE=ifClause(TARDIRS.0>0,BOOTABLETEMP'\'TARDIRS.1'.pft','')
    PFSFILE=PFFILE's'
    PFSTXT=,
        'output "Loading os2\dll  files "'||CRLF||,
        'add file 'VIRTUALDRIVE'\os2\dll\*'||CRLF
    rc=createPFFile(PFSFILE, PFSTXT, PFFILE, NEXTPFFILE)

    /* Tar files */
    do I=1 to TARDIRS.0
        II=I+1
        PFFILE=BOOTABLETEMP'\'TARDIRS.I'.pft'
        NEXTPFFILE=ifClause(TARDIRS.0>I,BOOTABLETEMP'\'TARDIRS.II'.pft','')
        PFSFILE=PFFILE's'
        PFSTXT=,
            'output "Loading 'left(TARDIRS.I,8)' tar file "'||CRLF
        rc=createTARFile(TARDIRS.I)
        rc=createPFFile(PFSFILE, PFSTXT, PFFILE, NEXTPFFILE)
    end

    /* MEMBOOT.BIN setup */
    MEMDISKSIZEMINFREE=0
    MEMDISKSIZEFREED=0
    if left(MEMDISKSIZE,1)='+' then do
        MEMDISKSIZEMINFREE=substr(MEMDISKSIZE,2)*1024
        MEMDISKMAXTARSIZE=0
        /* Determine space needed */
        drop FILETREE.
        rc=sysFileTree(VIRTUALDRIVE'\*', 'FILETREE', 'S')
        do I=1 to FILETREE.0
            FNAME=translate(word(FILETREE.I,5))
            FSIZE=word(FILETREE.I,3)
            FSIZE=max((FSIZE+511)%512*512,512)
            MEMDISKSIZE=MEMDISKSIZE+FSIZE
            if FNAME=translate(VIRTUALDRIVE'\untar.dll') |,
               FNAME=translate(VIRTUALDRIVE'\untar.msg') |,
               FNAME=translate(VIRTUALDRIVE'\loaddll.sys') |,
               FNAME=translate(VIRTUALDRIVE'\loaddll.msg')
            then
                MEMDISKSIZEFREED=MEMDISKSIZEFREED+FSIZE
            do II=1 to TARDIRS.0
                if FNAME=translate(VIRTUALDRIVE'\'TARDIRS.II'.tar') then do
                    MEMDISKMAXTARSIZE=max(MEMDISKMAXTARSIZE,FSIZE)
                    MEMDISKSIZE=MEMDISKSIZE-FSIZE
                end
            end
        end
        drop FILETREE.
        MEMDISKSIZEFREED=MEMDISKSIZEFREED+MEMDISKMAXTARSIZE
        MEMDISKSIZE=MEMDISKSIZE+MEMDISKMAXTARSIZE
        MEMDISKSIZE=MEMDISKSIZE+max(0,MEMDISKSIZEMINFREE-MEMDISKSIZEFREED)
        MEMDISKSIZE=(MEMDISKSIZE+1023)%1024
        MEMDISKSIZE=MEMDISKSIZE+384 /* Reserve for overhead */
        MEMDISKSIZEFREED=MEMDISKSIZEFREED+max(0,MEMDISKSIZEMINFREE-MEMDISKSIZEFREED)
    end
    rc=sCopyFile(MEMDISKMEMBOOTBIN, BOOTABLETEMP'\memboot.bin')
    RSPFILE=BOOTABLETEMP'\memboot.rsp'
    RSPTXT=''||,
        '$Disk_KiB$='MEMDISKSIZE||CRLF||,
        '$OSminMib$='MEMDISKMINMB||CRLF||,
        '$ADDD_KiB$=0'||CRLF||,
        '$ADDD_PER$=30'||CRLF||,
        '$PartName$='||MEMDISKVOLUMELABEL||CRLF||,
        '$VolName_$='||MEMDISKVOLUMELABEL||CRLF||,
        '$V_Letter$='MEMDISKDRIVE||CRLF||,
        '$PartType$=true'||CRLF||,
        '$TermEmul$=true'||CRLF||,
        '$BootTarg$='ifSet(CDBOOTMENU,'*','m')||CRLF||,
        '$Update_A$=false'||CRLF||,
        '$Update_B$=false'||CRLF||,
        '$SerialDB$=false'||CRLF||,
        '$Indicato$=1'||CRLF||,
        '$eCS_Menu$='||MEMDISKMENU||CRLF||,
        '$DebugOpt$=0'||CRLF||,
        '$LoadFont$=false'||CRLF||,
        '$PaletteM$=false'||CRLF||,
        '$CALL_CSM$=false'||CRLF||,
        '$UpdPPrim$=true'||CRLF||,
        '$ProgChar$='||CRLF||,
        '$ConfigEd$='||ifSet(MEMDISKEDIT,'true','false')||CRLF||,
        '$A20_Acc_$=2'||CRLF||,
        '$LVM_L_HD$=true'||CRLF||,
        '$SaveFile$=yes'||CRLF
    rc=createFile(RSPFILE,RSPTXT)
    call progress 2, 'Configuring' BOOTABLETEMP'\memboot.bin'
    rc=runCMD(MEMDISKMEMCFGEXE BOOTABLETEMP'\memboot.bin' '<' BOOTABLETEMP'\memboot.rsp', 'Failed to configure' BOOTABLETEMP'\memboot.bin')
    rc=sCopyFile(BOOTABLETEMP'\memboot.bin',  WRITERDRIVE'\BOOTIMGS')
    TXT='Created MEMDISK boot image of' trunc(2*MEMDISKSIZE/1024+0.5)/2 'MiB'
    if MEMDISKSIZEMINFREE<>0 then
        TXT=TXT 'leaving approximately' trunc(2*MEMDISKSIZEFREED/1024/1024+0.5)/2 'MiB free'
    call progress 1, TXT
    return rc


/* ------------------------------------------------------------------------ */
/* Create a PF file                                                         */
/* ARG1: PFS File                                                           */
/* ARG2: Content                                                            */
/* ARG3: PF File                                                            */
/* ARG4: Nexy PF File                                                       */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
createPFFile: procedure expose MEMDISKPACKFILEEXE CRLF WRITERDRIVE CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    PFSFILE=ARG(1)
    PFFILE=ARG(3)
    NEXTPFFILE=ARG(4)
    PFSTXT='create archive' PFFILE||CRLF||,
        ARG(2)||,
        ifClause(NEXTPFFILE<>'','next pointer' NEXTPFFILE||CRLF)||,
        'close archive'||CRLF
    rc=createFile(PFSFILE, PFSTXT)
    call progress 2, 'Creating' PFFILE
    rc=runCMD(MEMDISKPACKFILEEXE PFSFILE, 'Failed to create' PFFILE)
    rc=sCopyFile(PFFILE, WRITERDRIVE'\BOOTIMGS')
    return 0


/* ------------------------------------------------------------------------ */
/* Create a TAR file from a directory                                       */
/* ARG1: Directory to TAR                                                   */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
createTARFile: procedure expose MEMDISKTAREXE PFSTXT CRLF VIRTUALDRIVE CRLF NRASKED NRASKEDSAVED ASKEDLINES. ASKEDCOMMENT. CLEANUP REXXEXIT_CLEANUP BOOTABLETEMP VERBOSE IGNOREOUTPUT
    PREVDIR=directory()
    PREVDIR_=directory(driveName(VIRTUALDRIVE))
    rc=directory(VIRTUALDRIVE'\TOTAR.###')
    TARDIR=ARG(1)
    TARFILE=VIRTUALDRIVE'\'TARDIR'.tar'
    call progress 2, 'Creating' TARFILE
    rc=runCMD(MEMDISKTAREXE '-cpvf' TARFILE TARDIR, 'Failed to create' TARFILE)
    rc=directory(PREVDIR_)
    rc=directory(PREVDIR)
    PFSTXT=PFSTXT||'add file' TARFILE||CRLF
    return 0


/* ------------------------------------------------------------------------ */
/* Copy files to boot image                                                 */
/* ARG1: Image number                                                       */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
copyImageFiles:
    IMAGENR=ARG(1)

    call progress 1, 'Copying boot files to' bootImageName()

    /* Files to copy to root directory of boot image */
    if IMAGENR=1 | (isSet(TWODISKBOOT) & \isSet(MEMDISK)) then
        rc=copyFiles(VITALROOTFILES.IMAGENR, VIRTUALDRIVE'\', 'T')

    /* Files to copy to os2 directory of boot image */
    rc=createDir(VIRTUALDRIVE'\OS2')
    if IMAGENR=2 then do
        FILES=VITALFILES
        if isSet(CDROM)     then FILES=FILES CDROMDMDFILE
        if isSet(CDFS)      then FILES=FILES CDFSIFSFILE
        if isSet(FAT32)     then FILES=FILES FAT32IFSFILE
        if isSet(NTFS)      then FILES=FILES NTFSIFSFILE
        if isSet(HPFS)      then FILES=FILES HPFSIFSFILE
        if isSet(JFS)       then FILES=FILES JFSIFSFILE
        if isSet(UDF)       then FILES=FILES UDFIFSFILE
        if isSet(RAMDISK)   then FILES=FILES RAMDISKIFSFILE
        if isSet(UNICODE)   then FILES=FILES UNICODEFILE
        if   isSet(AUTOCHECK) & \isSet(MEMDISK) then FILES=FILES autoCheckFiles()
        rc=copyFiles(FILES, VIRTUALDRIVE'\os2', 'T')
    end

    /* Files to copy to os2\boot directory of boot image */
    if IMAGENR=1 then do
        rc=createDir(VIRTUALDRIVE'\OS2\BOOT')
        BOOTFILES=VITALBOOTFILES DASDFILE
        if isSet(HD4DISK)       then BOOTFILES=BOOTFILES HD4DISKFILE
        if isSet(CREATEBOOTCD) & \isSet(MEMDISK) then BOOTFILES=BOOTFILES BOOTFLTFILE
        if isSet(WPS)           then BOOTFILES=BOOTFILES WPSBOOTFILES
        if isSet(FLOPPY)        then BOOTFILES=BOOTFILES FLOPPYADDFILE
        if isSet(PARALLELPORT)  then BOOTFILES=BOOTFILES PARALLELPORTFILE
        if isSet(PPZIP)         then BOOTFILES=BOOTFILES PPZIPADDFILE
        if isSet(NON512DASD)    then BOOTFILES=BOOTFILES NON512DASDFILE
        if isSet(DASDFALLBACK)  then BOOTFILES=BOOTFILES DASDFALLBACKFILE
        if isSet(IDE)           then BOOTFILES=BOOTFILES IDEADDFILE
        if isSet(AHCI)          then BOOTFILES=BOOTFILES AHCIADDFILE
        if isSet(ATAPI)         then BOOTFILES=BOOTFILES ATAPIFLTFILE ifClause(\isSet(DANIATAPI)&\isSet(LATESTATAPI),ATAPICDFLTFILE)
        if isSet(LVM)           then BOOTFILES=BOOTFILES LVMDMDFILE
        if isSet(MEMDISK)       then BOOTFILES=BOOTFILES MEMDISKADDFILE
        if isSet(SCSIDMD)       then BOOTFILES=BOOTFILES SCSIDMDFILE
        if isSet(ASPI)          then BOOTFILES=BOOTFILES ASPIDMDFILE
        if isSet(REMOVABLESCSI) then BOOTFILES=BOOTFILES REMOVABLESCSIFILE
        if isSet(ACPI)          then BOOTFILES=BOOTFILES ACPIPSDFILE ACPIPSDCFGFILE
        if isSet(ACPIAPMADD)    then BOOTFILES=BOOTFILES ACPIAPMADDFILE

        do I=1 to SCSI
            BOOTFILES=BOOTFILES SCSIADDFILE.I
        end
        do I=1 to EXTRABASEDEV
            BOOTFILES=BOOTFILES EXTRABASEDEVFILE.I
        end
        if isSet(USB) then do
            BOOTFILES=BOOTFILES USBDFILE USBHIDFILE
            if USBOHCI>0 then BOOTFILES=BOOTFILES USBOHCIFILE
            if USBUHCI>0 then BOOTFILES=BOOTFILES USBUHCIFILE
            if USBEHCI>0 then BOOTFILES=BOOTFILES USBEHCIFILE
            if isSet(USBCDROM)   then BOOTFILES=BOOTFILES USBCDROMFILE
            if isSet(USBSTORAGE) then BOOTFILES=BOOTFILES USBSTORAGEFILE
        end
        rc=copyFiles(BOOTFILES, VIRTUALDRIVE'\os2\boot', 'T')
    end
    if IMAGENR=2 & isSet(APM) & APMGSVDAEMONFILE<>'' then do
        rc=createDir(VIRTUALDRIVE'\OS2\BOOT')
        rc=copyFiles(APMGSVDAEMONFILE, VIRTUALDRIVE'\OS2\BOOT', 'T')
    end

    /* Files to copy to os2\dll directory of boot image */
    if IMAGENR=2 then do
        DLLFILES=VITALDLLFILES
        if isSet(ACPI)                        then DLLFILES=DLLFILES ACPIDLLFILE
        if isSet(AUTOCHECK) & \isSet(MEMDISK) then DLLFILES=DLLFILES autoCheckDLLFiles()
        if strip(DLLFILES)<>'' then
            rc=createDir(VIRTUALDRIVE'\OS2\DLL')
        rc=copyFiles(DLLFILES, VIRTUALDRIVE'\os2\dll', 'T')
    end

    /* Unicode support files */
    if IMAGENR=2 & isSet(UNICODE) then do
        if isSet(CREATEBOOTCD) & isSet(MEMDISK) then
            DESTDRIVE=VIRTUALDRIVE'\TOTAR.###'
        else
            DESTDRIVE=VIRTUALDRIVE
        /* Language directory tree */
        rc=createDir(DESTDRIVE'\LANGUAGE')
        rc=createDir(DESTDRIVE'\LANGUAGE\CODEPAGE')
        rc=createDir(DESTDRIVE'\LANGUAGE\FONT')
        rc=createDir(DESTDRIVE'\LANGUAGE\KEYBOARD')
        rc=createDir(DESTDRIVE'\LANGUAGE\LOCALE')
        UNIEXTRACODEPAGES=UNICODEPAGE UNIEXTRACODEPAGES
        FILES=''
        do I=1 to words(UNIEXTRACODEPAGES)
            FILES=FILES UNICODECODEPAGEROOT||word(UNIEXTRACODEPAGES,I)
        end
        rc=copyFiles(FILES,               DESTDRIVE'\language\codepage', 'T')
        rc=copyFiles(UNICODEKEYBOARDFILE, DESTDRIVE'\language\keyboard', 'T')
        rc=copyFiles(UNICODEUSERFILE,     DESTDRIVE'\language\locale',   'T')
    end

    /* HPFS386 support files */
    if IMAGENR=2 & isSet(HPFS386) then do
        if isSet(CREATEBOOTCD) & isSet(MEMDISK) then
            DESTDRIVE=VIRTUALDRIVE'\TOTAR.###'
        else
            DESTDRIVE=VIRTUALDRIVE
        rc=createDir(DESTDRIVE'\IBM386FS')
        rc=copyFiles(HPFS386IFSFILE HPFS386FILES, DESTDRIVE'\ibm386fs', 'T')
        rc=createFile(DESTDRIVE'\IBM386FS\HPFS386.INI', HPFS386INI)
    end

    return rc


/* ------------------------------------------------------------------------ */
/* Copy files to destination                                                */
/* ARG1: Destination                                                        */
/*         CD=CD/DVD or Maintenance Partition                               */
/*         IMG=Boot Image (of CD/DVD) or Maintenance Partition              */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
copyOtherFiles:
    DEST=ARG(1)
    IMGNR=2
    call progress 1, 'Copying other files to' destinationName()

    /* Files to copy to os2 directory of the CD/DVD or image */
    FILES=''
    if                                     DEST=EXTRAFILESDESTINATION      then FILES=FILES EXTRAFILES
    if isSet(OS2APPS)                    & DEST=OS2APPSDESTINATION         then FILES=FILES OS2APPFILES
    if isSet(OS2EXTRA)                   & DEST=OS2EXTRADESTINATION        then FILES=FILES OS2EXTRAFILES
    if isSet(REXX)                       & DEST=REXXDESTINATION            then FILES=FILES REXXFILES ifSet(WPSBOOT,'',REXXCOMMANDLINEFILE)
    if isSet(WPS)                        & DEST=WPSDESTINATION             then FILES=FILES WPSFILES VIDEODRIVERFILES.VIDEODRIVER
    if                                     DEST=ifSet(MAKEROOM,'CD','IMG') then FILES=FILES makeRoomFiles()
    if isSet(AUTOCHECK) & isSet(MEMDISK) & DEST='IMG'                      then FILES=FILES autoCheckFiles()

    if  DEST=EXTRADEVICEDESTINATION     then
        do I=1 to EXTRADEVICE
            FILES=FILES EXTRADEVICEFILE.I
        end
    FILES=strip(FILES)
    if FILES<>'' then do
        rc=createDir(DESTDRIVE'\OS2')
        rc=copyFiles(FILES, DESTDRIVE'\os2',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end
    if isSet(REXX) & DEST=REXXDESTINATION then do
        do I=1 to REXXFILES_.0
            select
                when isSet(OBJECTREXX)=isSet(OBJECTREXXINSTALLED) then rc=sCopyFile(REXXFILES_.I,  DESTDRIVE'\os2')
                when isSet(OBJECTREXX)                            then rc=sCopyFile(REXXFILES_O.I, DESTDRIVE'\os2\'REXXFILES_.I)
                otherwise                                              rc=sCopyFile(REXXFILES_C.I, DESTDRIVE'\os2\'REXXFILES_.I)
            end
        end
    end

    /* Files to copy to os2\dll directory of CD/DVD or image */
    FILES=''
    if                                                    DEST=EXTRADLLFILESDESTINATION    then FILES=FILES EXTRADLLFILES
    if isSet(OS2APPS)                                   & DEST=OS2APPSDESTINATION          then FILES=FILES OS2APPDLLFILES
    if isSet(OS2EXTRA)                                  & DEST=OS2EXTRADESTINATION         then FILES=FILES OS2EXTRADLLFILES
    if isSet(REXX)                                      & DEST=REXXDESTINATION             then FILES=FILES REXXDLLFILES
    if isSet(WPS)                                       & DEST=WPSDESTINATION              then FILES=FILES WPSDLLFILES VIDEODRIVERDLLFILES.VIDEODRIVER
    if                                                    DEST=ifSet(MAKEROOM,'CD','IMG')  then FILES=FILES makeRoomDLLFiles()
    if isSet(TCPIP431) & isSet(TCPIPDHCP) & \isSet(WPS) & DEST=NETWORKINGDESTINATION       then FILES=FILES TCPIP431WPSDLLFILES
    if isSet(PEER) & \isSet(OS2APPS)                    & DEST=NETWORKINGDESTINATION       then FILES=FILES PEERDLLFILES
    if isSet(AUTOCHECK) & isSet(MEMDISK)                & DEST='IMG'                       then FILES=FILES autoCheckDLLFiles()
    FILES=strip(FILES)
    if FILES<>'' then do
        rc=createDir(DESTDRIVE'\OS2\DLL')
        rc=copyFiles(FILES, DESTDRIVE'\os2\dll',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end
    if isSet(REXX) & DEST=REXXDESTINATION then do
        rc=createDir(DESTDRIVE'\OS2\DLL')
        do I=1 to REXXDLLFILES_.0
            select
                when isSet(OBJECTREXX)=isSet(OBJECTREXXINSTALLED) then rc=sCopyFile(REXXDLLFILES_.I,  DESTDRIVE'\os2\dll')
                when isSet(OBJECTREXX)                            then rc=sCopyFile(REXXDLLFILES_O.I, DESTDRIVE'\os2\dll\'REXXDLLFILES_.I)
                otherwise                                              rc=sCopyFile(REXXDLLFILES_C.I, DESTDRIVE'\os2\dll\'REXXDLLFILES_.I)
            end
        end
    end


    /* Files to copy to ibmcom directory of CD/DVD or image */
    if isSet(NETWORKING) & DEST=IBMCOMDESTINATION then do
        rc=createDir(DESTDRIVE'\IBMCOM')
        FILES=NETWORKINGIBMCOMFILES
        rc=copyFiles(FILES,DESTDRIVE'\IBMCOM',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to ibmcom\dll directory of CD/DVD or image */
    if isSet(NETWORKING) & DEST=IBMCOMDESTINATION then do
        rc=createDir(DESTDRIVE'\IBMCOM\DLL')
        FILES=NETWORKINGIBMCOMDLLFILES
        if isSet(NETBIOS)      then FILES=FILES NETBIOSIBMCOMDLLFILES
        if isSet(NETBIOSTCPIP) then FILES=FILES NETBIOSTCPIPIBMCOMDLLFILES
        if FILES<>'' then
            rc=copyFiles(FILES,DESTDRIVE'\IBMCOM\DLL',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to ibmcom\macs directory of CD/DVD or image */
    if isSet(NETWORKING) & (isSet(TCPIPLAN) | isSet(NETBIOS)) & DEST=IBMCOMDESTINATION then do
        rc=createDir(DESTDRIVE'\IBMCOM\MACS')
        FILES=NETWORKCARDFILE
        rc=copyFiles(FILES,DESTDRIVE'\IBMCOM\MACS',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to ibmcom\protocol directory of CD/DVD or image */
    if isSet(NETWORKING) & DEST=IBMCOMDESTINATION then do
        rc=createDir(DESTDRIVE'\IBMCOM\PROTOCOL')
        FILES=NETWORKINGIBMCOMPROTOCOLFILES
        if isSet(NETBIOS)      then FILES=FILES NETBIOSIBMCOMPROTOCOLFILES
        if isSet(NETBIOSTCPIP) then FILES=FILES NETBIOSTCPIPIBMCOMPROTOCOLFILES
        rc=copyFiles(FILES,DESTDRIVE'\IBMCOM\PROTOCOL',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to mptn\bin directory of CD/DVD or image */
    if isSet(NETWORKING) & isSet(TCPIP) & DEST=NETWORKINGDESTINATION then do
        rc=createDir(DESTDRIVE'\MPTN\BIN')
        FILES=TCPIPMPTNBINFILES
        rc=copyFiles(FILES,DESTDRIVE'\MPTN\BIN',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to mptn\dll directory of CD/DVD or image */
    if isSet(NETWORKING) & isSet(TCPIP) & DEST=NETWORKINGDESTINATION then do
        rc=createDir(DESTDRIVE'\MPTN\DLL')
        FILES=TCPIPMPTNDLLFILES
        rc=copyFiles(FILES,DESTDRIVE'\MPTN\DLL',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to mptn\etc directory of CD/DVD or image */
    if isSet(ACPI) & DEST='IMG' then do
        rc=createDir(DESTDRIVE'\MPTN\ETC')
        rc=copyFiles(ACPIDCFGFILE,DESTDRIVE'\MPTN\ETC',ifSet(CREATEBOOTCD,'T',''))
    end

    /* Files to copy to mptn\protocol directory of CD/DVD or image */
    if isSet(NETWORKING) & isSet(TCPIP) & DEST=NETWORKINGDESTINATION then do
        rc=createDir(DESTDRIVE'\MPTN\PROTOCOL')
        FILES=TCPIPMPTNPROTOCOLFILES EXTRATCPIPMPTNPROTOCOLFILES
        rc=copyFiles(FILES,DESTDRIVE'\MPTN\PROTOCOL',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to tcpip\bin directory of CD/DVD or image */
    if isSet(NETWORKING) & isSet(TCPIP) & DEST=NETWORKINGDESTINATION then do
        rc=createDir(DESTDRIVE'\TCPIP\BIN')
        FILES=TCPIPBINFILES
        rc=copyFiles(FILES,DESTDRIVE'\TCPIP\BIN',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to tcpip\dll directory of CD/DVD or image */
    if isSet(NETWORKING) & isSet(TCPIP) & DEST=NETWORKINGDESTINATION then do
        rc=createDir(DESTDRIVE'\TCPIP\DLL')
        FILES=TCPIPDLLFILES
        rc=copyFiles(FILES,DESTDRIVE'\MPTN\DLL',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to ibmlan\services directory of image (note: files can't go to CD!) */
    if isSet(PEER) & DEST='IMG' then do
        rc=createDir(DESTDRIVE'\IBMLAN\SERVICES')
        FILES=PEERIBMLANSERVICESFILES
        rc=copyFiles(FILES,DESTDRIVE'\IBMLAN\SERVICES',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to ibmlan\netlib directory of CD/DVD or image */
    if isSet(PEER) & DEST=NETWORKINGDESTINATION then do
        rc=createDir(DESTDRIVE'\IBMLAN\NETLIB')
        FILES=PEERIBMLANNETLIBFILES
        rc=copyFiles(FILES,DESTDRIVE'\IBMLAN\NETLIB',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to ibmlan\netprog directory of CD/DVD or image */
    if isSet(PEER) & DEST=NETWORKINGDESTINATION then do
        rc=createDir(DESTDRIVE'\IBMLAN\NETPROG')
        FILES=PEERIBMLANNETPROGFILES
        rc=copyFiles(FILES,DESTDRIVE'\IBMLAN\NETPROG',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to ibmlan\netprog\env\ibm\nwdde directory of CD/DVD or image */
    if isSet(PEER) & DEST=NETWORKINGDESTINATION then do
        rc=createDir(DESTDRIVE'\IBMLAN\NETPROG\ENV\IBM\NWDDE')
        FILES=PEERIBMLANNETPROGENVIBMNWDDEFILES
        rc=copyFiles(FILES,DESTDRIVE'\IBMLAN\NETPROG\ENV\IBM\NWDDE',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to muglib directory of CD/DVD or image */
    if isSet(PEER) & DEST=NETWORKINGDESTINATION then do
        rc=createDir(DESTDRIVE'\MUGLIB')
        FILES=PEERMUGLIBFILES
        rc=copyFiles(FILES,DESTDRIVE'\MUGLIB',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to muglib\dll directory of CD/DVD or image */
    if isSet(PEER) & DEST=NETWORKINGDESTINATION then do
        rc=createDir(DESTDRIVE'\MUGLIB\DLL')
        FILES=PEERMUGLIBDLLFILES
        rc=copyFiles(FILES,DESTDRIVE'\MUGLIB\DLL',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to psfonts directory of CD/DVD */
    if isSet(WPS) & DEST=WPSFONTSDESTINATION then do
        rc=createDir(DESTDRIVE'\PSFONTS')
        FILES=WPSPSFONTSFILES WPSEXTRAFONTFILES
        rc=copyFiles(FILES, DESTDRIVE'\psfonts')
        rc=createDir(DESTDRIVE'\PSFONTS\PFM')
        FILES=WPSPSFONTSPFMFILES WPSEXTRAPFMFONTFILES
        rc=copyFiles(FILES, DESTDRIVE'\psfonts\pfm')
    end

    /* Files to copy to os2\help directory of CD/DVD */
    if isSet(WPS) & DEST=WPSHELPDESTINATION then do
        rc=createDir(DESTDRIVE'\OS2\HELP')
        FILES=WPSHELPFILES VIDEODRIVERHELP.VIDEODRIVER EXTRAHELPFILES
        if isSet(WPSAPPS)      then FILES=FILES WPSAPPSHELPFILES
        if isSet(WPSEPM)       then FILES=FILES WPSEPMHELPFILES
        if isSet(REXX)         then FILES=FILES REXXHELPFILES
        if isSet(TCPIP)        then FILES=FILES TCPIPHELPFILES
        if isSet(MINILVM)      then FILES=FILES MINILVMHELPFILES
        rc=copyFiles(FILES, DESTDRIVE'\os2\help')
    end
    if isSet(WPS) & DEST=WPSHELPDESTINATION & isSet(REXX) then do
        do I=1 to REXXHELPFILES_.0
            select
                when isSet(OBJECTREXX)=isSet(OBJECTREXXINSTALLED) then rc=sCopyFile(REXXHELPFILES_.I,  DESTDRIVE'\os2\help')
                when isSet(OBJECTREXX)                            then rc=sCopyFile(REXXHELPFILES_O.I, DESTDRIVE'\os2\help\'REXXHELPFILES_.I)
                otherwise                                              rc=sCopyFile(REXXHELPFILES_C.I, DESTDRIVE'\os2\help\'REXXHELPFILES_.I)
            end
        end
    end

    if isSet(WPSAPPS) & DEST=WPSAPPSDESTINATION then do
        /* Files to copy to os2\apps directory of CD/DVD */
        rc=createDir(DESTDRIVE'\OS2\APPS')
        rc=copyFiles(WPSAPPSFILES, DESTDRIVE'\OS2\APPS')

        /* Files to copy to os2\apps\dll directory of CD/DVD */
        rc=createDir(DESTDRIVE'\OS2\APPS\DLL')
        rc=copyFiles(WPSAPPSDLLFILES, DESTDRIVE'\OS2\APPS\DLL')
    end

    if isSet(WPSEPM) & DEST=WPSEPMDESTINATION then do
        /* Files to copy to os2\apps directory of CD/DVD */
        rc=createDir(DESTDRIVE'\OS2\APPS')
        rc=copyFiles(WPSEPMFILES, DESTDRIVE'\OS2\APPS')

        /* Files to copy to os2\apps\dll directory of CD/DVD */
        rc=createDir(DESTDRIVE'\OS2\APPS\DLL')
        rc=copyFiles(WPSEPMDLLFILES, DESTDRIVE'\OS2\APPS\DLL')
    end

    if isSet(WPSREGEDIT2) & DEST=WPSREGEDIT2DESTINATION then do
        /* Files to copy to os2\system directory of CD/DVD */
        rc=createDir(DESTDRIVE'\OS2\SYSTEM')
        rc=copyFiles(WPSREGEDIT2SYSTEMFILES, DESTDRIVE'\OS2\SYSTEM')

        /* Files to copy to os2\dll directory of CD/DVD */
        rc=createDir(DESTDRIVE'\OS2\DLL')
        rc=copyFiles(WPSREGEDIT2DLLFILES, DESTDRIVE'\OS2\DLL')
    end

    /* Extra directories */
    do I=1 to EXTRADIRECTORIES
        if DEST=EXTRADIRECTORYDESTINATION.I then do
            rc=createDir(DESTDRIVE||EXTRADIRECTORY.I)
            rc=copyFiles(EXTRADIRECTORYFILES.I,DESTDRIVE||EXTRADIRECTORY.I,,ifClause(isSet(EXTRADIRECTORYIGNORENOTFOUND.I)|isSet(EXTRADIRECTORYRECURSE.I),'I',''))
            if isSet(EXTRADIRECTORYRECURSE.I) then do
                rc=parseFileList(EXTRADIRECTORYFILES.I)
                drop EDFILELIST.
                EDFILELIST.0=FILELIST.0
                do II=1 to FILELIST.0
                    EDFILELIST.II=FILELIST.II
                end
                do II=1 to EDFILELIST.0
                    BASEDIR=filespec('Drive', EDFILELIST.II)||filespec('Path', EDFILELIST.II)
                    if substr(BASEDIR,2,2)<>':\' then
                        call warningMessage 'Ignored invalid recursive extra directory specification' EDFILELIST.II
                    else do
                        FILEMASK=filespec('Name', EDFILELIST.II)
                        drop DIRTREE.
                        rc=sysFileTree(BASEDIR'*', 'DIRTREE', 'DOS')
                        if rc<>0 then call error 'Failed to retrieve directory tree for recursive extra directory' EDFILELIST.II
                        do III=1 to DIRTREE.0
                            NEWDIR=EXTRADIRECTORY.I||substr(DIRTREE.III,length(BASEDIR))
                            rc=createDir(DESTDRIVE||NEWDIR)
                            rc=copyFiles('"'||DIRTREE.III||'\'||FILEMASK||'"',DESTDRIVE||NEWDIR,,'I')
                        end
                    end
                end
            end
        end
    end

    do I=1 to EXTRAZIPWPIS
        if DEST=EXTRAZIPWPIDESTINATION.I then do
            ZIPWPIFILE=fileExists(EXTRAZIPWPIFILE.I)
            if ZIPWPIFILE='' then call error 'Unable to find zip file' EXTRAZIPWPIFILE.I
            rc=createDir(DESTDRIVE||EXTRAZIPWPIDIRECTORY.I)
            if isSet(EXTRAZIPWPIISWPI.I) then do
                call progress 2, 'Unpacking WPI' ZIPWPIFILE '=>' DESTDRIVE||EXTRAZIPWPIDIRECTORY.I
                PREVDIR=directory()
                PREVDIR_=directory(driveName(DESTDRIVE||EXTRAZIPWPIDIRECTORY.I))
                rc=directory(DESTDRIVE||EXTRAZIPWPIDIRECTORY.I)
                /* Create command file for running wic making sure wpirtl.dll is on the libpath */
                rc=createFile(BOOTABLETEMP'\wic.cmd','@SET ENDLIBPATH=%ENDLIBPATH%;'pathName(WPIRTLDLLFILE)';'||CRLF||'@'WICCOMMAND '"'ZIPWPIFILE'" -x' EXTRAZIPWPIOPTIONS.I)
                rc=runCmd('cmd /c' BOOTABLETEMP'\wic.cmd', 'Unable to unpack WPI' ZIPWPIFILE 'to' DESTDRIVE||EXTRAZIPWPIDIRECTORY.I 'rc('rc')')
                rc=directory(PREVDIR_)
                rc=directory(PREVDIR)
            end
            else do
                call progress 2, 'Unziping' ZIPWPIFILE '=>' DESTDRIVE||EXTRAZIPWPIDIRECTORY.I
                rc=runCMD(UNZIPCOMMAND '-C -o -d "'DESTDRIVE||EXTRAZIPWPIDIRECTORY.I'" "'ZIPWPIFILE'"' EXTRAZIPWPIOPTIONS.I, 'Unable to unzip' ZIPWPIFILE 'to' DESTDRIVE||EXTRAZIPWPIDIRECTORY.I 'rc('rc')')
            end
        end
    end

    return 0

/* ------------------------------------------------------------------------ */
/* Create config.sys file                                                   */
/* ARG: None                                                                */
/* RET: 0                                                                   */
/* ------------------------------------------------------------------------ */
createConfigSys:
    call progress 1, 'Creating config.sys file on' bootImageName()

    /* Drive and paths for boot image files */
    BASEDRIVE=destinationDrive('IMG')
    if \isSet(MEMDISK) then do
        BASEPATH=BASEDRIVE'\OS2'
        BASEDLLPATH=BASEDRIVE'\OS2\DLL'
    end
    else do
        BASEPATH=BASEDRIVE
        BASEDLLPATH=BASEDRIVE
    end

    /* Drive and paths for files that can reside on boot image or on CD/DVD (MAKEROOM) */
    RESTDRIVE=destinationDrive(ifSet(MAKEROOM,'CD','IMG'))
    RESTPATH=RESTDRIVE'\OS2'
    RESTDLLPATH=RESTDRIVE'\OS2\DLL'

    /* Other drives and paths */
    NETWORKINGDRIVE=destinationDrive(NETWORKINGDESTINATION)
    IBMCOMDRIVE=destinationDrive(IBMCOMDESTINATION)
    GENMACDRIVE=destinationDrive(GENMACDESTINATION)
    XWLANDRIVE=destinationDrive(XWLANDDESTINATION)
    HPFS386PATH=destinationDrive('IMG')'\IBM386FS'
    EXTRADEVICEPATH=destinationDrive(EXTRADEVICEDESTINATION)'\OS2'
    WPSPATH=destinationDrive(WPSDESTINATION)'\OS2'
    WPSHELPPATH=destinationDrive(WPSHELPDESTINATION)'\OS2\HELP'
    WPSAPPSPATH=destinationDrive(WPSAPPSDESTINATION)'\OS2\APPS'
    WPSEPMPATH=destinationDrive(WPSEPMDESTINATION)'\OS2\APPS'
    WPSREGEDIT2PATH=destinationDrive(WPSREGEDIT2DESTINATION)'\OS2'

    /* Truncate (makeroom) file names only if they are written to the boot image */
    if isSet(MAKEROOM) | \isSet(CREATEBOOTCD) then
        TRUNCATE=''
    else
        TRUNCATE='T'

    /* Show all output of commands executed from the config.sys? */
    if isSet(BOOTMESSAGES) then
        IGNORECMDOUTPUT=''
    else
        IGNORECMDOUTPUT='> /dev/nul'

    /* Add DLLBASING=OFF? */
    if isSet(DLLBASINGLINE) then
        rc=configLine('DLLBASING=OFF')

    /* Allow access to >16MB early in boot? */
    if isSet(EARLYMEMINIT) then
        rc=configLine('EARLYMEMINIT=TRUE')

    /* Memory Disk preparations */
    if isSet(MEMDISK) then do
        do I=1 to TARDIRS.0
            rc=configLine('DEVICE='BASEDRIVE'\loaddll.sys'  BASEDRIVE'\untar.dll untar' BASEDRIVE'\'TARDIRS.I'.tar')
        end
        if TARDIRS.0>0 then do
            rc=configLine('DEVICE='BASEDRIVE'\erasefil.sys' BASEDRIVE'\untar.dll')
            rc=configLine('DEVICE='BASEDRIVE'\erasefil.sys' BASEDRIVE'\untar.msg')
            rc=configLine('DEVICE='BASEDRIVE'\erasefil.sys' BASEDRIVE'\loaddll.sys')
            rc=configLine('DEVICE='BASEDRIVE'\erasefil.sys' BASEDRIVE'\loaddll.msg')
        end
    end

    /* Reservedriveletter */
    if RESERVEDRIVELETTER<>'' then
        rc=configLine('RESERVEDRIVELETTER='RESERVEDRIVELETTER)

    /* Paths */
    LIBPATHLINE='LIBPATH=.;'
    LIBPATHLINE=addToPathLine(LIBPATHLINE, ifClause(BASEDLLPATH=BASEDRIVE,BASEDRIVE'\',BASEDLLPATH))
    LIBPATHLINE=addToPathLine(LIBPATHLINE, BASEDRIVE'\OS2\DLL')
    LIBPATHLINE=addToPathLine(LIBPATHLINE, CDROMDRIVE'\OS2\DLL')
    LIBPATHLINE=addToPathLine(LIBPATHLINE, RESTDLLPATH)
    if isSet(GCCRT)       then LIBPATHLINE=addtoPathLine(LIBPATHLINE, destinationDrive(GCCRTDESTINATION)||GCCRTDIRECTORY)
    if isSet(WPSAPPS)     then LIBPATHLINE=addToPathLine(LIBPATHLINE, WPSAPPSPATH'\DLL')
    if isSet(WPSEPM)      then LIBPATHLINE=addToPathLine(LIBPATHLINE, WPSEPMPATH'\DLL')
    if isSet(WPSREGEDIT2) then LIBPATHLINE=addToPathLine(LIBPATHLINE, WPSREGEDIT2PATH'\DLL')
    if isSet(NETWORKING)  then LIBPATHLINE=addToPathLine(LIBPATHLINE, IBMCOMDRIVE'\IBMCOM\DLL')
    if isSet(TCPIP)       then LIBPATHLINE=addToPathLine(LIBPATHLINE, NETWORKINGDRIVE'\MPTN\DLL;'NETWORKINGDRIVE'\TCPIP\DLL')
    if isSet(PEER)        then LIBPATHLINE=addToPathLine(LIBPATHLINE, NETWORKINGDRIVE'\IBMLAN\NETLIB;'NETWORKINGDRIVE'\MUGLIB\DLL')
    if isSet(HPFS386)     then LIBPATHLINE=addToPathLine(LIBPATHLINE, HPFS386PATH)
    PATHLINE='SET PATH='
    PATHLINE=addToPathLine(PATHLINE, ifClause(BASEPATH=BASEDRIVE,BASEDRIVE'\',BASEPATH))
    PATHLINE=addToPathLine(PATHLINE, BASEDRIVE'\OS2')
    PATHLINE=addToPathLine(PATHLINE, CDROMDRIVE'\OS2')
    PATHLINE=addToPathLine(PATHLINE, RESTPATH)
    if isSet(WPSAPPS)     then PATHLINE=addToPathLine(PATHLINE, WPSAPPSPATH)
    if isSet(WPSEPM)      then PATHLINE=addToPathLine(PATHLINE, WPSEPMPATH)
    if isSet(WPSREGEDIT2) then PATHLINE=addToPathLine(PATHLINE, WPSREGEDIT2PATH'\SYSTEM')
    if isSet(NETWORKING)  then PATHLINE=addToPathLine(PATHLINE, IBMCOMDRIVE'\IBMCOM')
    if isSet(TCPIP)       then PATHLINE=addToPathLine(PATHLINE, NETWORKINGDRIVE'\MPTN\BIN')
    if isSet(TCPIP)       then PATHLINE=addToPathLine(PATHLINE, NETWORKINGDRIVE'\TCPIP\BIN')
    if isSet(PEER)        then PATHLINE=addToPathLine(PATHLINE, NETWORKINGDRIVE'\IBMLAN\NETPROG')
    if isSet(PEER)        then PATHLINE=addToPathLine(PATHLINE, NETWORKINGDRIVE'\MUGLIB')
    if isSet(HPFS386)     then PATHLINE=addToPathLine(PATHLINE, HPFS386PATH)
    DPATHLINE='SET DPATH='
    DPATHLINE=addToPathLine(DPATHLINE, ifClause(BASEPATH=BASEDRIVE,BASEDRIVE'\',BASEPATH))
    DPATHLINE=addToPathLine(DPATHLINE, BASEDRIVE'\OS2')
    DPATHLINE=addToPathLine(DPATHLINE, CDROMDRIVE'\OS2')
    DPATHLINE=addToPathLine(DPATHLINE, RESTPATH)
    if isSet(WPSAPPS)     then DPATHLINE=addToPathLine(DPATHLINE, WPSAPPSPATH)
    if isSet(WPSEPM)      then DPATHLINE=addToPathLine(DPATHLINE, WPSEPMPATH)
    if isSet(WPSREGEDIT2) then DPATHLINE=addToPathLine(DPATHLINE, WPSREGEDIT2PATH'\SYSTEM')
    if isSet(NETWORKING)  then DPATHLINE=addToPathLine(DPATHLINE, IBMCOMDRIVE'\IBMCOM')
    if isSet(TCPIP)       then DPATHLINE=addToPathLine(DPATHLINE, NETWORKINGDRIVE'\MPTN\BIN')
    if isSet(TCPIP)       then DPATHLINE=addToPathLine(DPATHLINE, NETWORKINGDRIVE'\MPTN')
    if isSet(PEER)        then DPATHLINE=addToPathLine(DPATHLINE, NETWORKINGDRIVE'\IBMLAN\NETPROG')
    if isSet(PEER)        then DPATHLINE=addToPathLine(DPATHLINE, NETWORKINGDRIVE'\IBMLAN')
    if isSet(PEER)        then DPATHLINE=addToPathLine(DPATHLINE, NETWORKINGDRIVE'\MUGLIB')
    if isSet(HPFS386)     then DPATHLINE=addToPathLine(DPATHLINE, HPFS386PATH)
    HELPPATHLINE='SET HELP='WPSHELPPATH';'
    BOOKSHELFLINE='SET BOOKSHELF='WPSHELPPATH';'

    do I=1 to EXTRADIRECTORIES
        ADDTOPATHS=EXTRADIRECTORYADDTOPATHS.I
        do while words(ADDTOPATHS)>0
            ADDTOPATH=word(ADDTOPATHS,1)
            ADDTOPATHS=strip(delword(ADDTOPATHS,1,1))
            ADDTOPATHTXT=destinationDrive(EXTRADIRECTORYDESTINATION.I)||EXTRADIRECTORY.I
            select
                when ADDTOPATH='LIBPATH'   then LIBPATHLINE=addToPathLine(LIBPATHLINE, ADDTOPATHTXT)
                when ADDTOPATH='PATH'      then PATHLINE=addToPathLine(PATHLINE, ADDTOPATHTXT)
                when ADDTOPATH='DPATH'     then DPATHLINE=addToPathLine(DPATHLINE, ADDTOPATHTXT)
                when ADDTOPATH='HELP'      then HELPPATHLINE=addToPathLine(HELPPATHLINE, ADDTOPATHTXT)
                when ADDTOPATH='BOOKSHELF' then BOOKSHELFLINE=addToPathLine(BOOKSHELFLINE, ADDTOPATHTXT)
                otherwise ;
            end
        end
    end
    do I=1 to EXTRAZIPWPIS
        ADDTOPATHS=EXTRAZIPWPIADDTOPATHS.I
        do while words(ADDTOPATHS)>0
            ADDTOPATH=word(ADDTOPATHS,1)
            ADDTOPATHS=strip(delword(ADDTOPATHS,1,1))
            ADDTOPATHTXT=destinationDrive(EXTRADIRECTORYDESTINATION.I)||EXTRADIRECTORY.I
            select
                when ADDTOPATH='LIBPATH'   then LIBPATHLINE=addToPathLine(LIBPATHLINE, ADDTOPATHTXT)
                when ADDTOPATH='PATH'      then PATHLINE=addToPathLine(PATHLINE, ADDTOPATHTXT)
                when ADDTOPATH='DPATH'     then DPATHLINE=addToPathLine(DPATHLINE, ADDTOPATHTXT)
                when ADDTOPATH='HELP'      then HELPPATHLINE=addToPathLine(HELPPATHLINE, ADDTOPATHTXT)
                when ADDTOPATH='BOOKSHELF' then BOOKSHELFLINE=addToPathLine(BOOKSHELFLINE, ADDTOPATHTXT)
                otherwise ;
            end
        end
    end
    rc=configLine(LIBPATHLINE)
    rc=configLine(PATHLINE)
    rc=configLine(DPATHLINE)
    if isSet(WPS) then do
        rc=configLine(HELPPATHLINE)
        rc=configLine(BOOKSHELFLINE)
    end

    /* Unicode */
    if isSet(UNICODE) then do
        rc=configLine('SET LANG='UNILANG)
        rc=configLine('SET ULSPATH='BASEDRIVE'\language')
        rc=configLine('DEVICE='BASEPATH'\'fileName(UNICODEFILE,'T'))
    end

    /* File systems */
    if isSet(CDFS)    then rc=configLine('IFS='BASEPATH'\'fileName(CDFSIFSFILE,'T')      CDFSOPTIONS)
    if isSet(FAT32)   then rc=configLine('IFS='BASEPATH'\'fileName(FAT32IFSFILE,'T')     FAT32OPTIONS)
    if isSet(FAT32)   then rc=configLine(ifSet(FAT32CACHEF,'','REM ')'CALL='RESTPATH'\'fileName(FAT32CACHEFILE,'T') FAT32CACHEOPTIONS)
    if isSet(NTFS)    then rc=configLine('IFS='BASEPATH'\'fileName(NTFSIFSFILE,'T')      NTFSOPTIONS)
    if isSet(HPFS)    then rc=configLine('IFS='BASEPATH'\'fileName(HPFSIFSFILE,'T')      HPFSOPTIONS)
    if isSet(HPFS386) then rc=configLine('IFS='HPFS386PATH'\'fileName(HPFS386IFSFILE,'T')  HPFS386OPTIONS)
    if isSet(JFS)     then rc=configLine('IFS='BASEPATH'\'fileName(JFSIFSFILE,'T')       JFSOPTIONS)
    if isSet(UDF)     then rc=configLine('IFS='BASEPATH'\'fileName(UDFIFSFILE,'T')       UDFOPTIONS)

    /* Ramdisk */
    if isSet(RAMDISK) then do
        rc=configLine('IFS='BASEPATH'\'fileName(RAMDISKIFSFILE,'T') RAMDISKOPTIONS)
        rc=configLine('CALL='RESTPATH'\'fileName(RAMDISKFILE,TRUNCATE) RAMDISKDRIVE 'RAMDISK' IGNORECMDOUTPUT)
    end

    /* Base device driver for keyboard */
    rc=configLine('BASEDEV=ibmkbd.sys')

    /* Base device for chkdsk large disk support */
    rc=configLine('BASEDEV=chkdsk.sys')

    /* DASD Fallback */
    if isSet(DASDFALLBACK)                  then rc=configLine('BASEDEV='fileName(DASDFALLBACKFILE,'T')        DASDFALLBACKOPTIONS)

    /* Non-512 bytes/sector removable media support */
    if isSet(NON512DASD) then rc=configLine('BASEDEV='fileName(NON512DASDFILE) NON512DASDOPTIONS)

    /* Removable SCSI devices geometry filter support */
    if isSet(REMOVABLESCSI) then rc=configLine('BASEDEV='fileName(REMOVABLESCSIFILE,'T') REMOVABLESCSIOPTIONS)

    /* Base device drivers for storage devices */
    if \isSet(LVM) & isSet(MEMDISK)                   then rc=configLine('BASEDEV='fileName(MEMDISKADDFILE, 'T') MEMDISKADDOPTIONS)
    if isSet(FLOPPY)                                  then rc=configLine('BASEDEV='fileName(FLOPPYADDFILE,'T')   FLOPPYOPTIONS)
    if isSet(AHCI) & \isSet(SCSIBEFOREIDE) & isSet(AHCIBEFOREIDE)
                                                      then rc=configLine('BASEDEV='fileName(AHCIADDFILE,'T')    AHCIOPTIONS)
    if isSet(IDE)  & \isSet(SCSIBEFOREIDE) & (isSet(IBMIDE) | isSet(DANIIDE))
                                                      then rc=configLine('BASEDEV='fileName(IDEADDFILE,'T')      IDEOPTIONS)
    if isSet(AHCI) & \isSet(SCSIBEFOREIDE) & \isSet(AHCIBEFOREIDE)
                                                      then rc=configLine('BASEDEV='fileName(AHCIADDFILE,'T')    AHCIOPTIONS)
    if isSet(SCSIDMD)                                 then rc=configLine('BASEDEV='fileName(SCSIDMDFILE,'T')     SCSIDMDOPTIONS)
    do I=1 to SCSI
        rc=configLine('BASEDEV='fileName(SCSIADDFILE.I,'T') SCSIOPTIONS.I)
    end
    if isSet(ACHI) & isSet(SCSIBEFOREIDE) & isSet(AHCIBEFOREIDE)
                                                      then rc=configLine('BASEDEV='fileName(AHCIADDFILE,'T') AHCIOPTIONS)
    if isSet(IDE) & isSet(SCSIBEFOREIDE) & (isSet(IBMIDE) | isSet(DANIIDE))
                                                      then rc=configLine('BASEDEV='fileName(IDEADDFILE,'T') IDEOPTIONS)
    if isSet(ACHI) & isSet(SCSIBEFOREIDE) & \isSet(AHCIBEFOREIDE)
                                                      then rc=configLine('BASEDEV='fileName(AHCIADDFILE,'T') AHCIOPTIONS)
    if isSet(ATAPI) then do
        rc=configLine('BASEDEV='fileName(ATAPIFLTFILE,'T') ATAPIOPTIONS)
        if \isSet(DANIATAPI) & \isSet(LATESTATAPI) then rc=configLine('BASEDEV='fileName(ATAPICDFLTFILE,'T') ATAPICDOPTIONS)
    end
    if isSet(LVM) & isSet(MEMDISK) then rc=configLine('BASEDEV='fileName(MEMDISKADDFILE, 'T')  MEMDISKADDOPTIONS)
    if isSet(PPZIP)                then rc=configLine('BASEDEV='fileName(PPZIPADDFILE,'T')     PPZIPOPTIONS)
    if isSet(HD4DISK)              then rc=configLine('BASEDEV='fileName(HD4DISKFILE,'T')      HD4DISKOPTIONS)
                                        rc=configLine('BASEDEV='fileName(DASDFILE,'T')         DASDOPTIONS)
    if isSet(LVM)                  then rc=configLine('BASEDEV='fileName(LVMDMDFILE,'T')       LVMOPTIONS)

    /* Extra base device drivers */
    do I=1 to EXTRABASEDEV
        rc=configLine('BASEDEV='fileName(EXTRABASEDEVFILE.I,'T') EXTRABASEDEVOPTIONS.I)
    end

    /* Base device driver providing bootable CD/DVD support */
    if isSet(CREATEBOOTCD) & \isSet(MEMDISK) then rc=configLine('BASEDEV='fileName(BOOTFLTFILE,'T') BOOTOPTIONS)

    /* Codepage, display, and keyboard layout */
    rc=configLine('COUNTRY='COUNTRY','BASEPATH'\country.sys')
    rc=configLine('CODEPAGE='CODEPAGE)
    rc=configLine('DEVINFO=KBD,'KEYBOARD','BASEPATH'\keyboard.dcp')
    rc=configLine('DEVINFO=SCR,VGA,'BASEPATH'\viotbl.dcp')

    /* CDROM support */
    if isSet(CDROM) then rc=configLine('DEVICE='BASEPATH'\'fileName(CDROMDMDFILE,'T') CDROMOPTIONS)

    /* ASPI support */
    if isSet(ASPI)  then rc=configLine('BASEDEV='fileName(ASPIDMDFILE,'T') ASPIOPTIONS)

    /* Test device driver */
    if isSet(TESTCFGSYS) then  rc=configLine('DEVICE='RESTPATH'\testcfg.sys')

    /* Test device driver (v2)  */
    if isSet(TESTCFG2SYS) then  rc=configLine('DEVICE='RESTPATH'\testcfg2.sys')

    /* DOS.SYS device driver */
    rc=configLine('DEVICE='RESTPATH'\dos.sys')

    /* Mouse support */
    if isSet(MOUSE) then do
        rc=configLine('DEVICE='RESTPATH'\'fileName(MOUSEPOINTDDFILE,TRUNCATE))
        if MOUSEDEVFILE<>'' then
            rc=configLine('DEVICE='RESTPATH'\'fileName(MOUSEDEVFILE,TRUNCATE) MOUSEDEVFILEOPTIONS)
        if \(isSet(WPS) & VIDEODRIVER=2 & isSet(MSVMADDITIONS)) & \isSet(VBOXADDITIONS) then
            rc=configLine('DEVICE='RESTPATH'\'fileName(MOUSEFILE,TRUNCATE) MOUSEFILEOPTIONS)
    end

    /* USB */
    if isSet(USB) then do
        do USBOHCI
            rc=configLine('BASEDEV='fileName(USBOHCIFILE,'T') USBOHCIOPTIONS)
        end
        do USBUHCI
            rc=configLine('BASEDEV='fileName(USBUHCIFILE,'T') USBUHCIOPTIONS)
        end
        do USBEHCI
            rc=configLine('BASEDEV='fileName(USBEHCIFILE,'T') USBEHCIOPTIONS)
        end
        rc=configLine('BASEDEV='fileName(USBDFILE,  'T') USBDOPTIONS)
        rc=configLine('BASEDEV='fileName(USBHIDFILE,'T') USBHIDOPTIONS)
        if isSet(USBKEYB)    then rc=configLine('DEVICE='RESTPATH'\'fileName(USBKEYBFILE,   TRUNCATE) USBKEYBOPTIONS)
        if isSet(USBCOM)     then rc=configLine('DEVICE='RESTPATH'\'fileName(USBCOMFILE,    TRUNCATE) USBCOMOPTIONS)
        if isSet(USBMOUSE)   then rc=configLine('DEVICE='RESTPATH'\'fileName(USBMOUSEFILE,  TRUNCATE) USBMOUSEOPTIONS)
        if isSet(USBPRINTER) then rc=configLine('DEVICE='RESTPATH'\'fileName(USBPRINTERFILE,TRUNCATE) USBPRINTEROPTIONS)
        if isSet(USBCDROM)   then rc=configLine('BASEDEV='fileName(USBCDROMFILE,'T')   USBCDROMOPTIONS)
        if isSet(USBSTORAGE) then rc=configLine('BASEDEV='fileName(USBSTORAGEFILE,'T') USBSTORAGEOPTIONS)
    end

    /* Serial port support */
    if isSet(SERIALPORT) then
        rc=configLine('DEVICE='RESTPATH'\'fileName(SERIALPORTFILE,TRUNCATE) SERIALPORTOPTIONS)

    /* SIO Serial port support */
    if isSet(SIOSERIALPORT) then
        rc=configLine('DEVICE='RESTPATH'\'fileName(SIOSERIALPORTFILE,TRUNCATE) SIOSERIALPORTOPTIONS)

    /* SIO2K Serial port support */
    if isSet(SIO2KSERIALPORT) then do
        if isSet(SIO2KPATCH) then
            rc=configLine('DEVICE='RESTPATH'\'fileName(SIO2KPATCHFILE,TRUNCATE))
        rc=configLine('DEVICE='RESTPATH'\'fileName(SIO2KUARTFILE,TRUNCATE) SIO2KUARTOPTIONS)
        rc=configLine('DEVICE='RESTPATH'\'fileName(SIO2KSERIALPORTFILE,TRUNCATE) SIO2KSERIALPORTOPTIONS)
    end

    /* Parallel port support */
    if isSet(PARALLELPORT) then do
        rc=configLine('BASEDEV='fileName(PARALLELPORTFILE,TRUNCATE) PARALLELPORTOPTIONS)
        rc=configLine('PRINTMONBUFSIZE=134,134,134')
    end

    /* APM */
    if isSet(APM) then do
                                  rc=configLine('DEVICE='RESTPATH'\'fileName(APMFILE,TRUNCATE) APMOPTIONS)
        if APMDAEMONFILE<>'' then rc=configLine('RUN='RESTPATH'\'fileName(APMDAEMONFILE,TRUNCATE) APMDAEMONOPTIONS)
    end

    /* ACPI */
    if isSet(ACPI) then do
        rc=configLine('PSD=ACPI.PSD' ACPIPSDOPTIONS)
        if isSet(ACPIAPMADD) then rc=configline('BASEDEV=APM.ADD')
        if isSet(ACPIDAEMON) then rc=configLine('RUN='RESTPATH'\'fileName(ACPIDAEMONFILE,TRUNCATE))
    end

    /* APM / ACPI APM */
    if isSet(APM) | isSet(ACPIAPMADD) then do
        if APM_ENH_SUSPEND<>''      then rc=configLine('SET ENH_SUSPEND='APM_ENH_SUSPEND)
        if APM_SNAP_APM_REPAINT<>'' then rc=configLine('SET SNAP_APM_REPAINT='APM_SNAP_APM_REPAINT)
    end

    /* Temp directory */
    if \isSet(CREATEBOOTCD) | isSet(RAMDISK) | isSet(MEMDISK) then do
        if \isSet(CREATEBOOTCD) then do
            TEMPPATH=WRITERDRIVE'\Temp'
            rc=createDir(TEMPPATH)
        end
        else do
            TEMPPATH=RAMDISKDRIVE'\Temp'
            rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE) '/c if not exist' TEMPPATH 'mkdir' TEMPPATH IGNORECMDOUTPUT)
        end
        rc=configLine('SET TEMP='TEMPPATH)
        rc=configLine('SET TMP='TEMPPATH)
        rc=configLine('SUPPRESSPOPUPS='left(TEMPPATH,1))
    end

    /* WPS */
    if isSet(WPS) then do
        rc=configLine('DEVICE='WPSPATH'\pmdd.sys')
        if isSet(CREATEBOOTCD) & (WPSDESTINATION<>'IMG' | isSet(RAMDISK)) then do
            rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE) '/c if not exist' RAMDISKDRIVE'\OS2 mkdir' RAMDISKDRIVE'\OS2' IGNORECMDOUTPUT)
            rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE) '/c copy'  WPSPATH'\*.ini' RAMDISKDRIVE'\OS2' IGNORECMDOUTPUT)
        end
        rc=configLine('SET USER_INI='WPSDRIVE'\os2\os2.ini')
        rc=configLine('SET SYSTEM_INI='WPSDRIVE'\os2\os2sys.ini')
        rc=configLine('SET AUTOSTART='AUTOSTART)
        rc=configLine('SET RESTARTOBJECTS=STARTUPFOLDERSONLY')
        if isSet(WPSEPM)           then rc=configLine('SET EPMPATH='WPSEPMPATH)
        if isSet(WPSWCKILLFEATURE) then rc=configLine('SET SCKILLFEATUREENABLED=ON')

        /* Make sure required regedit2 files are in OS2\System of the boot drive */
        if isSet(WPSREGEDIT2) & isSet(CREATEBOOTCD) & WPSREGEDIT2DESTINATION<>'IMG' then do
            BASESYSTEM=BASEDRIVE'\OS2\SYSTEM'
            rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE) '/c if not exist' BASEDRIVE'\OS2        mkdir' BASEDRIVE'\OS2'        IGNORECMDOUTPUT)
            rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE) '/c if not exist' BASEDRIVE'\OS2\SYSTEM mkdir' BASEDRIVE'\OS2\SYSTEM' IGNORECMDOUTPUT)
            rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE) '/c if not exist' BASEDRIVE'\OS2\SYSTEM\registry.rep copy' WPSREGEDIT2PATH'\SYSTEM\registry.rep' BASEDRIVE'\OS2\SYSTEM' IGNORECMDOUTPUT)
        end
    end

    /* Networking */
    if isSet(NETWORKING) then do
        if \isSet(CREATEBOOTCD) | (isSet(MEMDISK) & IBMCOMDESTINATION='IMG') then
            rc=configLine('DEVICE='IBMCOMDRIVE'\ibmcom\lanmsgdd.os2 /I:'IBMCOMDRIVE'\IBMCOM /S')
        if isSet(GENMAC) then
            rc=configLine('RUN='GENMACDRIVE'\GENMAC\driver\HELPERW.EXE')
        if isSet(TCPIPLAN) | isSet(NETBIOS) then do
            rc=configLine('CALL='IBMCOMDRIVE'\ibmcom\protocol\netbind.exe')
            rc=configLine('DEVICE='IBMCOMDRIVE'\ibmcom\protman.os2  /I:'IBMCOMDRIVE'\IBMCOM')
        end

        /* TCP/IP */
        if isSet(TCPIP) then do
            TCPIPTEMP=TCPIPMPTNPROTOCOLFILES EXTRATCPIPMPTNPROTOCOLFILES
            do while TCPIPTEMP<>''
                parse upper var TCPIPTEMP DEV TCPIPTEMP
                rc=configLine('DEVICE='NETWORKINGDRIVE'\MPTN\PROTOCOL'||substr(DEV,lastpos('\',DEV)))
            end
            rc=configLine('SET HOSTNAME='TCPIPHOSTNAME)
            rc=configLine('SET USE_HOSTS_FIRST=1')
            ETCDIR=NETWORKINGDRIVE'\mptn\etc'
            if isSet(CREATEBOOTCD) & (isSet(RAMDISK)|isSet(MEMDISK)) & NETWORKINGDESTINATION='CD' then do
                TARGETDRIVE=ifSet(RAMDISK,RAMDISKDRIVE,destinationDrive('IMG'))
                rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE) '/c if not exist' TARGETDRIVE'\etc mkdir' TARGETDRIVE'\etc' IGNORECMDOUTPUT)
                rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE) '/c xcopy' ETCDIR'\*' TARGETDRIVE'\etc' IGNORECMDOUTPUT)
                ETCDIR=TARGETDRIVE'\ETC'
                drop TARGETDRIVE
            end
            rc=configLine('SET ETC='ETCDIR)
            rc=configLine('RUN='NETWORKINGDRIVE'\mptn\bin\cntrl.exe')
            rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE) '/c' NETWORKINGDRIVE'\mptn\bin\mptstart.cmd' IGNORECMDOUTPUT)
        end

        if \isSet(CREATEBOOTCD) | (isSet(MEMDISK) & IBMCOMDESTINATION='IMG') then
            rc=configLine('RUN='IBMCOMDRIVE'\ibmcom\lanmsgex.exe')

        /* NETBIOS */
        if isSet(NETBIOS) then do
            rc=configLine('DEVICE='IBMCOMDRIVE'\ibmcom\protocol\netbeui.os2')
        end

        /* NETBIOS over TCP/IP */
        if isSet(NETBIOSTCPIP) then do
            rc=configLine('DEVICE='IBMCOMDRIVE'\ibmcom\protocol\tcpbeui.os2')
            rc=configLine('RUN='IBMCOMDRIVE'\ibmcom\protocol\nbtcp.exe')
        end

        /* NETBIOS or NETBIOS over TCP/IP */
        if isSet(NETBIOS) | isSet(NETBIOSTCPIP) then do
            rc=configLine('DEVICE='IBMCOMDRIVE'\ibmcom\protocol\netbios.os2')
        end

        /* Card driver */
        if isSet(TCPIPLAN) | isSet(NETBIOS) then
            rc=configLine('DEVICE='IBMCOMDRIVE'\ibmcom\macs\'fileName(NETWORKCARDFILE))

        /* PEER */
        if isSet(PEER) then do
            rc=configLine('DEVICE='NETWORKINGDRIVE'\ibmlan\netprog\rdrhelp.200')
            rc=configLine('IFS='NETWORKINGDRIVE'\ibmlan\netprog\netwksta.200 /I:'destinationDrive('IMG')'\IBMLAN /N')
            rc=configLine('DEVICE='NETWORKINGDRIVE'\ibmlan\netprog\vnetapi.os2')
            rc=configLine('RUN='NETWORKINGDRIVE'\ibmlan\netprog\lsdaemon.exe')
            rc=configLine('RUN='NETWORKINGDRIVE'\ibmlan\netprog\vnrminit.exe')
            rc=configLine('SET NWDBPATH='NETWORKINGDRIVE'\ibmlan\netprog')
            rc=configLine('SET DLSINI='NETWORKINGDRIVE'\ibmlan\netprog\netgui.ini')
        end
    end

    /* Extra device drivers */
    do I=1 to EXTRADEVICE
        rc=configLine('DEVICE='EXTRADEVICEPATH'\'||fileName(EXTRADEVICEFILE.I,'T') EXTRADEVICEOPTIONS.I)
    end

    /* System settings */
    rc=configLine('MEMMAN='MEMMAN)
    if pos('NOSWAP',translate(MEMMAN))=0 then
        rc=configLine('SWAPPATH='SWAPPATH)
    rc=configLine('IOPL=YES')
    rc=configLine('PROTECTONLY='PROTECTONLY)
    rc=configLine('AUTOFAIL=YES') /* Necessary for WPS CD/DVD boot! */
    rc=configLine('THREADS='THREADS)
    rc=configLine('VIRTUALADDRESSLIMIT='VIRTUALADDRESSLIMIT)
    rc=configLine('DISKCACHE='FATCACHEOPTIONS)
    rc=configLine('MAXWAIT=1')

    /* Display settings */
    if isSet(WPS) then
    select
        when isSet(WPS) & VIDEODRIVER=2 then do
            rc=configLine('SET DISPLAYTYPE=VGA')
            rc=configLine('SET VIDEO_DEVICES=VIO_SVGA')
            rc=configLine('SET VIO_SVGA=DEVICE(BVHVGA,BVHSVGA)')
            if isSet(MSVMADDITIONS) then
                rc=configLine('SET C1=VMGRADD')
            else
                rc=configLine('SET C1=GENGRADD')
            rc=configLine('SET GRADD_CHAINS=C1')
            rc=configLine('DEVICE='WPSPATH'\gradd.sys')
        end
        when isSet(WPS) & VIDEODRIVER=3 then do
            SNAPBOOTDRIVE=destinationDrive('IMG')

            rc=configLine('SET VIDEO_DEVICES=VIO_SVGA')
            rc=configLine('SET VIO_SVGA=DEVICE(BVHVGA,BVHSVGA)')
            rc=configLine('SET C1=SDDGRADD')
            rc=configLine('SET GRADD_CHAINS=C1')
            rc=configLine('SET DISPLAYTYPE=VGA')
            rc=configLine('SET GREEXT=SDDGREXT')
            rc=configLine('SET SNAP_PATH='SNAPBOOTDRIVE'\OS2\DRIVERS\SNAP')
            rc=configLine('DEVICE='WPSPATH'\gradd.sys')
            rc=configLine('DEVICE='WPSPATH'\sddhelp.sys')

            /* Create (temp) paths and copy files */
            rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE)' /c if not exist 'SNAPBOOTDRIVE'\OS2                             mkdir 'SNAPBOOTDRIVE'\OS2' IGNORECMDOUTPUT)
            rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE)' /c if not exist 'SNAPBOOTDRIVE'\OS2\drivers                     mkdir 'SNAPBOOTDRIVE'\OS2\drivers' IGNORECMDOUTPUT)
            rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE)' /c if not exist 'SNAPBOOTDRIVE'\OS2\drivers\snap                mkdir 'SNAPBOOTDRIVE'\OS2\drivers\snap' IGNORECMDOUTPUT)
            rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE)' /c if not exist 'SNAPBOOTDRIVE'\OS2\drivers\snap\config         mkdir 'SNAPBOOTDRIVE'\OS2\drivers\snap\config' IGNORECMDOUTPUT)
            rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE)' /c if not exist 'SNAPBOOTDRIVE'\OS2\drivers\snap\config\modes   mkdir 'SNAPBOOTDRIVE'\OS2\drivers\snap\config\modes' IGNORECMDOUTPUT)
            rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE)' /c if not exist 'SNAPBOOTDRIVE'\OS2\drivers\snap\config\options mkdir 'SNAPBOOTDRIVE'\OS2\drivers\snap\config\options' IGNORECMDOUTPUT)
            rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE)' /c if not exist 'SNAPBOOTDRIVE'\OS2\install                     mkdir 'SNAPBOOTDRIVE'\OS2\install' IGNORECMDOUTPUT)

            if isSet(CREATEBOOTCD) & WPSDESTINATION<>'IMG' then do
                /* Make sure specific SNAP files are in the boot volume and writable */
                rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE)' /c copy' destinationDrive('CD')'\OS2\svgadata.pmi 'SNAPBOOTDRIVE'\OS2\svgadata.pmi')
                rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE)' /c copy' destinationDrive('CD')'\OS2\drivers\snap\*'        SNAPBOOTDRIVE'\OS2\drivers\snap\*')
                rc=configLine('CALL='fileName(SHELLFILE,TRUNCATE)' /c copy' destinationDrive('CD')'\OS2\drivers\snap\config\*' SNAPBOOTDRIVE'\OS2\drivers\snap\config\*')

                rc=configLine('CALL='destinationDrive(OS2APPSDESTINATION)'\os2\attrib.exe -r 'SNAPBOOTDRIVE'\OS2\svgadata.pmi')
                rc=configLine('CALL='destinationDrive(OS2APPSDESTINATION)'\os2\attrib.exe -r 'SNAPBOOTDRIVE'\OS2\drivers\* /S')
            end
        end
        otherwise do
            rc=configLine('SET DISPLAYTYPE=VGA')
            rc=configLine('SET VIDEO_DEVICES=VIO_VGA')
            rc=configLine('SET VIO_VGA=DEVICE(BVHVGA)')
        end
    end

    /* Command environment */
    TZ=value('TZ',,'OS2ENVIRONMENT')
    if TZ<>'' then rc=configLine('SET TZ='TZ)
    rc=configLine('SET PROMPT='SHELLPROMPT)
    rc=configLine('SET KEYS=ON')

    /* Shell */
    rc=configLine('PROTSHELL='ifSet(HPFS386,HPFS386PATH'\bootsh.exe ')RESTPATH'\'fileName(PROTSHELLFILE,TRUNCATE) PROTSHELLFILEOPTIONS)
    if isSet(WPS) then do
        rc=configLine('SET RUNWORKPLACE='RESTPATH'\'fileName(WPSRUNWORKPLACEFILE,TRUNCATE) WPSRUNWORKPLACEFILEOPTIONS)
    end
    rc=configLine('SET OS2_SHELL='RESTPATH'\'fileName(SHELLFILE,TRUNCATE) SHELLFILEOPTIONS)
    rc=configLine('SET COMSPEC='RESTPATH'\'fileName(SHELLFILE,TRUNCATE))

    /* REXX */
    if isSet(REXX) & \isSet(WPSBOOT) then
        rc=configLine('RUN='destinationDrive(REXXDESTINATION)'\os2\'fileName(REXXCOMMANDLINEFILE))

    /* MS Virtual Machine Additions */
    if isSet(MSVMADDITIONS) then do
        rc=configLine('DEVICE='RESTPATH'\vmdrv.sys')
        if (isSet(WPS) & VIDEODRIVER=2) then
            rc=configLine('DEVICE='RESTPATH'\vmmouse.sys')
        rc=configLine('IFS='RESTPATH'\vmmap.ifs')
    end

    /* VirtualBox Additions */
    if isSet(VBOXADDITIONS) then do
        rc=configLine('DEVICE='destinationDrive('CD')''VBOXADDITIONSDIRECTORY'\vboxguest.sys')
        rc=configLine('DEVICE='destinationDrive('CD')''VBOXADDITIONSDIRECTORY'\vboxmouse.sys')
    end

    /* Additional config.sys lines */
    rc=configLine(parseText(EXTRACONFIGSYS))

    /* Close config.sys */
    rc=lineout(VIRTUALDRIVE'\config.sys')

    return 0


/* ------------------------------------------------------------------------ */
/* Create INI files                                                         */
/* ARG: None                                                                */
/* RET: 0                                                                   */
/* Note: Destination is determined by the DESTINATION and DEST variables    */
/*   CD=CD/DVD or Maintenance Partition                                     */
/*   IMG=Boot Image (of CD/DVD) or Maintenance Partition                    */
/* ------------------------------------------------------------------------ */
createINIFiles:
    IMGNR=2
    call progress 1, 'Creating INI files on' destinationName()

    /* Determine addtional fonts to register */
    rc=parseFileList(WPSEXTRAFONTFILES)
    do I=1 to FILELIST.0
        EXT=translate(right(FILELIST.I,4))
        FNAME=translate(fileName(FILELIST.I))
        if EXT='.FON' | EXT='.OFM' | EXT='.CMB' | EXT='.TTF' then
            EXTRAUSERINI=EXTRAUSERINI||CRLF||'"PM_Fonts" "'FNAME'" "$$@destinationDrive(WPSFONTSDESTINATION)$$\PSFONTS\'FNAME'"'
    end

    /* Create ini file from EXTRAxxxINI settings */
    if strip(EXTRAUSERINI)<>'' then do
        EXTRAUSERINIRCFILES=EXTRAUSERINIRCFILES+1
        EXTRAUSERINIRCFILE.EXTRAUSERINIRCFILES=BOOTABLETEMP'\inirc.xtr'
        EXTRAUSERINIRCFILEUPDATE.EXTRAUSERINIRCFILES='YES'
        rc=createFile(EXTRAUSERINIRCFILE.EXTRAUSERINIRCFILES, EXTRAUSERINI)
    end
    if strip(EXTRASYSINI)<>'' then do
        EXTRASYSINIRCFILES=EXTRASYSINIRCFILES+1
        EXTRASYSINIRCFILE.EXTRASYSINIRCFILES=BOOTABLETEMP'\sysinirc.xtr'
        EXTRASYSINIRCFILEUPDATE.EXTRASYSINIRCFILES='YES'
        rc=createFile(EXTRASYSINIRCFILE.EXTRASYSINIRCFILES, EXTRASYSINI)
    end

    /* Create the ini files based on the templates */
    do IRC=1 to EXTRAUSERINIRCFILES
        rc=parseINIRC(ifExists(EXTRAUSERINIRCFILE.IRC, EXTRAUSERINIRCFILE.IRC), BOOTABLETEMP'\inirc.'||IRC, BOOTABLETEMP'\os2.ini', EXTRAUSERINIRCFILEUPDATE.IRC)
    end

    do IRC=1 to EXTRASYSINIRCFILES
        rc=parseINIRC(ifExists(EXTRASYSINIRCFILE.IRC, EXTRASYSINIRCFILE.IRC), BOOTABLETEMP'\sysinirc.'||IRC, BOOTABLETEMP'\os2sys.ini', EXTRASYSINIRCFILEUPDATE.IRC)
    end

    /* Move the ini files to destination */
    rc=sCopyFile(BOOTABLETEMP'\os2.ini', DESTDRIVE'\OS2')
    rc=sCopyFile(BOOTABLETEMP'\os2sys.ini', DESTDRIVE'\OS2')

    return 0


/* ------------------------------------------------------------------------ */
/* Create Extra INI files                                                   */
/* ARG: None                                                                */
/* RET: 0                                                                   */
/* Note: Destination is determined by the DESTINATION and DEST variables    */
/*   CD=CD/DVD or Maintenance Partition                                     */
/*   IMG=Boot Image (of CD/DVD) or Maintenance Partition                    */
/* ------------------------------------------------------------------------ */
createExtraINIFiles:
    do IRC=1 to EXTRAINIFILES
        if DEST=EXTRAINIFILEDESTINATION.IRC then do
            if EXTRAINIFILECONTENTINI.IRC<>'' then do
                rc=sCopyFile(ifExists(EXTRAINIFILECONTENTINI.IRC,EXTRAINIFILECONTENTINI.IRC),BOOTABLETEMP'\'EXTRAINIFILENAME.IRC)
            end
            if EXTRAINIFILECONTENTRC.IRC<>'' then
                rc=parseINIRC(ifExists(EXTRAINIFILECONTENTRC.IRC,EXTRAINIFILECONTENTRC.IRC), BOOTABLETEMP'\xtrinirc.a'IRC, BOOTABLETEMP'\'EXTRAINIFILENAME.IRC, 'YES')
            if EXTRAINIFILECONTENT.IRC<>'' then do
                rc=createFile(BOOTABLETEMP'\extraini.b'IRC,EXTRAINIFILECONTENT.IRC)
                rc=parseINIRC(BOOTABLETEMP'\extraini.b'IRC,BOOTABLETEMP'\xtrinirc.c'IRC,BOOTABLETEMP'\'EXTRAINIFILENAME.IRC, 'YES')
            end
        end
    end

    do IRC=1 to EXTRAINIFILES
        if DEST=EXTRAINIFILEDESTINATION.IRC then do
            rc=createDir(DESTDRIVE||EXTRAINIFILEDIRECTORY.IRC)
            rc=sCopyFile(BOOTABLETEMP'\'EXTRAINIFILENAME.IRC, DESTDRIVE||EXTRAINIFILEDIRECTORY.IRC)
        end
    end

    return 0



/* ------------------------------------------------------------------------ */
/* Parse INI file template and create INI file                              */
/* ARG1: Source RC file                                                     */
/* ARG2: Destination RC file                                                */
/* ARG3: Destination INI file                                               */
/* ARG4: Update?                                                            */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
parseINIRC:
    SOURCERC=ARG(1)
    DESTINATIONRC=ARG(2)
    DESTINATIONINI=ARG(3)
    UPDATEINI=ARG(4)

    call progress 2, 'Parsing INI file template' SOURCERC ifSet(UPDATEINI,'(updating)')

    PREINITXT='CODEPAGE $$CODEPAGE$$'||CRLF||CRLF||,
            'STRINGTABLE'||CRLF||,
            'BEGIN'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||CRLF
    POSTINITXT=CRLF||'END'||CRLF

    drop INIXAPPLICATIONS.
    drop INIXKEYS.
    drop INIXVALUES.
    NRINIX=0

    drop INIAPPLICATIONS.
    drop INIKEYS.
    drop INIVALUES.
    NRINI=0

    if lines(SOURCERC)=0 then
        call error SOURCERC 'is empty or does not exist'

    if \isSet(UPDATEINI) then
        rc=createFile(DESTINATIONRC, PREINITXT)

    rc=0
    RCLINENR=0
    do while lines(SOURCERC)>0
        RCLINENR=RCLINENR+1
        ORGRCLINE=linein(SOURCERC)
        RCLINE=ORGRCLINE
        call progress 8, SOURCERC '('RCLINENR'):' RCLINE
        if \(left(strip(RCLINE),2)='/*' | left(strip(RCLINE),1)=';') then do
            RCLINE=parseText(RCLINE)

           if strip(RCLINE)<>'' then do
                parse value RCLINE with '"'RCAPPL'"' '"'RCKEY'"' RCVALUE
                RCVALUE=strip(RCVALUE)
                if RCVALUE='""' then RCVALUE='0x00' /* Convert empty string to hex 0x00 */
                RCLINEOK=(left(RCVALUE,1)='"' & right(RCVALUE,1)='"')
                RCLINEOK=RCLINEOK | (translate(left(RCVALUE,2))='0X' & length(RCVALUE)>2 & strip(translate(RCVALUE,'XX','xX 01234567890abcdefABCDEF',' '))='X')
                RCLINEOK=RCLINEOK & RCAPPL<>'' & RCKEY<>''
                if \RCLINEOK then
                    call error 'Syntax error in ini file template' SOURCERC 'Line:' RCLINENR||CRLF||'==>'||ORGRCLINE

                if translate(left(RCVALUE,2))='0X' then do
                    NRINIX=NRINIX+1
                    INIXAPPLICATIONS.NRINIX=RCAPPL
                    INIXKEYS.NRINIX=RCKEY
                    INIXVALUES.NRINIX=substr(RCVALUE,3)
                end
                else
                    if isSet(UPDATEINI) then do
                        NRINI=NRINI+1
                        INIAPPLICATIONS.NRINI=RCAPPL
                        INIKEYS.NRINI=RCKEY
                        INIVALUES.NRINI=strip(RCVALUE,,'"')||x2c('0')
                    end
                    else do
                        RCAPPL='"'RCAPPL'"'
                        RCKEY='"'RCKEY'"'
                        if length(RCAPPL)<31 then RCAPPL=left(RCAPPL,31)
                        if length(RCKEY)<31  then RCKEY=left(RCKEY,31)
                        RCLINE='    'RCAPPL' 'RCKEY' 'RCVALUE
                        rc=lineout(DESTINATIONRC, RCLINE)
                    end
           end
        end
    end

    /* Close files */
    if \isSet(UPDATEINI) then do
        rc=lineout(DESTINATIONRC, POSTINITXT)
        rc=lineout(DESTINATIONRC)
        if rc<>0 then
            call error 'Failed to close file' DESTINATIONRC 'rc('rc')'
    end
    rc=stream(SOURCERC, 'C', 'CLOSE')
    if rc<>'READY:' & rc<>'' then
        call error 'Failed to close file' SOURCERC 'rc('rc')'

    /* Make ini */
    if \isSet(UPDATEINI) then do
        rc=runCMD('makeini' DESTINATIONINI DESTINATIONRC, 'Failed to create ini file using template' SOURCERC)
    end

    /* Set hex values */
    do I=1 to NRINIX
        /* Only write X value if in update mode, or not already existing */
        if isSet(UPDATEINI) | sysIni(DESTINATIONINI, INIXAPPLICATIONS.I, INIXKEYS.I)='ERROR:' then do
            rc=sysIni(DESTINATIONINI, INIXAPPLICATIONS.I, INIXKEYS.I, X2C(INIXVALUES.I))
            if rc<>'' then
                call error 'Failed to add hex value' INIXAPPLICATIONS.I||'('||INIXKEYS.I||')=0x'||INIXVALUES.I 'from template' SOURCERC 'to ini file' DESTINATIONINI', RC='RC
        end
    end

    /* Update values */
    do I=1 to NRINI
        rc=sysIni(DESTINATIONINI, INIAPPLICATIONS.I, INIKEYS.I, INIVALUES.I)
        if rc<>'' then
            call error 'Failed to add/update value' INIAPPLICATIONS.I||'('||INIKEYS.I||')='||INIVALUES.I 'from template' SOURCERC 'to ini file' DESTINATIONINI', RC='RC
    end

    return 0


/* ------------------------------------------------------------------------ */
/* Create IBMCOM networking files                                           */
/* ARG: None                                                                */
/* RET: 0                                                                   */
/* Note: Destination is determined by the DESTINATION and DEST variables    */
/*   CD=CD/DVD or Maintenance Partition                                     */
/*   IMG=Boot Image (of CD/DVD) or Maintenance Partition                    */
/* ------------------------------------------------------------------------ */
createIBMCOMNetworkFiles:
    call progress 1, 'Creating IBMCOM network files on' destinationName()

    /* protocol.ini */
    if isSet(TCPIPLAN) | isSet(NETBIOS) then do
        NAME=translate(fileName(NETWORKCARDFILE))
        IDX=pos('.',NAME)
        if IDX>0 then NAME=left(NAME,IDX-1)
        select
            when NAME='FASTNIC' then NAME='FSTNIC'
            when NAME='GENM32W' then NAME='WRND32'
            otherwise ;
        end

        /* [PROT_MAN] and [IBMLXCFG] */
        TXT=''||,
          '[PROT_MAN]'||CRLF||CRLF||,
          '  DRIVERNAME = PROTMAN$'||CRLF||CRLF||,
          '[IBMLXCFG]'||CRLF||CRLF||,
          ifSet(NETBIOS,     '  netbeui_nif = netbeui.nif'||CRLF)||,
          ifSet(NETBIOSTCPIP,'  tcpbeui_nif = tcpbeui.nif'||CRLF)||,
          ifSet(TCPIP,       '  tcpip_nif = tcpip.nif'||CRLF)||,
          '  NETWORK_nif = NETWORK.nif'||CRLF

        /* [NETBIOS] */
        if isSet(NETBIOS) | isSet(NETBIOSTCPIP) then
            TXT=TXT||CRLF||,
              '[NETBIOS]'||CRLF||CRLF||,
              '  DriverName = netbios$'||CRLF
        ADAPTER=0
        if isSet(NETBIOS) then do
            TXT=TXT||'  ADAPTER'ADAPTER' = netbeui$,'ADAPTER||CRLF
            ADAPTER=ADAPTER+1
        end
        if isSet(NETBIOSTCPIP) then
            TXT=TXT||'  ADAPTER'ADAPTER' = tcpbeui$,'ADAPTER||CRLF

        /* [netbeui_nif] */
        if isSet(NETBIOS) then
            TXT=TXT||CRLF||,
              '[netbeui_nif]'||CRLF||CRLF||,
              '  DriverName = netbeui$'||CRLF||,
              '  Bindings = NETWORK_nif'||CRLF

        /* [tcpbeui_nif] */
        if isSet(NETBIOSTCPIP) then
            TXT=TXT||CRLF||,
              '[tcpbeui_nif]'||CRLF||,
              ''||CRLF||,
              '   DriverName = tcpbeui$'||CRLF||,
              '   Bindings = 'ifSet(NETBIOS,',')'NETWORK_nif'||CRLF||,
              '   NODETYPE = "B-Node"'||CRLF

        /* [TCPIP_NIF] */
        if isSet(TCPIP) then
            TXT=TXT||CRLF||,
              '[tcpip_nif]'||CRLF||CRLF||,
              '  DriverName = TCPIP$'||CRLF||,
              '  Bindings = NETWORK_nif'||CRLF

        /* [NETWORK_nif] */
        TXT=TXT||CRLF||,
          '[NETWORK_nif]'||CRLF||CRLF||,
          '  DriverName = 'NAME'$'||CRLF

        /* copy GENMAC lines from the original protocol.ini */
        if isSet(GENMAC) then do
            ORGPROTOCOLINI=fileExists(pathName(fileExists('protman.os2'))||'\protocol\protocol.ini')
            if ORGPROTOCOLINI='' then call error 'Unable to locate original protocol.ini'
            MATCHED=0
            DONE=0
            do while \DONE & lines(ORGPROTOCOLINI)>0
                ORGLINE=space(lineIn(ORGPROTOCOLINI))
                if MATCHED>0 | pos('DriverName = 'GENMACDRIVER,ORGLINE)>0 then
                    MATCHED=MATCHED+1
                if MATCHED>1 then
                    if strip(ORGLINE)='' then
                        DONE=1
                    else
                        TXT=TXT||ORGLINE||CRLF
            end
            if MATCHED<=1 then call error 'Unable to locate properly formatted GENMAC section for 'GENMACDRIVER' in original protocol.ini'
            rc=stream(ORGPROTOCOLINI, 'C', 'CLOSE')
        end
        rc=createFile(DESTDRIVE'\ibmcom\protocol.ini',TXT)
    end

    return 0


/* ------------------------------------------------------------------------ */
/* Create TCPIP networking files                                            */
/* ARG: None                                                                */
/* RET: 0                                                                   */
/* Note: Destination is determined by the DESTINATION and DEST variables    */
/*   CD=CD/DVD or Maintenance Partition                                     */
/*   IMG=Boot Image (of CD/DVD) or Maintenance Partition                    */
/* ------------------------------------------------------------------------ */
createTCPIPNetworkFiles:
    call progress 1, 'Creating TCPIP network files on' destinationName()

    /* mptstart.cmd */
    TXT=''||,
      'inetwait'||ifSet(TCPIPLAN,,' 1')||CRLF||,
      'if errorlevel 1 goto end'||CRLF||,
      'call 'destinationDrive(DEST)'\mptn\bin\setup.cmd'||CRLF||,
      ':end'||CRLF
    rc=createFile(DESTDRIVE'\mptn\bin\mptstart.cmd',TXT)
    /* setup.cmd */
    TXT=''||,
      'route -fh'||CRLF||,
      'arp -f'||CRLF||,
      'ifconfig lo 127.0.0.1'||CRLF
    if isSet(TCPIPLAN) then do
        if isSet(TCPIPDHCP) then
            TXT=TXT||'dhcpstrt -i lan0 -d 'TCPIPDHCPTIMEOUT||CRLF
        else
            TXT=TXT||'ifconfig lan0' TCPIPHOSTADDRESS||CRLF
        if  strip(TCPIPGATEWAYADDRESS)<>'' then
            TXT=TXT||'route add default' TCPIPGATEWAYADDRESS||CRLF
    end
    rc=createFile(DESTDRIVE'\mptn\bin\setup.cmd',TXT)

    /* ETC Directory */
    rc=createDir(DESTDRIVE'\MPTN\ETC')

    /* dhcpcd.cfg */
    if isSet(TCPIPDHCP) then do
        TXT=''||,
          'clientid   MAC'||CRLF||,
          'option 1                     # Subnet Mask'||CRLF||,
          'option 3                     # Router'||CRLF||,
          'option 6                     # Domain Name Server'||CRLF||,
          ifSet(TCPIPDHCPUPDATEHOST,'option 12 "'||TCPIPHOSTNAME||'" # Host Name'||CRLF)||,
          'option 15                    # Domain Name'||CRLF||,
          'option 28                    # Broadcast Address'||CRLF||,
          'option 33                    # Static Routes'||CRLF||,
          'option 60 "IBMWARP_V4.1"     # Vendor Class'||CRLF||,
          'option 77 "IBMWARP_V4.1"     # User Class'||CRLF
        rc=createFile(DESTDRIVE'\mptn\etc\dhcpcd.cfg',TXT)
    end

    /* protocol */
    TXT=''||,
      'icmp  1 ICMP'||CRLF||,
      'ip    4 IP'||CRLF||,
      'tcp   6 TCP'||CRLF||,
      'udp  17 UDP'||CRLF
    rc=createFile(DESTDRIVE'\mptn\etc\protocol',TXT)

    /* services */
    TXT=''||,
      'ftp-data         20/tcp'||CRLF||,
      'ftp-data         20/udp'||CRLF||,
      'ftp              21/tcp'||CRLF||,
      'ftp              21/udp'||CRLF||,
      'telnet           23/tcp'||CRLF||,
      'telnet           23/udp'||CRLF||,
      'name             42/tcp'||CRLF||,
      'name             42/udp'||CRLF||,
      'nameserver       42/tcp'||CRLF||,
      'nameserver       42/udp'||CRLF||,
      'domain           53/tcp'||CRLF||,
      'domain           53/udp'||CRLF||,
      'tftp             69/tcp'||CRLF||,
      'tftp             69/udp'||CRLF||,
      'finger           79/tcp'||CRLF||,
      'finger           79/udp'||CRLF||,
      'hostname        101/tcp'||CRLF||,
      'hostname        101/udp'||CRLF||,
      'exec            512/tcp'||CRLF||,
      'login           513/tcp'||CRLF||,
      'shell           514/tcp'||CRLF
    rc=createFile(DESTDRIVE'\mptn\etc\services',TXT)

    /* hosts */
    TXT=''||,
      '127.0.0.1 localhost'||CRLF||,
      ifClause(TCPIPHOSTADDRESS<>'' & isSet(TCPIPLAN) & \isSet(TCPIPDHCP), TCPIPHOSTADDRESS TCPIPHOSTNAME'.'TCPIPDOMAINNAME||CRLF)||,
      TCPIPHOSTS
    rc=createFile(DESTDRIVE'\mptn\etc\hosts',TXT)

    /* resolv2 */
    if TCPIPDNSADDRESS<>'' then do
        TXT=''||,
          'domain' TCPIPDOMAINNAME||CRLF||,
          'nameserver' TCPIPDNSADDRESS||CRLF
        rc=createFile(DESTDRIVE'\mptn\etc\resolv2',TXT)
    end

    return 0


/* ------------------------------------------------------------------------ */
/* Create PEER networking files                                             */
/* ARG: None                                                                */
/* RET: 0                                                                   */
/* Note: Destination is determined by the DESTINATION and DEST variables    */
/*   CD=CD/DVD or Maintenance Partition                                     */
/*   IMG=Boot Image (of CD/DVD) or Maintenance Partition                    */
/* ------------------------------------------------------------------------ */
createPeerNetworkFiles:
    call progress 1, 'Creating PEER network files on' destinationName()

    /* ibmlan.ini */
    rc=createDir(DESTDRIVE'\IBMLAN')
    TXT=''||,
        '[networks]'||CRLF
    if isSet(NETBIOS) then
        TXT=TXT||'  net1 = NETBEUI$,0,LM10,34,100,14'||CRLF||,
            ifSet(NETBIOSTCPIP, '  net2 = TCPBEUI$,1,LM10,34,100,14'||CRLF)
    else
        TXT=TXT||'  net1 = TCPBEUI$,0,LM10,34,100,14'||CRLF
    TXT=TXT||CRLF||,
        '[requester]'||CRLF||,
        '  charcount = 16'||CRLF||,
        '  chartime = 250'||CRLF||,
        '  charwait = 3600'||CRLF||,
        '  keepconn = 600'||CRLF||,
        '  keepsearch = 600'||CRLF||,
        '  maxcmds = 16'||CRLF||,
        '  maxerrorlog = 100'||CRLF||,
        '  maxthreads = 10'||CRLF||,
        '  maxwrkcache = 64'||CRLF||,
        '  numalerts = 12'||CRLF||,
        '  numcharbuf = 10'||CRLF||,
        '  numservices = 7'||CRLF||,
        '  numworkbuf = 40'||CRLF||,
        '  numdgrambuf = 14'||CRLF||,
        '  othdomains = '||CRLF||,
        '  printbuftime = 90'||CRLF||,
        '  sesstimeout = 45'||CRLF||,
        '  sizcharbuf = 512'||CRLF||,
        '  sizerror = 1024'||CRLF||,
        '  sizworkbuf = 4096'||CRLF||,
        '  wrkheuristics = 1111111121311111110001011120111221001111100'||CRLF||,
        '  wrkservices = MESSENGER'||CRLF||,
        '  wrknets = NET1'||ifClause(isSet(NETBIOS)&isSet(NETBIOSTCPIP),',NET2')||CRLF||,
        '  Computername = 'PEERCOMPUTERNAME||CRLF||,
        '  Domain = 'PEERDOMAIN||CRLF||,
        CRLF||,
        '[messenger]'||CRLF||,
        '  logfile = messages.log'||CRLF||,
        '  sizmessbuf = 4096'||CRLF||,
        CRLF||,
        '[peer]'||CRLF||,
        '  auditing = resource'||CRLF||,
        '  guestacct = guest'||CRLF||,
        '  autodisconnect = -1'||CRLF||,
        '  maxauditlog = 100'||CRLF||,
        '  maxchdevjob = 2'||CRLF||,
        '  maxchdevs = 1'||CRLF||,
        '  maxconnections = 26'||CRLF||,
        '  maxlocks = 64'||CRLF||,
        '  maxopens = 160'||CRLF||,
        '  maxsearches = 150'||CRLF||,
        '  maxsessopens = 80'||CRLF||,
        '  maxsessreqs = 25'||CRLF||,
        '  maxsessvcs = 1'||CRLF||,
        '  maxshares = 16'||CRLF||,
        '  maxusers = 5'||CRLF||,
        '  numbigbuf = 4'||CRLF||,
        '  numfiletasks = 1'||CRLF||,
        '  numreqbuf = 48'||CRLF||,
        '  sizreqbuf = 4096'||CRLF||,
        '  srvhidden = no'||CRLF||,
        '  srvheuristics = 111101411113110013311'||CRLF||,
        '  SRVSERVICES = '||CRLF||,
        '  srvnets = NET1'||ifClause(isSet(NETBIOS)&isSet(NETBIOSTCPIP),',NET2')||CRLF||,
        '  SRVCOMMENT = 'PEERDESCRIPTION||CRLF||,
        CRLF||,
        '[replicator]'||CRLF||,
        '  replicate = IMPORT'||CRLF||,
        '  importpath = 'DESTDRIVE'\ibmlan\repl\import'||CRLF||,
        '  tryuser = yes'||CRLF||,
        '  password = '||CRLF||,
        CRLF||,
        '[services]'||CRLF||,
        '  messenger = services\msrvinit.exe'||CRLF||,
        '  peer = services\peerinit.exe'||CRLF||,
        '  replicator = services\replicat.exe'||CRLF||,
        '  requester = services\wksta.exe'||CRLF
    rc=createFile(DESTDRIVE'\ibmlan\ibmlan.ini',TXT)

    /* net.acc */
    INFILE=fileExists(PEERNETACC)
    if INFILE='' then call error 'NET.ACC File:' PEERNETACC 'not found'
    FILELEN=stream(INFILE,'C','QUERY SIZE')
    OUTFILE=DESTDRIVE'\IBMLAN\ACCOUNTS\NET.ACC'
    rc=createDir(DESTDRIVE'\IBMLAN\ACCOUNTS')
    TXT='/**/'||CRLF||'exit charout("'OUTFILE'",charin(,,'FILELEN'))' /* external command to copy net.acc file */
    rc=createFile(BOOTABLETEMP'\cpnetacc.cmd',TXT)
    rc=runCMD('call' BOOTABLETEMP'\cpnetacc.cmd < "'INFILE'"', 'Error copying NET.ACC file' INFILE)

    return 0

/* ------------------------------------------------------------------------ */
/* Create files on CD/DVD / maintenance partition                           */
/* ARG1: Destination                                                        */
/*         CD=CD/DVD or Maintenance Partition                               */
/*         IMG=Boot Image (of CD/DVD) or Maintenance Partition              */
/* RET:  0                                                                  */
/* ------------------------------------------------------------------------ */
createFiles:
    DEST=ARG(1)
    IMGNR=2

    if isSet(NETWORKING) & DEST=IBMCOMDESTINATION     then rc=createIBMCOMNetworkFiles()
    if isSet(TCPIP)      & DEST=NETWORKINGDESTINATION then rc=createTCPIPNetworkFiles()
    if isSet(PEER)       & DEST='IMG'                 then rc=createPeerNetworkFiles()
    if isSet(WPS)        & DEST=WPSDESTINATION        then rc=createINIFiles()

    rc=createExtraINIFiles()

    if isSet(WPS) & isSet(LVM) & DEST=WPSDESTINATION then do
        call progress 1, 'Creating lvm icons and command-files on' destinationName()
        TXT='lvm.exe /RediscoverPRM'
        rc=createFile(DESTDRIVE'\OS2\lvmrfrsh.cmd',TXT)
        rc=sCopyFile('bootAble.lvmrefresh.ico',DESTDRIVE'\OS2\lvmrfrsh.ico')
        TXT='lvm.exe'
        rc=createFile(DESTDRIVE'\OS2\lvmgui.cmd',TXT)
        rc=sCopyFile('bootAble.lvmgui.ico',DESTDRIVE'\OS2\lvmgui.ico')
    end

    FIRSTCREATED=0
    do I=1 to EXTRACREATEFILES
        if DEST=EXTRACREATEFILEDESTINATION.I then do
            if \FIRSTCREATED then do
                FIRSTCREATED=1
                call progress 1, 'Creating extra files on' destinationName()
            end
            rc=createDir(DESTDRIVE||EXTRACREATEFILEDIRECTORY.I)
            rc=createFile(DESTDRIVE||EXTRACREATEFILEDIRECTORY.I||'\'||EXTRACREATEFILENAME.I,parseText(EXTRACREATEFILECONTENT.I))
        end
    end

    if DEST='CD' & isSet(CREATECLONECD) then do
        rc=createDir(DESTDRIVE||CREATECLONECDDIRECTORY)
        FILE=DESTDRIVE||CREATECLONECDDIRECTORY||'\cloneCD.cmd'
        TXT='@bootAble /v 'CREATECLONECDVERBOSE CREATECLONECDCMDLINEARGS 'cloneCD.cfg'
        rc=createFile(FILE,TXT)
        FILE=DESTDRIVE||CREATECLONECDDIRECTORY||'\cloneCD.cfg'
        TXT="; Additional cloneCD settings ------------------------------------------------"||CRLF||CRLF||,
            "SYSINSTX='"fileName(SYSINSTX)"'"||CRLF||,
            "CREATEBOOTCD='NO'"||CRLF||,
            "CREATECLONECD='NO'"||CRLF||,
            "WRITEASKEDTOFILE=''"||CRLF||,
            "WRITEASKEDTOFILENAME=''"||CRLF||,
            "PEERIBMLANROOT="ifClause(NETWORKINGDESTINATION='CD',"'"CDROMDRIVE"'","OSBOOTDRIVE")"||'\IBMLAN'"||CRLF||,
            "NETWORKCARDFILE='"destinationDrive(IBMCOMDESTINATION)"\ibmcom\macs\"fileName(NETWORKCARDFILE)"'"||CRLF||,
            "TCPIPDHCP='"TCPIPDHCP"'"||CRLF||,
            "TCPIPDHCPUPDATEHOST='"TCPIPDHCPUPDATEHOST"'"||CRLF||,
            "RAMDISK='"CREATECLONECDRAMDISK"'"||CRLF||,
            "WRITERDRIVE='"CREATECLONECDMAINTENANCEDRIVE"'"||CRLF||,
            "REMOVEROFLAG='"CREATECLONECDREMOVEROFLAG"'"||CRLF||,
            "FORMATMAINTENANCEJFS='NO'"||CRLF||,
            "VERBOSE="CREATECLONECDVERBOSE||CRLF||,
            "ADDTOSEARCHPATH='"ifClause(WPSFONTSDESTINATION<>'IMG',CDROMDRIVE"\PSFONTS;"CDROMDRIVE"\PSFONTS\PFM;")"'"||CRLF||,
            "ADDTOSEARCHPATH='"ifClause(isSet(VBOXADDITIONS),CDROMDRIVE"\OS2Additions;")"'"||CRLF||,
            "ADDTOSEARCHPATH='"ifClause(isSet(AIRBOOT),CDROMDRIVE||AIRBOOTDIRECTORY)"'"||CRLF||,
            "SNAPSEDISTRIBUTIONFILE=fileExists('"fileName(SNAPSEDISTRIBUTIONFILE)"')"
        rc=writeAskedLinesToFile(FILE,TXT)
    end

    return 0

/*-----------------------------------------------------------------------------+
|                                                                              |
|  Default error handlers:            Added by D. Bissett Oct. 17 2010         |
|                                                                              |
+-----------------------------------------------------------------------------*/

SYNTAX:
  SAY 'SYNTAX ERROR:' errortext( rc ) 'in:'
  SAY 'Line='sigl
  SAY sourceline( sigl )
  SAY 'Tap any key to exit'
  x=sysGetKey()
  EXIT

HALT:
  SAY 'HALT occurred in:'
  SAY 'Line='sigl
  SAY sourceline( sigl )
  SAY 'Tap any key to exit'
  x=sysGetKey()
  EXIT
