/*Ŀ*/
/*                                                                          */
/* PROGRAM NAME: PMSPY                                                      */
/* -------------                                                            */
/*  A PM program that is used to look at or 'spy' on the message queue of   */
/*  other PM applications windows.                                          */
/*                                                                          */
/* COPYRIGHT:                                                               */
/* ----------                                                               */
/*  Copyright (C) International Business Machines Corp., 1992               */
/*                                                                          */
/* DISCLAIMER OF WARRANTIES:                                                */
/* -------------------------                                                */
/*  The following [enclosed] code is sample code created by IBM Corporation.*/
/*  This sample code is not part of any standard IBM product and is provided*/
/*  to you solely for the purpose of assisting you in the development of    */
/*  your applications.  The code is provided "AS IS", without warranty of   */
/*  any kind.  IBM shall not be liable for any damages arising out of your  */
/*  use of the sample code, even if they have been advised of the           */
/*  possibility of such damages.                                            */
/*                                                                          */
/* For details on what this program does etc., please see the PMSPY.C file. */
/*                                                                          */
/**/

/*Ŀ*/
/* PMSPYHK.C                                                                */
/*                                                                          */
/* Program to spy on other windows message queues                           */
/*                                                                          */
/*                                                                          */
/* Notes:                                                                   */
/*                                                                          */
/* 1. This code MUST be in a DLL since we "hook" into the system queue!     */
/*                                                                          */
/* 2. There are TWO types of routines in this module:                       */
/*                                                                          */
/*    a) Routines called directly by PM...these are the "hook" routines     */
/*                                                                          */
/*       SpyInputHookProc                                                   */
/*       SpySendMsgHookProc                                                 */
/*                                                                          */
/*    b) Routines called only by PMSPY instances                            */
/*                                                                          */
/*       SpyRegister                                                        */
/*       SpyDeRegister                                                      */
/*                                                                          */
/*       SpySetTarget                                                       */
/*       SpyUnSetTarget                                                     */
/*       SpyQueryTargetWindow                                               */
/*       SpyQueryTargetQueue                                                */
/*                                                                          */
/*       SpySetTargetIsWindow                                               */
/*       SpyQueryTargetIsWindow                                             */
/*                                                                          */
/*       SpySetTargetMsgRange                                               */
/*                                                                          */
/*       SpyDllVersion                                                      */
/*       SpyQueryDllRegisterMax                                             */
/*                                                                          */
/**/

/*Ŀ*/
/* Includes                                                                 */
/**/
#define INCL_BASE
#define INCL_GPI
#define INCL_WIN

#include <os2.h>                        /* the PM header file           */

#include <stdio.h>
#include "pmspyhk.h"

/*Ŀ*/
/* Local data structures, macros, etc                                       */
/**/
typedef struct
{
  /*Ŀ*/
  /* Control Flag: TRUE --> this item is actively SPYing                    */
  /*               FALSE--> this item marks end of the variable length list */
  /**/
  BOOL      fSpyingActive;

  /*Ŀ*/
  /* Define who we're SPYing on                                             */
  /**/
  HWND      hwndTarget;                 /* which Window */
  HMQ        hmqTarget;                 /* which Queue */

  BOOL      fTargetIsWindow;            /* SPYing on Window (vs. Queue) */

  /*Ŀ*/
  /* Additional control data                                                */
  /**/
  HWND      hwndSpy;                    /* destination SPY window */
  MSG        MsgSpy;                    /* PM message to send with SPY data */

  MSG       msgLow,                     /* MSG range to pass on to SPYee */
            msgHigh;

  /*Ŀ*/
  /* QMSG data for passing onto PMSPY instances                             */
  /*                                                                        */
  /* - 'normalized' data passed regardless of "hook" source                 */
  /* - NOTE: we can do this because each message is really 'single-threaded'*/
  /*         through this DLL because WinSendMsg is used to:                */
  /*         (a) invoke the "hook" routines and                             */
  /*         (b) invoke PMSPY to 'spy' the MSG                              */
  /**/
  QMSG      qMsg;

  /*Ŀ*/
  /* Define who we're SPYing on                                             */
  /*                                                                        */
  /* - used to not "spy" on PMSPY agents which causes PM recursion          */
  /**/
  HMQ        hmqSpy;                    /* PM queue of destination SPY window */

}   dllSPY,
  *PdllSPY;

/*Ŀ*/
/* Macros to generate initialized Spys[] data elements                      */
/**/
#define spyEOT {BOOL_FALSE,                            \
                SPY_BAD_HWND,SPY_BAD_HMQ,BOOL_FALSE,   \
                SPY_BAD_HWND,WM_NULL,                  \
                0,0xFFFF}

#define spyDEF {BOOL_TRUE,                             \
                SPY_BAD_HWND,SPY_BAD_HMQ,BOOL_FALSE,   \
                SPY_BAD_HWND,WM_NULL,                  \
                0,0xFFFF}

#define ID_NOT_DEFINED   0xFFFF

/*Ŀ*/
/* Macro to validate a specified SPY instance ID                            */
/*                                                                          */
/* - if the ID is not valid, a RETURN is generated using the specified value*/
/* - if the ID is     valid, set pointer to ID's data                       */
/*                                                                          */
/* - Tests for Validity:                                                    */
/*   a) ID is less than current 1-origin count of PMPSY instances           */
/*   b) ID is defined in Id2Spys[]                                          */
/**/
#define IF_BAD_ID_RETURN_ELSE_SET(idSpy,retValue,pSpy)               \
if ( (idSpy < uSpyInstances) && (Id2Spys[idSpy] != ID_NOT_DEFINED) ) \
  pSpy = &Spys[ Id2Spys[idSpy] ];                                    \
else                                                                 \
  return(retValue);

/*Ŀ*/
/* Localized DLL data for controlling using PMSPY instances                 */
/**/
static USHORT  uSpyInstances = 0;             /* # active SPYee(s) */
static HMODULE hmodDLL       = 0;             /* 'handle' of PMSPYHK.DLL */

/*Ŀ*/
/* Translate a "PMSPY instance ID" (0..n) to it's supporting data           */
/**/
static USHORT Id2Spys[] =
{
  ID_NOT_DEFINED,       /* Instance #1 data */
  ID_NOT_DEFINED,       /* Instance #2 data */
  ID_NOT_DEFINED,       /* Instance #3 data */
  ID_NOT_DEFINED,       /* Instance #4 data */
  ID_NOT_DEFINED,       /* Instance #5 data */
  ID_NOT_DEFINED,       /* Instance #6 data */
  ID_NOT_DEFINED,       /* Instance #7 data */
  ID_NOT_DEFINED        /* Instance #8 data */
};

#define MAX_SPYEES    ( sizeof(Id2Spys) / sizeof(Id2Spys[0]) )

/*Ŀ*/
/* Supporting SPY data                                                      */
/*                                                                          */
/* Implementation Notes:                                                    */
/* ---------------------                                                    */
/* 1) because this table is searched sequentially each time one of our      */
/*    "hook" procedures is called to handle a MSG, we want to only search   */
/*    as many instances as we have currently active SPYees.  So, we keep    */
/*    table 'packed' as tight as possible.  By that, i mean that any time   */
/*    a PMSPY instance DeRegisters, we move all data elements below it      */
/*    up one slot.  Within a "hook" procedure we only 'look' at the first   */
/*    N items (where N is the number of active SPYees)                      */
/*                                                                          */
/* 2) to maintain the SPYEE value returned when a PMPSY instance Registers, */
/*    we also maintain Id2Spys[].                                           */
/**/
static dllSPY Spys[] =
{
  spyEOT,               /* room for Instance #1 */
  spyEOT,               /* room for Instance #2 */
  spyEOT,               /* room for Instance #3 */
  spyEOT,               /* room for Instance #4 */
  spyEOT,               /* room for Instance #5 */
  spyEOT,               /* room for Instance #6 */
  spyEOT,               /* room for Instance #7 */
  spyEOT,               /* room for Instance #8 */

  spyEOT                /* permanent End-Of-Table marker */
};

/*Ŀ*/
/* Initialized dllSPY items for adding/deleting Spys[] items                */
/**/
static const dllSPY defSPY = spyDEF;
static const dllSPY defEOT = spyEOT;

/*Ŀ*/
/* Input Hook procedure                                                     */
/*                                                                          */
/* This procedure gets calls after a MSG is removed from an application's   */
/* queue before being returned by a WinGetMsg or WinPeekMsg                 */
/*                                                                          */
/* Notes:                                                                   */
/* ------                                                                   */
/* 1) we are called by PM, not a PMSPY instance.  This implies that we      */
/*    don't know which PMSPY instance, if any, should SPY this message.     */
/*    That's why we have to scan the complete Spys[] table and route it     */
/*    to all interested SPYees...                                           */
/* 2) we always return FALSE so that any other hooks "downstream"           */
/*    can also have a chance at this message                                */
/*                                                                          */
/* Reference: PM Programming Reference: Volume Two,                         */
/*            Chapter 10 - "Functions Supplied by Application"              */
/**/

#pragma linkage (SpyInputHookProc, system)

BOOL SpyInputHookProc(HAB   habSpy_NotUsed,
                      PQMSG pQmsg,
                      BOOL  bRemove_NotUsed)
{
  register PdllSPY pSpy;

  /*Ŀ*/
  /* Check each of the 'active' SPYees to see if interested in this message */
  /**/
  for(/* Initialize */ pSpy = &Spys[0];      /* Start @ first Spyee */
      /* While      */ pSpy->fSpyingActive;  /* Not at End-Of-List */
      /* Iterate    */ pSpy++)               /* Move to next Spyee */
  {
    /*Ŀ*/
    /* Is this MSG within the active range?                                 */
    /**/
    if ( (pQmsg->msg < pSpy->msgLow) || (pQmsg->msg > pSpy->msgHigh) )
      continue;

    /*Ŀ*/
    /* Is this MSG destined for the correct Window?                         */
    /**/
    if ( pSpy->fTargetIsWindow )               /* SPYing a specific window?   */
    {
      if (pSpy->hwndTarget != pQmsg->hwnd)     /* correct window?             */
        continue;                              /* - ignore if not             */
    }
    /*Ŀ*/
    /* Is this MSG destined for the correct Queue?                          */
    /*                                                                      */
    /* - because we now can "hook" the PM system queue, we must ensure      */
    /*   that this isn't one of our own WinSendMsg() calls that is passing  */
    /*   "hooked" data to the registered SPY "catcher" window.  Failure     */
    /*   to test for this caused PM recursion that eventually overflowed    */
    /*   the stack (trust me, it happened!)                                 */
    /**/
    else
    {
      if (pSpy->hmqTarget == NULLHANDLE)       /* SPYIng the SYSTEM queue?    */
      {
                                               /* Yes: for SPY "catcher"?     */
        if (pSpy->hmqSpy == (HMQ) WinQueryWindowULong(pQmsg->hwnd,QWL_HMQ) )
          continue;                            /*      - ignore if so         */
      }
      else                                     /* No:  for specified QUEUE?   */
      {
        if (pSpy->hmqTarget != (HMQ) WinQueryWindowULong(pQmsg->hwnd,QWL_HMQ))
          continue;                            /*      - ignore if not        */
      }
    }

    /*Ŀ*/
    /* Minimum eligibility requirements met...                              */
    /**/
    pSpy->qMsg = *pQmsg;                    /* copy QMSG data */

    /*Ŀ*/
    /* Forward to window that will handle "hooked" item                     */
    /**/
    WinSendMsg(pSpy->hwndSpy,
               pSpy->MsgSpy,
               MPFROMP(&pSpy->qMsg),
               MPFROMSHORT(HK_INPUT) );     /* indicate HK source */
  }

  /*Ŀ*/
  /* Indicate "pass the message to the next hook"                           */
  /**/
  return(BOOL_FALSE);
}

/*Ŀ*/
/* SendMsg Hook procedure                                                   */
/*                                                                          */
/* This procedure gets calls whenever a window procedure is called          */
/* via WinSendMsg                                                           */
/*                                                                          */
/* Notes:                                                                   */
/* ------                                                                   */
/* 1) we are called by PM, not a PMSPY instance.  This implies that we      */
/*    don't know which PMSPY instance, if any, should SPY this message.     */
/*    That's why we have to scan the complete Spys[] table and route it     */
/*    to all interested SPYees...                                           */
/*                                                                          */
/* Reference: PM Programming Reference: Volume Two,                         */
/*            Chapter 10 - "Functions Supplied by Application"              */
/**/

#pragma linkage (SpySendMsgHookProc, system)

VOID SpySendMsgHookProc(HAB        habSpy_NotUsed,
                        PSMHSTRUCT pSmh,
                        BOOL       bTask_NotUsed)
{
  register PdllSPY pSpy;

  /*Ŀ*/
  /* Check each of the 'active' SPYees to see if interested in this message */
  /**/
  for(/* Initialize */ pSpy = &Spys[0];      /* Start @ first Spyee */
      /* While      */ pSpy->fSpyingActive;  /* Not at End-Of-List */
      /* Iterate    */ pSpy++)               /* Move to next Spyee */
  {
    /*Ŀ*/
    /* Is this MSG within the active range?                                 */
    /**/
    if ( (pSmh->msg < pSpy->msgLow) || (pSmh->msg > pSpy->msgHigh) )
      continue;

    /*Ŀ*/
    /* Is this MSG destined for the correct Window?                         */
    /**/
    if ( pSpy->fTargetIsWindow )               /* SPYing a specific window?   */
    {
      if (pSpy->hwndTarget != pSmh->hwnd)      /* ignore if wrong window?     */
        continue;
    }
    /*Ŀ*/
    /* Is this MSG destined for the correct Queue?                          */
    /*                                                                      */
    /* - because we now can "hook" the PM system queue, we must ensure      */
    /*   that this isn't one of our own WinSendMsg() calls that is passing  */
    /*   "hooked" data to the registered SPY "catcher" window.  Failure     */
    /*   to test for this caused PM recursion that eventually overflowed    */
    /*   the stack (trust me, it happened!)                                 */
    /**/
    else
    {
      if (pSpy->hmqTarget == NULLHANDLE)       /* SPYIng the SYSTEM queue?    */
      {
                                               /* Yes: for SPY "catcher"?     */
        if (pSpy->hmqSpy == (HMQ) WinQueryWindowULong(pSmh->hwnd,QWL_HMQ) )
          continue;                            /*      - ignore if so         */
      }
      else                                     /* No:  for specified QUEUE?   */
      {
        if (pSpy->hmqTarget != (HMQ) WinQueryWindowULong(pSmh->hwnd,QWL_HMQ))
          continue;                            /*      - ignore if not        */
      }
    }

    /*Ŀ*/
    /* Minimum eligibility requirements met...                              */
    /*                                                                      */
    /* - standardize to QMSG data                                           */
    /**/
    pSpy->qMsg.hwnd = pSmh->hwnd;
    pSpy->qMsg.msg  = pSmh->msg;
    pSpy->qMsg.mp1  = pSmh->mp1;
    pSpy->qMsg.mp2  = pSmh->mp2;

    /*Ŀ*/
    /* Forward to window that will handle "hooked" item                     */
    /**/
    WinSendMsg(pSpy->hwndSpy,
               pSpy->MsgSpy,
               MPFROMP(&pSpy->qMsg),
               MPFROMSHORT(HK_SENDMSG) );     /* indicate HK source */
  }
}

/*Ŀ*/
/* SpyHooks: set/clear PMPSY instance "hooks"                               */
/* -------------------------------------------------------------            */
/*                                                                          */
/* Parms:   hMQ............message queue to (un)"hook"                      */
/*          hAB............anchor block                                     */
/*          fSet...........True=set "Hooks", False=clear "hooks"            */
/*                                                                          */
/* Returns: TRUE...........successful                                       */
/*          FALSE..........unsuccessful                                     */
/**/
static BOOL SpyHooks(HMQ   hMQ,  /* message queue to "HOOK" */
                     HAB   hAB,  /* required HAB */
                     BOOL  bSet) /* tells whether to set or clear "hooks" */

{
  register BOOL   bResult = BOOL_TRUE; /* start as a OPTIMIST... */
  register int    i;                   /* loop counter */

  static   struct {
                    SHORT  sHookID;
                    PFN    pfnHook;
                  } HookList[] =
                  {
                    {HK_INPUT,   (PFN)SpyInputHookProc   }, /* Input   "hook" */
                    {HK_SENDMSG, (PFN)SpySendMsgHookProc }  /* SendMsg "hook" */
                  };

#define NO_OF_HOOKS ( sizeof(HookList) / sizeof(HookList[0]) )

  /*Ŀ*/
  /* Process each of the "Hook" items                                       */
  /**/
  for(/* Initialize */  i = 0;               /* @ first array element */
      /* While      */  i < NO_OF_HOOKS;     /* while more "hooks" to process */
      /* Iterate    */  i++)
  {
    /*Ŀ*/
    /* Perform Set or Clear operation based on input parameter              */
    /**/
    if ( bSet )
    {
      if ( ! WinSetHook(hAB,
                        hMQ,                      /* add "hook" on QUEUE */
                        HookList[i].sHookID,      /* "hook" type */
                        HookList[i].pfnHook,      /* "hook" procedure */
                        hmodDLL) )                /* "hook" procedure DLL handle */
        bResult = BOOL_FALSE;
    }
    else
    {
      if (! WinReleaseHook(hAB,
                           hMQ,                  /* remove "hook" from QUEUE */
                           HookList[i].sHookID,  /* "hook" type */
                           HookList[i].pfnHook,  /* "hook" procedure */
                           hmodDLL) )            /* "hook" procedure DLL handle */
        bResult = BOOL_FALSE;
    }
  }

  /*Ŀ*/
  /* Return the calculated result                                           */
  /**/
  return( bResult );
}

/*Ŀ*/
/* SpyRegister: register a new PMPSY instance for SPYing                    */
/* -------------------------------------------------------------------------*/
/*                                                                          */
/* Parms:   hwndSpy.........window to be sent MSGs of SPY activity          */
/*           MsgSpy.........PM message to send to SPY window with "hooked"  */
/*                                                                 data     */
/* Returns: 0-n.............registered SPY instance (used on other 'Spy'    */
/*                                                                 calls)   */
/*          SPY_BAD_SPYEE...unable to register new SPY instance (all        */
/*                          instances already in use)                       */
/**/
SPYEE  SpyRegister(HWND hwndSpy,  /* window to handle SPY data */
                  MSG   MsgSpy)   /* PM message for "hooked" data */

{
  register SPYEE idSpy = SPY_BAD_SPYEE;    /* start as a PESSEMIST... */

  char           errbuf[80];

  /*Ŀ*/
  /* Action taken is based on how SPYee are already active                  */
  /**/
  switch ( uSpyInstances )
  {
    /*Ŀ*/
    /* No SPYee is active yet:                                              */
    /*                                                                      */
    /* - load PMSPYHK.DLL for PM-OS/2 usage while "hooking"                 */
    /**/
    case 0:

         if (DosLoadModule(errbuf, sizeof(errbuf), SPY_HOOK_DLL, &hmodDLL) != 0 )
           break;

         /*Ŀ*/
         /* ** fall through to add new instance **                          */
         /**/

    /*Ŀ*/
    /* Some SPYees are already active, just add to table                    */
    /**/
    default:

         idSpy = uSpyInstances;     /* Save this instance # (0-origin) */

         uSpyInstances++;           /* Bump # active SPYees */

         /*Ŀ*/
         /* Add to Instance-to-SPY data translation table                   */
         /**/
         Id2Spys[idSpy] = idSpy;

         /*Ŀ*/
         /* Add to SPY data table                                           */
         /**/
         Spys[idSpy]         = defSPY;     /* default all fields */

         Spys[idSpy].hwndSpy = hwndSpy;    /* who to communicate "hooked" */
         Spys[idSpy].MsgSpy  = MsgSpy;     /* items to                    */

                                           /* agent's HMQ                 */
         Spys[idSpy].hmqSpy  = (HMQ) WinQueryWindowULong(hwndSpy, QWL_HMQ);
    break;

    /*Ŀ*/
    /* All SPYee instances are already active                               */
    /**/
    case MAX_SPYEES:

         idSpy = SPY_BAD_SPYEE;
    break;
  }

  /*Ŀ*/
  /* Return the registration result                                         */
  /**/
  return( idSpy );
}

/*Ŀ*/
/* SpyDeRegister: de-register a existing PMPSY instance from SPYing         */
/* --------------------------------------------------------------------     */
/*                                                                          */
/* Parms:   idSpy...........ID of the PMSPY instance to de-register         */
/*                                                                          */
/* Returns: TRUE............instance successfully de-registered             */
/*          FALSE...........instance NOT          de-registered (bad ID     */
/**/
BOOL SpyDeRegister(SPYEE   idSpy)    /* which SPY instance */

{
  register PdllSPY pSpy;
  register USHORT  i;

  /*Ŀ*/
  /* Validate that the ID is valid - if not immediate return is performed   */
  /*                               - if so, pSpy will locate it's data      */
  /**/
  IF_BAD_ID_RETURN_ELSE_SET(idSpy,BOOL_FALSE,pSpy);

  /*Ŀ*/
  /* First, insure SPYing "hook" is removed...                              */
  /**/
  SpyUnSetTarget(idSpy);

  /*Ŀ*/
  /* Action taken is based on how many SPYee are already active             */
  /**/
  switch ( uSpyInstances )
  {
    /*Ŀ*/
    /* This is the last SPYee....so free "hook" DLL (we loaded on registering)*/
    /**/
    case 1:

         DosFreeModule(hmodDLL);
         hmodDLL = 0;

         /*Ŀ*/
         /* ** fall through to delete instance **                           */
         /**/

    /*Ŀ*/
    /* Some SPYees will remain active, just delete this ID from tables      */
    /**/
    default:

         uSpyInstances--;           /* Decrement # active SPYees (1-origin) */

         /*Ŀ*/
         /* Pack all SPY data table items below this one up one slot        */
         /*                                                                 */
         /* Note: we're intentionally moving the MAX_SPYEES element since   */
         /*       if exists and is always an EOT element                    */
         /**/
         for(/* Initialize */ i = idSpy;       /* Start at this ID */
             /* While      */ i <= MAX_SPYEES; /* While all below not moved */
             /* Iterate    */ i++)             /* keep marching... */
         {
            Spys[i] = Spys[i + 1];
         }

         /*Ŀ*/
         /* Remove from Instance-to-SPY data translation table              */
         /**/
         Id2Spys[idSpy] = ID_NOT_DEFINED;

         /*Ŀ*/
         /* Adjust Id2Spy[] to locate 'Packed' SPY data table               */
         /*                                                                 */
         /* - we only do this if Spys[] data element was below the          */
         /*   SPYee we've just deleted and packed up a slot                 */
         /**/
         for(/* Initialize */ i = 0;           /* Start at first ID */
             /* While      */ i < MAX_SPYEES;  /* While all not processed */
             /* Iterate    */ i++)             /* keep marching... */
         {
            if ( (Id2Spys[i] != ID_NOT_DEFINED)  &&   /* defined? */
                 (Id2Spys[i] >  idSpy) )              /*below deleted ID? */
              Id2Spys[i]--;                           /*yes: adjust 1 slot */
         }

    break;
  }

  /*Ŀ*/
  /* Return that the ID was successfully de-registered                      */
  /**/
  return( BOOL_TRUE );
}

/*Ŀ*/
/* SpySetTarget: set new target window and queue (and begin SPYing)         */
/* ----------------------------------------------------------------         */
/*                                                                          */
/* Parms:   idSpy...........ID of desired PMSPY instance                    */
/*          hwndTarget......target Window to now SPY                        */
/*           hmqTarget......target Queue  to now SPY                        */
/*                                                                          */
/* Returns: TRUE............target successfully set                         */
/*          FALSE...........target NOT          set (bad ID)                */
/**/
BOOL SpySetTarget(register SPYEE idSpy,      /* SPY instance */
                           HWND  hwndTarget, /* Window to SPY */
                           HMQ   hmqTarget)  /* Queue  to SPY */

{
  register PdllSPY pSpy;

  /*Ŀ*/
  /* Validate that the ID is valid - if not immediate return is performed   */
  /*                               - if so, pSpy will locate it's data      */
  /**/
  IF_BAD_ID_RETURN_ELSE_SET(idSpy,BOOL_FALSE,pSpy);

  /*Ŀ*/
  /* Set instance's new Window and Queue                                    */
  /**/
  pSpy->hwndTarget = hwndTarget;
  pSpy-> hmqTarget =  hmqTarget;

  /*Ŀ*/
  /* Set "hook" on this Queue (and return result to caller)                 */
  /**/
  return( SpyHooks(hmqTarget,                          /* msg queue to "HOOK" */
                   WinQueryAnchorBlock(pSpy->hwndSpy), /* little ol' HAB */
                   BOOL_TRUE) );                       /* set them hooks... */
}

/*Ŀ*/
/*SpyUnSetTarget: unset instance's target window and queue (and free "hook")*/
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*Parms:   idSpy...........ID of the PMSPY instance to Un-Set               */
/*                                                                          */
/*Returns: TRUE............target successfully Un-Set                       */
/*         FALSE...........target NOT          Un-Set (bad ID)              */
/**/
BOOL SpyUnSetTarget(register SPYEE idSpy) /* SPY instance */

{
  register BOOL    bResult = BOOL_TRUE;        /* default to OPTIMIST result */
  register PdllSPY pSpy;

  /*Ŀ*/
  /* Validate that the ID is valid - if not immediate return is performed   */
  /*                               - if so, pSpy will locate it's data      */
  /**/
  IF_BAD_ID_RETURN_ELSE_SET(idSpy,BOOL_FALSE,pSpy);

  /*Ŀ*/
  /* Clear "hook" on this instance's QUEUE (if one was actively being SPYed)*/
  /**/
  if ( pSpy->hmqTarget != SPY_BAD_HMQ )
    bResult = SpyHooks(pSpy->hmqTarget,                    /* msg queue to "HOOK" */
                       WinQueryAnchorBlock(pSpy->hwndSpy), /* little ol' HAB */
                       BOOL_FALSE);                        /* clear them hooks... */

  /*Ŀ*/
  /* Set instance's new Window and Queue to "invalid" values                */
  /**/
  pSpy->hwndTarget = SPY_BAD_HWND;
  pSpy->hmqTarget  = SPY_BAD_HMQ;

  return( bResult );
}

/*Ŀ*/
/* SpyQueryTargetWindow: query instance's target window                     */
/* -------------------------------------------------------------            */
/*                                                                          */
/* Parms:   idSpy...........ID of desired PMSPY instance                    */
/*                                                                          */
/* Returns: !SPY_BAD_HWND...window successfully queried                     */
/*          SPY_BAD_HWND....window NOT          queried (bad ID)            */
/**/
HWND SpyQueryTargetWindow(register SPYEE idSpy) /* SPY instance */

{
  register PdllSPY pSpy;

  /*Ŀ*/
  /* Validate that the ID is valid - if not immediate return is performed   */
  /*                               - if so, pSpy will locate it's data      */
  /**/
  IF_BAD_ID_RETURN_ELSE_SET(idSpy,SPY_BAD_HWND,pSpy);

  /*Ŀ*/
  /* Return instance's Window                                               */
  /**/
  return( pSpy->hwndTarget );
}

/*Ŀ*/
/* SpyQueryTargetQueue: query instance's target queue                       */
/* -----------------------------------------------------------              */
/*                                                                          */
/* Parms:   idSpy...........ID of desired PMSPY instance                    */
/*                                                                          */
/* Returns: !SPY_BAD_HMQ....queue successfully queried                      */
/*          SPY_BAD_HMQ.....queue NOT          queried (bad ID)             */
/**/
HMQ SpyQueryTargetQueue(register SPYEE idSpy) /*SPY instance */

{
  register PdllSPY pSpy;

  /*Ŀ*/
  /* Validate that the ID is valid - if not immediate return is performed   */
  /*                               - if so, pSpy will locate it's data      */
  /**/
  IF_BAD_ID_RETURN_ELSE_SET(idSpy,SPY_BAD_HMQ,pSpy);

  /*Ŀ*/
  /* Return instance's Queue                                                */
  /**/
  return( pSpy->hmqTarget );
}

/*Ŀ*/
/* SpySetTargetIsWindow: set that this instance is SPYing the Window/Queue  */
/* -----------------------------------------------------------------------  */
/*                                                                          */
/* Parms:   idSpy...........ID of desired PMSPY instance                    */
/*          fTargetIsWindow.TRUE=SPY on window, FALSE=SPY on queue          */
/*                                                                          */
/* Returns: TRUE............successful                                      */
/*          FALSE...........NOT successful (bad ID)                         */
/**/
BOOL SpySetTargetIsWindow(register SPYEE idSpy,      /* SPY instance */
                                            BOOL  fTargetIsWindow)  /* SPYing on window? */

{
  register PdllSPY pSpy;

  /*Ŀ*/
  /* Validate that the ID is valid - if not immediate return is performed   */
  /*                               - if so, pSpy will locate it's data      */
  /**/
  IF_BAD_ID_RETURN_ELSE_SET(idSpy,BOOL_FALSE,pSpy);

  /*Ŀ*/
  /* Return instance's Queue                                                */
  /**/
  pSpy->fTargetIsWindow = fTargetIsWindow;

  /*Ŀ*/
  /* Return that is worked!                                                 */
  /**/
  return( BOOL_TRUE );
}

/*Ŀ*/
/* SpyQueryTargetIsWindow: query if this instance is SPYing the Window/Queue*/
/* -------------------------------------------------------------------------*/
/*                                                                          */
/* Parms:   idSpy...........ID of desired PMSPY instance                    */
/*                                                                          */
/* Returns: TRUE............SPYing window                                   */
/*          FALSE...........SPYing queue -or- bad ID                        */
/**/
BOOL SpyQueryTargetIsWindow(register SPYEE idSpy) /* SPY instance */

{
  register PdllSPY pSpy;

  /*Ŀ*/
  /* Validate that the ID is valid - if not immediate return is performed   */
  /*                               - if so, pSpy will locate it's data      */
  /**/
  IF_BAD_ID_RETURN_ELSE_SET(idSpy,BOOL_FALSE,pSpy);

  /*Ŀ*/
  /* Return instance's SPY state                                            */
  /**/
  return( pSpy->fTargetIsWindow );
}

/*Ŀ*/
/* SpySetTargetMsgRange: set instance's MSG interest range                  */
/* ------------------------------------------------------------------------ */
/*                                                                          */
/* Parms:   idSpy...........ID of desired PMSPY instance                    */
/*          msgLow..........low  MSG in inclusive range of desired MSGs     */
/*          msgHigh.........high MSG in inclusive range of desired MSGs     */
/*                                                                          */
/* Returns: TRUE............MSG interest range set                          */
/*          FALSE...........MSG interest range NOT set (bad ID or MSG range */
/**/
BOOL SpySetTargetMsgRange(register SPYEE idSpy,      /* SPY instance */
                                   MSG   msgLow,     /* lowest  MSG in range */
                                   MSG   msgHigh)    /* highest MSG in range */

{
  register PdllSPY pSpy;

  /*Ŀ*/
  /* Validate that the ID is valid - if not immediate return is performed   */
  /*                               - if so, pSpy will locate it's data      */
  /**/
  IF_BAD_ID_RETURN_ELSE_SET(idSpy,BOOL_FALSE,pSpy);

  /*Ŀ*/
  /* Validate specified MSG range: low must really be lower or the same!    */
  /**/
  if ( msgLow <= msgHigh )
  {
    pSpy->msgLow  = msgLow;
    pSpy->msgHigh = msgHigh;

    return( BOOL_TRUE );
  }
  else
    return( BOOL_FALSE );
}

/*Ŀ*/
/* SpyDllVersion: Return the Version of the SpyDll Module                   */
/* ------------------------------------------------------                   */
/*                                                                          */
/* Parms:   NONE                                                            */
/*                                                                          */
/* Returns: numeric version ID of this PMSPY support ID                     */
/**/
USHORT SpyDllVersion(VOID)
{
  return( DLLVERSION );
}

/*Ŀ*/
/* SpyQueryDllRegisterMax: Return maximum number of concurrent PMSPY instances*/
/* -------------------------------------------------------------------------*/
/*                                                                          */
/* Parms:   NONE                                                            */
/*                                                                          */
/* Returns: maximum number of PMSPY instances that can be registered        */
/* concurrently                                                             */
/**/
USHORT SpyQueryDllRegisterMax(VOID)
{
  return( MAX_SPYEES );
}
